/* ***************************************************************************************
* FILE:          NodeControlWidget2D.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  NodeControlWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef NodeControl_WIDGET2D_H
#define NodeControl_WIDGET2D_H

#include <bitset>
#include <Widgets/2D/NodeControl/generated/NodeControlWidget2DBase.h>
#include <Widgets/2D/ControlTemplate/ControlTemplateCloneableWidget.h>

/* ***************************************************************************************
* Provides control over a 2D node allowing to change its properties by using data binding sources or to expose them outside composites.
* There is no internal storage in the widget for the property values, they are set/retrieved directly on/from the node.
*************************************************************************************** */
class NodeControlWidget2D : public NodeControlWidget2DBase
   IMPLEMENTS_CLONEABLE_WIDGET
{
      enum PropertiesEnum
      {
         //Transformations
         enPosition = 0,
         enPivotPoint,
         enPivotOffset,
         enBoundingRectangle,
         enRotation,
         enScale,

         //Node
         enScopeMask,
         enAlphaValue,
         enRenderingEnabled,
         enRenderOrderRank,
         enClippingRectangle,

         //Layout
         enVerticalAlignment,
         enHorizontalAlignment,
         enSize,
         enMinimumSize,
         enMaximumSize,
         enMargin,
         enStretchBehavior,
         enLayoutDirection,
         enCollapsible,

         //Grid Layout
         enRow,
         enRowSpan,
         enColumn,
         enColumnSpan,

         enPropertyCount
      };

      typedef std::bitset<static_cast<int>(NodeControlWidget2D::enPropertyCount)> PropertyConfig;

#define DefNodeControlProperty(propType, propName) \
      typedef propType propName##Type; \
      void set##propName(const propName##Type& value); \
      propName##Type get##propName() const;\
      void set##propName##Enabled(bool value) { _config.set(static_cast<size_t>(NodeControlWidget2D::en##propName), value); } \
      bool is##propName##Enabled() const { return _config.test(static_cast<size_t>(NodeControlWidget2D::en##propName)); }

#define CdaNodeControlEnabledProperty(propName, propCategory) \
      CdaProperty(propName##Enabled, bool, is##propName##Enabled, set##propName##Enabled)\
         CdaDescription("Enables configuration of the property " #propName)\
         CdaCategory(propCategory)\
      CdaPropertyEnd()

#define CdaNodeControlProperty(propName, propCategory) \
      CdaBindableProperty(propName, propName##Type, get##propName, set##propName)\
         CdaDescription("See Node's property " #propName)\
         CdaCategory(propCategory)\
         CdaVisibilityTest(is##propName##Enabled)\
      CdaBindablePropertyEnd()

   public:

      CGI_WIDGET_RTTI_DECLARATION(NodeControlWidget2D, NodeControlWidget2DBase);

      NodeControlWidget2D();
      virtual ~NodeControlWidget2D();

      virtual bool CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap);
      virtual void Update() override;

      //Transformations
      DefNodeControlProperty(Candera::Vector2, Position)
      DefNodeControlProperty(Candera::Vector2, PivotPoint)
      DefNodeControlProperty(Candera::Vector2, PivotOffset)
      DefNodeControlProperty(Candera::Rectangle, BoundingRectangle)
      DefNodeControlProperty(Candera::Float, Rotation)
      DefNodeControlProperty(Candera::Vector2, Scale)

      //Node
      DefNodeControlProperty(Candera::UInt32, ScopeMask)
      DefNodeControlProperty(Candera::Float, AlphaValue)
      DefNodeControlProperty(bool, RenderingEnabled)
      DefNodeControlProperty(Candera::Int, RenderOrderRank)
      DefNodeControlProperty(Candera::Rectangle, ClippingRectangle)

      //Layout
      DefNodeControlProperty(Candera::VerticalAlignment, VerticalAlignment)
      DefNodeControlProperty(Candera::HorizontalAlignment, HorizontalAlignment)
      DefNodeControlProperty(Candera::Vector2, Size)
      DefNodeControlProperty(Candera::Vector2, MinimumSize)
      DefNodeControlProperty(Candera::Vector2, MaximumSize)
      DefNodeControlProperty(Candera::Margin, Margin)
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 3)))
      DefNodeControlProperty(LayoutAlignment::StretchBehavior::Enum, StretchBehavior)
      DefNodeControlProperty(LayoutAlignment::LayoutDirection::Enum, LayoutDirection)
#else
      DefNodeControlProperty(Candera::StretchBehavior, StretchBehavior)
      DefNodeControlProperty(Candera::LayoutDirection, LayoutDirection)
#endif
      DefNodeControlProperty(bool, Collapsible)

      //Grid Layout
      DefNodeControlProperty(Candera::UInt8, Row)
      DefNodeControlProperty(Candera::UInt8, RowSpan)
      DefNodeControlProperty(Candera::UInt8, Column)
      DefNodeControlProperty(Candera::UInt8, ColumnSpan)

   private:
      CdaWidget2DDef(NodeControlWidget2D, NodeControlWidget2DBase)
      CdaProperties()

      //Transformations
      CdaNodeControlEnabledProperty(Position, "Transformations")
      CdaNodeControlProperty(Position, "Transformations")
      CdaNodeControlEnabledProperty(PivotPoint, "Transformations")
      CdaNodeControlProperty(PivotPoint, "Transformations")
      CdaNodeControlEnabledProperty(PivotOffset, "Transformations")
      CdaNodeControlProperty(PivotOffset, "Transformations")
      CdaNodeControlEnabledProperty(BoundingRectangle, "Transformations")
      CdaNodeControlProperty(BoundingRectangle, "Transformations")
      CdaNodeControlEnabledProperty(Rotation, "Transformations")
      CdaNodeControlProperty(Rotation, "Transformations")
      CdaNodeControlEnabledProperty(Scale, "Transformations")
      CdaNodeControlProperty(Scale, "Transformations")

      //Node
      CdaNodeControlEnabledProperty(ScopeMask, "Node")
      CdaNodeControlProperty(ScopeMask, "Node")
      CdaNodeControlEnabledProperty(AlphaValue, "Node")
      CdaNodeControlProperty(AlphaValue, "Node")
      CdaNodeControlEnabledProperty(RenderOrderRank, "Node")
      CdaNodeControlProperty(RenderOrderRank, "Node")
      //CdaNodeControlEnabledProperty(ClippingRectangle, "Node")
      //CdaNodeControlProperty(ClippingRectangle, "Node")

      //CdaNodeControlEnabledProperty(RenderingEnabled, "Node")
      //CdaNodeControlProperty(RenderingEnabled, "Node")
      CdaProperty(RenderingEnabledEnabled, bool, isRenderingEnabledEnabled, setRenderingEnabledEnabled)
      CdaDescription("DEPRECATED. Enables the configuration of the node property EnableRendering which is used to determine if the node is rendered or not. A node is effectively rendered if it and all its ancestors have rendering enabled.\n The same node property is configured also by the widget property Visible which is available in all widgets.\n Please notice that if the same property of a node is set from multiple sources then the result is unpredictable.")
      CdaCategory("Node")
      CdaPropertyEnd()
      CdaBindableProperty(RenderingEnabled, RenderingEnabledType, getRenderingEnabled, setRenderingEnabled)
      CdaDescription("DEPRECATED. Configures the node property EnableRendering which is used to determine if the node is rendered or not. A node is effectively rendered if it and all its ancestors have rendering enabled.\n The same node property is configured also by the widget property Visible which is available in all widgets.\n Please notice that if the same property of a node is set from multiple sources then the result is unpredictable.")
      CdaCategory("Node")
      CdaVisibilityTest(isRenderingEnabledEnabled)
      CdaBindablePropertyEnd()

      //Layout
      CdaNodeControlEnabledProperty(VerticalAlignment, "Layout")
      CdaNodeControlProperty(VerticalAlignment, "Layout")
      CdaNodeControlEnabledProperty(HorizontalAlignment, "Layout")
      CdaNodeControlProperty(HorizontalAlignment, "Layout")
      CdaNodeControlEnabledProperty(Size, "Layout")
      CdaNodeControlProperty(Size, "Layout")
      CdaNodeControlEnabledProperty(MinimumSize, "Layout")
      CdaNodeControlProperty(MinimumSize, "Layout")
      CdaNodeControlEnabledProperty(MaximumSize, "Layout")
      CdaNodeControlProperty(MaximumSize, "Layout")
      CdaNodeControlEnabledProperty(Margin, "Layout")
      CdaNodeControlProperty(Margin, "Layout")
      CdaNodeControlEnabledProperty(StretchBehavior, "Layout")
      CdaNodeControlProperty(StretchBehavior, "Layout")
      CdaNodeControlEnabledProperty(LayoutDirection, "Layout")
      CdaNodeControlProperty(LayoutDirection, "Layout")
      CdaNodeControlEnabledProperty(Collapsible, "Layout")
      CdaNodeControlProperty(Collapsible, "Layout")

      //Grid Layout
      CdaNodeControlEnabledProperty(Row, "GridLayout")
      CdaNodeControlProperty(Row, "GridLayout")
      CdaNodeControlEnabledProperty(RowSpan, "GridLayout")
      CdaNodeControlProperty(RowSpan, "GridLayout")
      CdaNodeControlEnabledProperty(Column, "GridLayout")
      CdaNodeControlProperty(Column, "GridLayout")
      CdaNodeControlEnabledProperty(ColumnSpan, "GridLayout")
      CdaNodeControlProperty(ColumnSpan, "GridLayout")

      CdaPropertiesEnd()
      CdaWidgetDefEnd()

      //Returns the node of this widget (if set) or a default static node.
      //The default node is used to get the default value of the properties.
      const Candera::Node2D* getNodeOrDefaultNode() const;

      virtual bool composerPropVisibleFilter_Enabled() const override
      {
         return false;
      }

      PropertyConfig _config;
      bool _invalid;
};


#endif
