/* ***************************************************************************************
* FILE:          MarkerWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  MarkerWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "lint_deactivation.h"
#include "MarkerWidget2D.h"
#include <Widgets/utils/WidgetTraverser.h>


CGI_WIDGET_RTTI_DEFINITION(MarkerWidget2D);


/*****************************************************************************/
MarkerWidget2D::MarkerWidget2D()
{
}


/*****************************************************************************/
MarkerWidget2D::~MarkerWidget2D()
{
}


/*****************************************************************************/
bool MarkerWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const MarkerWidget2D* original = CLONEABLE_WIDGET_CAST<const MarkerWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetId1(original->GetId1());
      SetId2(original->GetId2());
      SetId3(original->GetId3());
      SetId4(original->GetId4());

      cloned = true;
   }
   return cloned;
}


/*****************************************************************************/
MarkerIdType MarkerWidget2D::getMarkerId() const
{
   return MarkerIdType(GetId1(), GetId2(), GetId3(), GetId4());
}


/*****************************************************************************/
class MarkerFinderCallback : public WidgetCheckCallback
{
   public:
      MarkerFinderCallback(const MarkerIdFilter& markerFilter)
         : _markerFilter(markerFilter), _marker(NULL)
      {
      }
      ~MarkerFinderCallback()
      {
         _marker = NULL;
      }

      bool CheckWidget(Candera::Widget2D* widget)
      {
         MarkerWidget2D* marker = Candera::Dynamic_Cast<MarkerWidget2D*>(widget);
         if ((marker != NULL) && _markerFilter(marker->getMarkerId()))
         {
            _marker = marker;
            return true;
         }

         //continue searching
         return false;
      }

      MarkerWidget2D* getMarker() const
      {
         return _marker;
      }

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(MarkerFinderCallback);

      MarkerIdFilter _markerFilter;
      MarkerWidget2D* _marker;
};


/*****************************************************************************/
MarkerWidget2D* MarkerWidget2D::findMarker(Candera::Scene2DContext* sceneContext, const MarkerIdFilter& markerFilter)
{
   if (sceneContext == NULL)
   {
      return NULL;
   }

   MarkerFinderCallback finderCallback(markerFilter);
   hmibase::widget::utils::MessageUtils::distribute(sceneContext, WidgetCheckReqMsg(&finderCallback));
   return finderCallback.getMarker();
}


/*****************************************************************************/
MarkerWidget2D* MarkerWidget2D::findDescendantMarker(Candera::Scene2DContext* sceneContext, Candera::Node2D& startNode, const MarkerIdFilter& markerFilter)
{
   if (sceneContext == NULL)
   {
      return NULL;
   }

   MarkerFinderCallback finderCallback(markerFilter);
   hmibase::widget::utils::MessageUtils::distributeToDescendants(sceneContext, startNode, WidgetCheckReqMsg(&finderCallback));
   return finderCallback.getMarker();
}
