/* ***************************************************************************************
* FILE:          SwipingStrategy.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SwipingStrategy is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "SwipingStrategy.h"


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SwipingStrategy.cpp.trc.h"
#endif
#include "PageLockoutSwipingBehavior.h"
#include "PageSwipingBehavior.h"
#include "ItemSwipingBehavior.h"
#include "ItemLockoutSwipingBehavior.h"
#include <Widgets/2D/List/Content/TimeDispatcher.h>


using namespace FeatStd;
using namespace Candera;


SwipingStrategy::SwipingStrategy(OrientationOperator& orientationOperator, ItemSpeedAnimationManager& itemSpeedAnimationManager, TimeDispatcher& timeDispatcher) :
   _orientationOperator(orientationOperator),
   _pagewiseLockOut(false),
   _fixedPageScrolling(false),
   _fixedPageScrollingOffset(0.0F),
   _listWidget(0),
   _swipingBehavior(0),
   _swiper(),
   _swipeCalculator(itemSpeedAnimationManager),
   _listPageSwipingBehaviorInfo(),
   _listLockoutPageSwipingBehaviorInfo(),
   _coverflow(false),
   _overscrollFunctionRetriever(),
   _overscroller(orientationOperator, timeDispatcher)
{
   _swiper.SetOverscrollMover(&_overscroller);
}


SwipingStrategy::~SwipingStrategy()
{
   if (0 != _swipingBehavior)
   {
      FEATSTD_DELETE(_swipingBehavior);
   }
   _swipingBehavior = 0;
   _listWidget = 0;
}


void SwipingStrategy::Init(ListWidget2D* listWidget, PositioningHandler* positioningHandler, TimeDispatcher* timeDispatcher, SwipingListener* swipingListener)
{
   _listWidget = listWidget;

   _swiper.Init(positioningHandler, listWidget);

   _swipeCalculator.Init(listWidget, timeDispatcher, &_swiper);
   _swipeCalculator.RegisterListener(swipingListener);
}


void SwipingStrategy::SetPagewiseLockOut(bool val)
{
   _pagewiseLockOut = val;
}


void SwipingStrategy::SetFixedPageScrolling(bool val)
{
   _fixedPageScrolling = val;
}


void SwipingStrategy::SetFixedPageScrollingOffset(FeatStd::Float val)
{
   _fixedPageScrollingOffset = val;
}


void SwipingStrategy::SetLockOutFixMovementOffset(FeatStd::Float val)
{
   _listLockoutPageSwipingBehaviorInfo.SetScrollOffsetPercentage(val);
}


void SwipingStrategy::SetSwipingAcceleration(const FeatStd::Float& swipingAcceleration)
{
   _swipeCalculator.SetSwipingAcceleration(swipingAcceleration);
}


void SwipingStrategy::SetMaxSwipingDistance(FeatStd::Float val)
{
   _swipeCalculator.SetMaxSwipingDistance(val);
}


void SwipingStrategy::SetVisibleArea(const Candera::Vector2& val)
{
   _listPageSwipingBehaviorInfo.SetVisibleArea(val);
   _listLockoutPageSwipingBehaviorInfo.SetVisibleArea(val);
}


void SwipingStrategy::SetNormalizer(Normalizer* normalizer)
{
   _swiper.SetNormalizer(normalizer);
}


void SwipingStrategy::SetCoverflow(bool val)
{
   _coverflow = val;
}


void SwipingStrategy::BuildSwipingBehavior()
{
   if (0 != _swipingBehavior)
   {
      FEATSTD_DELETE(_swipingBehavior);
      _swipingBehavior = 0;
   }

   if (_pagewiseLockOut)
   {
      if (0 != _listWidget)
      {
         if (!_coverflow)
         {
            _swipingBehavior = FEATSTD_NEW(PageLockoutSwipingBehavior)(_swiper, *_listWidget, _orientationOperator, _listLockoutPageSwipingBehaviorInfo);
         }
         else if (_listLockoutPageSwipingBehaviorInfo.GetScrollOffsetPercentage() > 0)
         {
            _swipingBehavior = FEATSTD_NEW(ItemLockoutSwipingBehavior)(*_listWidget, _orientationOperator, _listLockoutPageSwipingBehaviorInfo);
         }
      }
   }
   else if (_fixedPageScrolling)
   {
      if (0 != _listWidget)
      {
         _listPageSwipingBehaviorInfo.SetScrollOffsetPercentage(_fixedPageScrollingOffset);
         _swipingBehavior = FEATSTD_NEW(PageSwipingBehavior)(_swiper, *_listWidget, _orientationOperator, _listPageSwipingBehaviorInfo);
      }
   }

   if (0 == _swipingBehavior)
   {
      _swipingBehavior = FEATSTD_NEW(ItemSwipingBehavior)(_swiper);
   }
}


bool SwipingStrategy::Update(Candera::Node2D* searchNode, Candera::Node2D* contentNode, Candera::Vector2 visibleArea)
{
   bool invalidate(UpdateOverscroller(searchNode, contentNode, visibleArea));
   invalidate = _swipeCalculator.Update() || invalidate;
   return invalidate;
}


void SwipingStrategy::Reset()
{
   _swipeCalculator.Reset();
}


bool SwipingStrategy::OnDragGesture(const hmibase::input::gesture::GestureEvent& gestureData)
{
   return _swipeCalculator.OnDragGesture(gestureData);
}


bool SwipingStrategy::OnSwipeGesture(const hmibase::input::gesture::GestureEvent& gestureData)
{
   return _swipeCalculator.OnSwipeGesture(gestureData);
}


SwipingStateChecker* SwipingStrategy::GetSwipingStateChecker()
{
   return &_swipeCalculator;
}


ItemSizesContainer* SwipingStrategy::GetItemSizesContainer()
{
   return &_swiper;
}


void SwipingStrategy::StartSwipe(const Candera::Float estimatedDistance)
{
   if (0 != _swipingBehavior)
   {
      _swipingBehavior->StartSwipe(estimatedDistance);
   }
   _overscroller.StartSwipe(estimatedDistance);
}


bool SwipingStrategy::Swipe(const Candera::Float movementDistance, const bool forward)
{
   bool swiped(false);

   if (0 != _swipingBehavior)
   {
      swiped = _swipingBehavior->Swipe(movementDistance, forward);
   }
   _overscroller.Swipe(movementDistance, forward);

   return swiped;
}


void SwipingStrategy::EndSwipe()
{
   if (0 != _swipingBehavior)
   {
      _swipingBehavior->EndSwipe();
   }
   _overscroller.EndSwipe();
}


bool SwipingStrategy::IsIdleRequested() const
{
   bool idleRequested(false);

   if (0 != _swipingBehavior)
   {
      idleRequested = _swipingBehavior->IsIdleRequested();
   }

   return idleRequested;
}


bool SwipingStrategy::ShouldSnap() const
{
   bool shouldSnap(true);

   if (0 != _swipingBehavior)
   {
      shouldSnap = _swipingBehavior->ShouldSnap();
   }

   return shouldSnap;
}


void SwipingStrategy::SetOverscrollerListener(OverscrollerListener* listener)
{
   _overscroller.SetListener(listener);
}


void SwipingStrategy::BeginGesture(const Candera::Vector2& position)
{
   if (0 != _swipingBehavior)
   {
      _swipingBehavior->BeginGesture(position);
   }
   _overscroller.BeginGesture(position);
}


void SwipingStrategy::StartScroll(const Candera::Vector2& pos)
{
   if (0 != _swipingBehavior)
   {
      _swipingBehavior->StartScroll(pos);
   }
   _overscroller.StartScroll(pos);
}


bool SwipingStrategy::Scroll(const Candera::Float movementDistance, const bool forward)
{
   bool scrolled(false);

   if (0 != _swipingBehavior)
   {
      scrolled = _swipingBehavior->Scroll(movementDistance, forward);
   }
   _overscroller.Scroll(movementDistance, forward);

   return scrolled;
}


void SwipingStrategy::EndScroll(const Candera::Vector2& pos)
{
   if (0 != _swipingBehavior)
   {
      _swipingBehavior->EndScroll(pos);
   }
   _overscroller.EndScroll(pos);
}


bool SwipingStrategy::UpdateOverscroller(Candera::Node2D* searchNode, Candera::Node2D* contentNode, Candera::Vector2 visibleArea)
{
   OverscrollFunction* function(_overscrollFunctionRetriever.Retrieve(searchNode));

   _overscroller.SetOverscrollFunction(function);
   _overscroller.SetContentNode(contentNode);
   _overscroller.SetViewportSize(visibleArea);

   return _overscroller.Update(IsIdleRequested());
}
