/* ***************************************************************************************
* FILE:          Swiper.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Swiper is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Swiper.h"

#include <Candera/System/Mathematics/Math.h>
#include <FeatStd/Platform/PerfCounter.h>
#include <TouchGestures/GestureBasetypes.h>
#include <hmibase/util/Ticker.h>
#include <Widgets/2D/Common/FlexScrollable.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/Swiper.cpp.trc.h"
#endif


using namespace FeatStd;
using namespace Candera;


Swiper::Swiper() :
   _positioningHandler(0),
   _scrollable(0),
   _swipingTarget(0.0F),
   _previousAdjustedPosition(0.0F),
   _normalizer(0),
   _overscrollMover(0),
   _swipeStartedWithOverscroll(false),
   _swipingStartOverscroll(0)
{
}


Swiper::~Swiper()
{
   _positioningHandler = 0;
   _scrollable = 0;
   _normalizer = 0;
   _overscrollMover = 0;
}


void Swiper::Init(PositioningHandler* positioningHandler, FlexScrollable* scrollable)
{
   _positioningHandler = positioningHandler;
   _scrollable = scrollable;
}


bool Swiper::SwipeList(Float movingDistance, bool forward)
{
   return MoveList(movingDistance, forward);
}


bool Swiper::ScrollList(Float movingDistance, bool forward)
{
   return MoveList(movingDistance, forward);
}


void Swiper::SetItemsSizes(const ItemSizesContainer::SizesVector& val)
{
   _itemSizes = val;
}


const ItemSizesContainer::SizesVector& Swiper::GetItemsSizes() const
{
   return _itemSizes;
}


void Swiper::SetNormalizer(Normalizer* normalizer)
{
   _normalizer = normalizer;
}


void Swiper::SetOverscrollMover(OverscrollMover* mover)
{
   _overscrollMover = mover;
}


bool Swiper::MoveList(FeatStd::Float movingDistance, bool forward)
{
   bool swiped = false;
   Float overscrolledDistance(0.0F);

   if (0 != _overscrollMover)
   {
      overscrolledDistance = _overscrollMover->GetMovedDistance();
   }

   if ((0 != _normalizer) && (0 != _positioningHandler) && (0 != movingDistance))
   {
      const Float normalizedDistance = _normalizer->Normalize(movingDistance, _itemSizes);
      const Float currentPosition = _positioningHandler->GetCurrentPosition();
      const Float maxSwipingPosition = _positioningHandler->GetMaximumSwipingPosition();
      const Float maxPosition = _positioningHandler->GetMaximumPosition();
      Float newPosition = currentPosition - normalizedDistance;
      const bool circular = _positioningHandler->IsCircular();

      const bool greyZone = (currentPosition >= maxSwipingPosition) && (currentPosition <= maxPosition);

      const Float positionUpperLimit = greyZone ? maxPosition : maxSwipingPosition;

      bool move((greyZone && (!forward)) || (!greyZone));
      move = move && (overscrolledDistance == 0.0F);
      bool overscroll(!move);

      if (move)
      {
         swiped = true;

         if (!circular)
         {
            overscroll = LimitPosition(positionUpperLimit, newPosition);
         }

         _positioningHandler->SetCurrentPosition(newPosition);
      }

      if (overscroll)
      {
         Overscroll(movingDistance, forward);
      }
   }
   return swiped;
}


void Swiper::Overscroll(FeatStd::Float movingDistance, bool forward)
{
   if (0 != _overscrollMover)
   {
      _overscrollMover->Move(movingDistance, forward);
   }
}


bool Swiper::LimitPosition(const Float upperLimit, Float& correctedPosition)
{
   bool limitted(false);

   if (correctedPosition < 0)
   {
      correctedPosition = 0;
      limitted = true;
   }
   else if (correctedPosition > upperLimit)
   {
      correctedPosition = upperLimit;
      limitted = true;
   }
   else
   {
      //do nothing
   }

   return limitted;
}


FeatStd::Float Swiper::AdjustInitialSwipedDistance(FeatStd::Float calculatedSwipeDistance)
{
   _swipeStartedWithOverscroll = false;
   if ((0 != _positioningHandler) && (0 != _normalizer))
   {
      const Float currentPosition = _positioningHandler->GetCurrentPosition();
      const Float normalizedDistance = _normalizer->Normalize(calculatedSwipeDistance, _itemSizes);
      const Float maxPosition = _positioningHandler->GetMaximumSwipingPosition();
      const bool circular = _positioningHandler->IsCircular();

      _previousAdjustedPosition = currentPosition;
      _swipingTarget = currentPosition - normalizedDistance;

      bool forward(calculatedSwipeDistance < 0);

      if (circular)
      {
         if (forward)
         {
            while (_swipingTarget >= (2 * maxPosition))
            {
               _swipingTarget -= maxPosition;
            }
         }
         else
         {
            while (_swipingTarget < (-maxPosition))
            {
               _swipingTarget += maxPosition;
            }
         }
      }
      else
      {
         if (_swipingTarget < 0)
         {
            _swipingStartOverscroll = _normalizer->Denormalize(currentPosition - _swipingTarget, _itemSizes);
            if (0 != _overscrollMover)
            {
               _overscrollMover->SetInitialSwipeOverscrollDistance(_swipingStartOverscroll);
            }
            _swipeStartedWithOverscroll = true;
            _swipingTarget = 0;
         }
         else if (_swipingTarget > maxPosition)
         {
            _swipingStartOverscroll = _normalizer->Denormalize(currentPosition - _swipingTarget, _itemSizes);
            if (0 != _overscrollMover)
            {
               _overscrollMover->SetInitialSwipeOverscrollDistance(_swipingStartOverscroll);
            }
            _swipeStartedWithOverscroll = true;
            _swipingTarget = maxPosition;
         }
      }
   }

   return AdjustDistance(calculatedSwipeDistance, _swipeStartedWithOverscroll);
}


Float Swiper::AdjustSwipedDistance(Float calculatedSwipeDistance)
{
   return AdjustDistance(calculatedSwipeDistance, _swipeStartedWithOverscroll);
}


FeatStd::Float Swiper::AdjustDistance(FeatStd::Float calculatedSwipeDistance, bool overscroll)
{
   Float adjustedDistance(0.0);

   if ((0 != _positioningHandler) && (0 != _normalizer))
   {
      const Float currentPosition = _positioningHandler->GetCurrentPosition();
      const Float maxPosition = _positioningHandler->GetMaximumSwipingPosition();

      const bool circular = _positioningHandler->IsCircular();
      bool forward(calculatedSwipeDistance < 0);
      if (circular)
      {
         if (forward)
         {
            if ((currentPosition < _previousAdjustedPosition) && (_swipingTarget >= maxPosition))
            {
               _swipingTarget -= maxPosition;
            }
         }
         else
         {
            if ((currentPosition > _previousAdjustedPosition) && (_swipingTarget < 0))
            {
               _swipingTarget += maxPosition;
            }
         }
      }
      else
      {
         if (_swipingTarget < 0)
         {
            _swipingTarget = 0;
            overscroll = true;
         }
         else if (_swipingTarget > maxPosition)
         {
            _swipingTarget = maxPosition;
            overscroll = true;
         }
      }

      _previousAdjustedPosition = currentPosition;

      adjustedDistance = _normalizer->Denormalize(currentPosition - _swipingTarget, _itemSizes);

      if (overscroll && (0 != _overscrollMover))
      {
         adjustedDistance += _overscrollMover->GetExtraDistanceForSwipe();
      }
   }

   return adjustedDistance;
}


Courier::Float Swiper::GetTotalSize() const
{
   Float size(0.0F);
   for (ItemSizesContainer::SizesVector::ConstIterator it(_itemSizes.ConstBegin()); it != _itemSizes.ConstEnd(); ++it)
   {
      size += *it;
   }

   return size;
}
