/* ***************************************************************************************
* FILE:          ItemSnapper.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ItemSnapper is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "ItemSnapper.h"

#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
#include <Candera/EngineBase/Layout/Layouter.h>
#else
#include <Candera/Engine2D/Layout/Layouter.h>
#endif

using namespace Candera;

ItemSnapper::ItemSnapper(TargetPositioner& targetPositioner) :
   _active(false),
   _forward(false),
   _scrolling(true),
   _pagewiseSnap(false),
   _targetPositioner(targetPositioner),
   _threshold(0),
   _snapHoldMaxTime(0),
   _snapHoldTime(0),
   _touchPos(0),
   _normalizer(0)
{
}


ItemSnapper::~ItemSnapper()
{
}


void ItemSnapper::SetActive(bool active)
{
   _active = active;
}


void ItemSnapper::SetPagewiseSnap(bool pagewiseSnap)
{
   _pagewiseSnap = pagewiseSnap;
}


void ItemSnapper::SetForward(bool forward)
{
   _forward = forward;
}


void ItemSnapper::SetScrolling(bool scrolling)
{
   _scrolling = scrolling;
}


void ItemSnapper::SetThreshold(FeatStd::UInt32 threshold)
{
   _threshold = threshold;
}


void ItemSnapper::SetSnapHoldMaxTime(FeatStd::UInt32 holdMaxTime)
{
   _snapHoldMaxTime = holdMaxTime;
}


void ItemSnapper::SetTouchPos(FeatStd::Float touchPoint)
{
   _touchPos = touchPoint;
}


void ItemSnapper::SetNormalizer(Normalizer* normalizer)
{
   _normalizer = normalizer;
}


void ItemSnapper::StartHoldTimer()
{
   if (_snapHoldTimer.running())
   {
      _snapHoldTimer.restart();
   }
   else
   {
      _snapHoldTimer.start();
   }
}


void ItemSnapper::StopHoldTimer()
{
   if (_snapHoldTimer.running())
   {
      _snapHoldTimer.stop();
      _snapHoldTime = _snapHoldTimer.getElapsedTimeSinceStart();
   }
}


void ItemSnapper::Snap(Candera::Node2D& itemsNode, SizeContainer& itemSizeContainer, const Candera::Vector2& viewport, OrientationOperator orientationOperator, FeatStd::Float currentPosition, FeatStd::Float deltaPosition)
{
   if (_active)
   {
      if ((0 == _snapHoldMaxTime) || (_snapHoldTime < _snapHoldMaxTime))
      {
         SnappingInfo snappingInfo(itemsNode, orientationOperator.GetVectorComponent(viewport), orientationOperator, _forward, _scrolling, currentPosition, deltaPosition);
         Candera::Node2D* child(itemsNode.GetFirstChild());
         while (0 != child)
         {
            Candera::Vector2 size = itemSizeContainer.MeasureClientSize(child);
            Candera::Vector2 position(child->GetPosition());

            snappingInfo._nodePositions.Add(orientationOperator.GetVectorComponent(position));
            snappingInfo._nodeSizes.Add(orientationOperator.GetVectorComponent(size));

            child = child->GetNextSibling();
         }

         SnapImpl(snappingInfo);
      }
      else
      {
         // No snapping done if hold time is met
         _targetPositioner.SetTargetPosition(currentPosition, false);
      }
   }
}


TargetPositioner& ItemSnapper::GetTargetPositioner()
{
   return _targetPositioner;
}


ItemSnapper::SnappingInfo::SnappingInfo(Candera::Node2D& itemsNode, FeatStd::Float viewportSize, OrientationOperator orientationOperator, const bool forward, const bool scrolling, FeatStd::Float currentPosition, FeatStd::Float deltaPosition) :
   _itemsNode(itemsNode),
   _viewportSize(viewportSize),
   _orientationOperator(orientationOperator),
   _forward(forward),
   _scrolling(scrolling),
   _currentPosition(currentPosition),
   _deltaPosition(deltaPosition)
{
}
