/* ***************************************************************************************
* FILE:          ListWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ListWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "ListWidget2D.h"
#include "Widgets/2D/ControlTemplate/ControlTemplateCloningStrategy.h"
#include "Courier/Visualization/ViewController.h"
#include "Animations/FlexListLimiterAnimationFactory.h"
#include "FeatStd/Platform/PerfCounter.h"
#include "Candera/System/Mathematics/Rectangle.h"
#include <Candera/System/Rtti/Rtti.h>
#include "Widgets/2D/Common/ClippingDynamicPropertyHost.h"
#include "Widgets/2D/ControlTemplate/ControlTemplate.h"
#include "Widgets/2D/ControlTemplate/ControlTemplateBinding.h"
#include <Widgets/2D/List/Swiping/ItemSwipingBehavior.h>
#include <Widgets/2D/List/Swiping/PageSwipingBehavior.h>
#include <Widgets/2D/List/Swiping/PageLockoutSwipingBehavior.h>
#include <Widgets/2D/ScrollBar/ScrollBarWidget2D.h>
#include <Widgets/2D/ControlTemplate/AnimationModifier.h>
#include <Widgets/2D/Common/ListItemAnimationDynamicPropertyHost.h>
#include <Widgets/2D/ControlTemplate/AnimationCloner.h>
#include "hmibase/util/Ticker.h"
#include <Widgets/2D/List/Snapping/DefaultItemSnapper.h>
#include <Widgets/2D/List/Snapping/CenterItemSnapper.h>
#include <Widgets/2D/List/Snapping/AnimatedCenterItemSnapper.h>
#include <Widgets/2D/List/Snapping/EdgeItemSnapper.h>
#include <Widgets/2D/List/Swiping/DefaultNormalizer.h>
#include <Widgets/2D/List/Swiping/CoverflowNormalizer.h>
#include <Widgets/2D/ControlTemplate/PrepareClonedItemCachingBehavior.h>
#include <Widgets/2D/ControlTemplate/DefaultClonedItemCachingBehavior.h>
#include <Trace/ToString.h>

#if defined SESA_ARABIC_LAYOUT_FIX
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
#include "Candera/EngineBase/Layout/ArabicLayouterPatch.h"
#else
#include "Candera/Engine2D/Layout/ArabicLayouterPatch.h"
#endif
#endif

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ListWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(ListWidget2D);

using namespace Candera;
using namespace Courier;

//#define FLEX_LIST_TOUCH_TEST
#ifdef FLEX_LIST_TOUCH_TEST

#include "Candera/Time.h"

struct FlexListTestTouchMessageData
{
   FeatStd::UInt32              _time;
   Courier::TouchMsgState::Enum _state;
   Courier::XYDIM               _xPos;
   Courier::XYDIM               _yPos;
   Courier::PointerId           _pointerId;
   FeatStd::UInt32              _sourceId;
};


static void FlexListTestTouchMessages(bool startTest, ListWidget2D& widget)
{
   static FeatStd::Int32 s_state = -1;
   static FeatStd::UInt32 s_startTime = 0;
   static FeatStd::UInt32 s_listId = 0;
   static FlexListTestTouchMessageData s_testData[] =
   {
      //         { 1783978, TouchMsgState::Down, 434, 376, 0, 105 },
      //         { 1783985, TouchMsgState::Move, 434, 372, 0, 105 },
      //         { 1783993, TouchMsgState::Move, 434, 366, 0, 105 },
      //         { 1783996, TouchMsgState::Move, 434, 360, 0, 105 },
      //         { 1784003, TouchMsgState::Move, 434, 353, 0, 105 },
      //         { 1784007, TouchMsgState::Move, 434, 345, 0, 105 },
      //         { 1784012, TouchMsgState::Move, 431, 334, 0, 105 },
      //         { 1784021, TouchMsgState::Move, 429, 319, 0, 105 },
      //         { 1784023, TouchMsgState::Move, 426, 301, 0, 105 },
      //         { 1784028, TouchMsgState::Move, 421, 281, 0, 105 },
      //         { 1784035, TouchMsgState::Up  , 417, 261, 0, 105 },
      { 0, TouchMsgState::Down, 434, 376, 0, 105 },
      { 0, TouchMsgState::Move, 421, 375, 0, 105 },
      { 0, TouchMsgState::Move, 421, 262, 0, 105 },
      { 0, TouchMsgState::Up, 417, 261, 0, 105 },
   };
   static FeatStd::Int32 s_numberOfTestMessages = sizeof(s_testData) / sizeof(FlexListTestTouchMessageData);
   FeatStd::UInt32 time = Candera::Time::GetMilliSeconds();

   if (startTest)
   {
      s_state = 0;
      s_listId = widget.GetListId();
      widget.Invalidate();
      s_startTime = time;
   }
   else if (widget.GetListId() == s_listId)
   {
      if ((s_state >= 0) && (s_state < s_numberOfTestMessages))
      {
         FeatStd::UInt32 diffTime = time - s_startTime;
         if ((s_state == 0) || (s_testData[s_state]._time - s_testData[0]._time) <= diffTime)
         {
            Courier::TouchMsg* msg = COURIER_MESSAGE_NEW(Courier::TouchMsg)(s_testData[s_state]._state, s_testData[s_state]._xPos, s_testData[s_state]._yPos, s_testData[s_state]._pointerId, s_testData[s_state]._sourceId);
            if (0 != msg)
            {
               msg->Post();
               ++s_state;
            }
         }
         widget.Invalidate();
      }
   }
}


#endif


class ClippingTraverser : public Candera::TreeTraverser2D
{
      FEATSTD_LINT_CURRENT_SCOPE(1712, "No default constructor needed.")

   public:
      ClippingTraverser(Candera::Rectangle clippingRectangle, Candera::Rectangle touchableRectangle) :
         _clippingRectangle(clippingRectangle),
         _touchableRectangle(touchableRectangle)
      {
      }

      virtual ~ClippingTraverser()
      {
      }

   protected:
      virtual TraverserAction ProcessNode(Node2D& node)
      {
         RenderNode* renderNode = Dynamic_Cast<RenderNode*>(&node);
         Node2D* parentNode = node.GetParent();
         if (0 != parentNode)
         {
            Candera::Rectangle clClippingRect(_clippingRectangle);

            Matrix3x2 worldTransform(parentNode->GetWorldTransform());
            worldTransform.Inverse();

            clClippingRect.Transform(worldTransform);

            ClippingDynamicPropertyHost::SetClippingRectangle(node, clClippingRect);
            ClippingDynamicPropertyHost::SetTouchableRectangle(node, _touchableRectangle);

            if (0 != renderNode)
            {
               if (renderNode->HasClippingRect())
               {
                  const Candera::Rectangle& existingClippingRect(renderNode->GetClippingRect());
                  clClippingRect.Intersect(existingClippingRect);
               }
               renderNode->SetClippingRect(clClippingRect);
            }
         }
         return ProceedTraversing;
      }

   private:
      Candera::Rectangle _clippingRectangle;
      Candera::Rectangle _touchableRectangle;
};


Candera::Vector2 ListWidget2D::FocusedLayouter::OnMeasure(Candera::Node2D& node, const Candera::Vector2& /*clientArea*/)
{
   Candera::Float width = 0.0F;
   Candera::Float height = 0.0F;
   if (0 != _listWidget)
   {
      Node2D* focusedItem = _listWidget->GetFocusedItem();
      if (0 != focusedItem)
      {
         width = GetClientSize(*focusedItem).GetX();
         height = GetClientSize(*focusedItem).GetY();
      }
      Candera::Vector2 childArea(width, height);
      Candera::Node2D* child = node.GetFirstChild();
      while ((0 != child))
      {
         Candera::Layouter* childLayouter = child->GetLayouter();
         if (0 != childLayouter)
         {
            childLayouter->Measure(*child, childArea);
         }
         child = child->GetNextSibling();
      }
   }
   _measuredSize.SetX(width);
   _measuredSize.SetY(height);
   return _measuredSize;
}


#if defined SESA_ARABIC_LAYOUT_FIX
void ListWidget2D::FocusedLayouter::OnArrange(Candera::Node2D& node, const Candera::Rectangle& clientArea)
{
   if (Candera::ArabicLayouterPatch::IsSceneEnabled(node))
   {
      if (0 != _listWidget)
      {
         bool enabled = false;
         Candera::Rectangle childArea(clientArea);
         if (0 != _listWidget->GetFocusedItem())
         {
            if (ControlTemplateCloningStrategy::IsLayoutAreaSet(*_listWidget->GetFocusedItem()))
            {
               childArea = ControlTemplateCloningStrategy::GetLayoutArea(*_listWidget->GetFocusedItem());
            }
            enabled = true;
            Candera::Margin itemsMargin;
            if (0 != _listWidget->GetListItemsNode())
            {
               itemsMargin = Layouter::GetMargin(*_listWidget->GetListItemsNode());
            }
            Candera::Margin margin = GetMargin(node);
            childArea.SetLeft(childArea.GetLeft() - margin.GetLeft() + itemsMargin.GetLeft());
            childArea.SetTop(childArea.GetTop() - margin.GetTop() + itemsMargin.GetTop());
            childArea.SetWidth(childArea.GetWidth() + (margin.GetLeft() + margin.GetRight()));
            childArea.SetHeight(childArea.GetHeight() + (margin.GetTop() + margin.GetBottom()));
            SetMinimumSize(node, childArea.GetSize());
         }
         Candera::Node2D* child = node.GetFirstChild();
         while ((0 != child))
         {
            Candera::Layouter* childLayouter = child->GetLayouter();
            if (0 != childLayouter)
            {
               childLayouter->Arrange(*child, childArea);
            }
            child = child->GetNextSibling();
         }
         node.SetRenderingEnabled(enabled);
      }
   }
   else
   {
      if (0 != _listWidget)
      {
         bool enabled = false;
         Candera::Rectangle childArea(clientArea);
         if (0 != _listWidget->GetFocusedItem())
         {
            if (ControlTemplateCloningStrategy::IsLayoutAreaSet(*_listWidget->GetFocusedItem()))
            {
               childArea = ControlTemplateCloningStrategy::GetLayoutArea(*_listWidget->GetFocusedItem());
            }
            enabled = true;
            Candera::Margin itemsMargin;
            if (0 != _listWidget->GetListItemsNode())
            {
               itemsMargin = Layouter::GetMargin(*_listWidget->GetListItemsNode());
            }
            Candera::Margin margin = GetMargin(node);
            childArea.SetLeft(childArea.GetLeft() - margin.GetLeft() + itemsMargin.GetLeft());
            childArea.SetTop(childArea.GetTop() - margin.GetTop() + itemsMargin.GetTop());
            childArea.SetWidth(childArea.GetWidth() + (margin.GetLeft() + margin.GetRight()));
            childArea.SetHeight(childArea.GetHeight() + (margin.GetTop() + margin.GetBottom()));
            SetMinimumSize(node, childArea.GetSize());
         }
         Candera::Node2D* child = node.GetFirstChild();
         while ((0 != child))
         {
            Candera::Layouter* childLayouter = child->GetLayouter();
            if (0 != childLayouter)
            {
               childLayouter->Arrange(*child, childArea);
            }
            child = child->GetNextSibling();
         }
         ClippingTraverser traverser(_listWidget->_clippingRectangle, _listWidget->_touchableRectangle);
         traverser.Traverse(node);
         node.SetRenderingEnabled(enabled);
      }
      node.SetPosition(clientArea.GetPosition());
   }
}


void ListWidget2D::FocusedLayouter::OnClipping(Candera::Node2D& node, const Candera::Rectangle& /*clientArea*/)
{
   if (Candera::ArabicLayouterPatch::IsSceneEnabled(node))
   {
      if (0 != _listWidget)
      {
         ClippingTraverser traverser(_listWidget->_clippingRectangle, _listWidget->_touchableRectangle);
         traverser.Traverse(node);
      }
   }
}


#else
void ListWidget2D::FocusedLayouter::OnArrange(Candera::Node2D& node, const Candera::Rectangle& clientArea)
{
   if (0 != _listWidget)
   {
      bool enabled = false;
      Candera::Rectangle childArea(clientArea);
      if (0 != _listWidget->GetFocusedItem())
      {
         if (ControlTemplateCloningStrategy::IsLayoutAreaSet(*_listWidget->GetFocusedItem()))
         {
            childArea = ControlTemplateCloningStrategy::GetLayoutArea(*_listWidget->GetFocusedItem());
         }
         enabled = true;
         Candera::Margin itemsMargin;
         if (0 != _listWidget->GetListItemsNode())
         {
            itemsMargin = Layouter::GetMargin(*_listWidget->GetListItemsNode());
         }
         Candera::Margin margin = GetMargin(node);
         childArea.SetLeft(childArea.GetLeft() - margin.GetLeft() + itemsMargin.GetLeft());
         childArea.SetTop(childArea.GetTop() - margin.GetTop() + itemsMargin.GetTop());
         childArea.SetWidth(childArea.GetWidth() + (margin.GetLeft() + margin.GetRight()));
         childArea.SetHeight(childArea.GetHeight() + (margin.GetTop() + margin.GetBottom()));
         SetMinimumSize(node, childArea.GetSize());
      }
      Candera::Node2D* child = node.GetFirstChild();
      while ((0 != child))
      {
         Candera::Layouter* childLayouter = child->GetLayouter();
         if (0 != childLayouter)
         {
            childLayouter->Arrange(*child, childArea);
         }
         child = child->GetNextSibling();
      }
      ClippingTraverser traverser(_listWidget->_clippingRectangle);
      traverser.Traverse(node);
      node.SetRenderingEnabled(enabled);
   }
   node.SetPosition(clientArea.GetPosition());
}


#endif

static bool FlexListWidget2DDefaultFocusStrategy(ListWidget2D& listWidget, const ListChangeMsg& msg, bool /*force*/)
{
   const Int32 numberOfItems(Int32(listWidget.GetEffectiveNumberOfItems()));
   const Int32 msgValue(msg.GetValue());
   const Int32 numberOfPageScrollingItems = msgValue * numberOfItems; // number of items to scroll for (number of pages to scroll * number of items per page)
   const Int32 virtualListSize = Int32(listWidget.GetCompleteSize());
   Int32 focusIndex(listWidget.GetFocusedIndex());
   Int32 startIndex(Int32(listWidget.GetStartIndex()));

   bool handled = false; // Return value of function.
   bool setValues = false; // Commit modified values to listWidget.
   switch (msg.GetListChangeType())
   {
      case ListChangeSet:
         // Scroll to a specific position.
         if (focusIndex < msgValue)
         {
            focusIndex = msgValue;
         }
         else if (focusIndex > (msgValue + numberOfItems - 1))
         {
            focusIndex = msgValue + numberOfItems - 1;
         }
         startIndex = msgValue;

         setValues = true;
         break;

      case ListChangeUp:
         // Shift the focus up one element.
         focusIndex -= msgValue;
         if (focusIndex >= startIndex)
         {
            setValues = true;
         }
         else if (focusIndex >= 0)
         {
            startIndex = focusIndex;
            handled = true;
         }
         else
         {
            startIndex = 0;
            focusIndex = 0;
            handled = true;
         }
         break;

      case ListChangeDown:
         // Shift the focus down one element.
         focusIndex += msgValue;

         if (focusIndex < virtualListSize)
         {
            if (focusIndex >= (startIndex + numberOfItems))
            {
               startIndex = (focusIndex - numberOfItems) + 1;
               handled = true;
            }
            setValues = true;
         }
         else
         {
            focusIndex = virtualListSize - 1;
            handled = true;
         }

         break;

      case ListChangePageUp:
         // Shift the list window up one page.
         if (startIndex > numberOfPageScrollingItems)
         {
            startIndex -= numberOfPageScrollingItems;
            focusIndex = startIndex + numberOfItems - 1;
         }
         else
         {
            startIndex = 0;
            focusIndex = numberOfItems - 1;
         }
         handled = true;
         break;

      case ListChangePageDown:
         // Shift the list window down one page.

         if ((startIndex + numberOfPageScrollingItems + numberOfItems < virtualListSize))
         {
            startIndex += numberOfPageScrollingItems;
            focusIndex = startIndex;
         }
         else if (listWidget.GetScrollingType() == Candera::LastPageNotFilled)
         {
            //At least one element should be visible
            startIndex += numberOfPageScrollingItems;
            if (startIndex >= virtualListSize)
            {
               startIndex = virtualListSize - 1;
            }
            focusIndex = startIndex;
         }
         else
         {
            startIndex = virtualListSize - numberOfItems;
            focusIndex = startIndex;
         }
         handled = true;
         break;

      default:
         //do nothing.
         break;
   }

   if (setValues || handled)
   {
      listWidget.CheckAndSetFocusedIndex(focusIndex);
      listWidget.CheckAndSetStartIndex(UInt32(startIndex));
      listWidget.Invalidate();
   }

   return handled;
}


#ifndef WIN32 // FlexListWidget2DGetFocusStrategy error C2220 because unreferenced: warning treated as error - no 'object' file generated
CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(528, "Unused function.");
static ListWidget2D::FocusStrategy& FlexListWidget2DGetFocusStrategy()
{
   static ListWidget2D::FocusStrategy g_focusStrategy = FlexListWidget2DDefaultFocusStrategy;
   return g_focusStrategy;
}


#endif

void ListWidget2D::SetFocusStrategy(ListWidget2D::FocusStrategy focusStrategy)
{
   FlexListWidget2DGetFocusStrategy() = (0 == focusStrategy) ? FlexListWidget2DDefaultFocusStrategy : focusStrategy;
}


static void FlexListWidget2DDefaultMsgDisposer(const Courier::Message* originalMsg)
{
   Courier::Message* msg = const_cast<Courier::Message*>(originalMsg);

   FEATSTD_LINT_NEXT_EXPRESSION(1788, "The MessageReferrer wrapps the message do destroy it according to Courier rules.") \
   MessageReferrer msgReferrer(msg);
}


static const Courier::Message* FlexListWidget2DDefaultMsgTranslatorStrategy(const Courier::Message& originalMsg, ListWidget2D::MsgDisposer& disposer)
{
   const Courier::Message* translatedMsg = &originalMsg;
   disposer = 0;

   switch (originalMsg.GetId())
   {
#ifdef WIN32
      case HKStatusChangedUpdMsg::ID:
      {
         const HKStatusChangedUpdMsg* hkMsg = Courier::message_cast<const HKStatusChangedUpdMsg*>(&originalMsg);
         if (hkMsg->GetHKCode() == VK_UP && hmibase::HARDKEYSTATE_DOWN == hkMsg->GetHKState())
         {
            translatedMsg = COURIER_MESSAGE_NEW(ListChangeMsg)(0, ListChangeUp, 1, ListChangeMsgSourceHotKey);
            disposer = FlexListWidget2DDefaultMsgDisposer;
         }
         if (hkMsg->GetHKCode() == VK_DOWN && hmibase::HARDKEYSTATE_DOWN == hkMsg->GetHKState())
         {
            translatedMsg = COURIER_MESSAGE_NEW(ListChangeMsg)(0, ListChangeDown, 1, ListChangeMsgSourceHotKey);
            disposer = FlexListWidget2DDefaultMsgDisposer;
         }
      }
      break;
#endif
      case EncoderStatusChangedUpdMsg::ID:
      {
         const EncoderStatusChangedUpdMsg* encoderMsg = Courier::message_cast<const EncoderStatusChangedUpdMsg*>(&originalMsg);
         if (encoderMsg != NULL)
         {
            if (encoderMsg->GetEncSteps() < 0)
            {
               translatedMsg = COURIER_MESSAGE_NEW(ListChangeMsg)(0, ListChangeUp, encoderMsg->GetEncSteps(), ListChangeMsgSourceEncoder);
            }
            else
            {
               translatedMsg = COURIER_MESSAGE_NEW(ListChangeMsg)(0, ListChangeDown, encoderMsg->GetEncSteps(), ListChangeMsgSourceEncoder);
            }
         }
         disposer = FlexListWidget2DDefaultMsgDisposer;
      }
      break;

      default:
         break;
   }
   return translatedMsg;
}


static ListWidget2D::MsgTranslatorStrategy& FlexListWidget2DGetMsgTranslatorStrategy()
{
   static ListWidget2D::MsgTranslatorStrategy g_msgTranslatorStrategy = FlexListWidget2DDefaultMsgTranslatorStrategy;
   return g_msgTranslatorStrategy;
}


void ListWidget2D::SetMsgTranslatorStrategy(ListWidget2D::MsgTranslatorStrategy msgTranslatorStrategy)
{
   FlexListWidget2DGetMsgTranslatorStrategy() = (0 == msgTranslatorStrategy) ? FlexListWidget2DDefaultMsgTranslatorStrategy : msgTranslatorStrategy;
}


ListWidget2D::ListWidget2D() :
   _itemsInvalid(false),
   _delayItemValidation(false),
   _requestPending(false),
   _dataProviderInvalid(true),
   _activ(false),
   _initialized(false),
   _dataReceived(false),
   _expandAnimationRunning(false),
   _customAnimationsRunning(false),
   _itemsInstanceContainer(_ownerInfoRetriever),
   _itemSpeedAnimationManager(),
   _listContentUpdater(_ownerInfoRetriever, _orientationOperator, _orthogonalOrientationOperator, _itemsInstanceContainer, _sizesContainer, _templateRetriever, _contentProvider, _itemSpeedAnimationManager),
   _scrollingSpeed(ListContentUpdater::Slow),
   _frameTime(0),
   _listener(0),
   _limiterAnimationsInvalid(false),
   _currentLimitAnimationPlayerId(LimitAnimationPlayerCount),
   _immediatePositioningRequested(false),
   _startIndexInternallySet(false),
   _orientationOperator(Vertical, true),
   _orthogonalOrientationOperator(Horizontal, true),
   _firstSetAfterRenderingEnabled(true),
   _firstElementIndex(0),
   _movementStatus(ListMovementFinished),
   _oldMovementStatus(ListMovementFinished),
   _forceMovement(false),
   _currentListStatus(ListIdle),
   _currentEffectiveVisible(ListInVisible),
   _expandTriggerManager(_itemsInstanceContainer, &_listData),
   _statusChanged(false),
   _changeType(ListItemChange),
   _currentListChangeMsgSourceType(ListChangeMsgSourceUnknown),
   _animationContainer(_customAnimations),
   _receivedListCustomAnimationReqMsg(false),
   _swipingStrategy(_orientationOperator, _itemSpeedAnimationManager, *this),
   _immediatePositioning(false),
   _listChangeType(ListChangeSet),
   _snapper(0),
   _touchCamera(0),
   _restoreContentUpdaterState(false),
   _normalizer(0),
   _updateCalled(false),
   _expandAdjuster(_swipingStrategy.GetItemSizesContainer(), _listContentUpdater)
{
   SetFocusedIndex(-1);
   SetTouchable(true);
   SetFocusableStatus(true);

   SetDrag(true);
   SetSwipe(true);
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2d %p created\n", this));
}


ListWidget2D::~ListWidget2D()
{
   _listener = 0;

   Animation::AnimationTimeDispatcher::SharedPointer animationTimeDispatcher(GetAnimationTimeDispatcher());

   for (Int i(0); i < LimitAnimationPlayerCount; ++i)
   {
      if (!_currentLimitAnimationPlayer[i].player.PointsToNull())
      {
         _currentLimitAnimationPlayer[i].player->RemoveAnimationPlayerListener(this);
         if (!animationTimeDispatcher.PointsToNull())
         {
            animationTimeDispatcher->RemovePlayer(_currentLimitAnimationPlayer[i].player);
         }
      }
   }

   if (_snapper != 0)
   {
      FEATSTD_DELETE(_snapper);
      _snapper = 0;
   }

   if (0 != _normalizer)
   {
      FEATSTD_DELETE(_normalizer);
      _normalizer = 0;
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2d %p destroyed\n", this));
}


void DisposeAppData(Node2D* invalidItemTemplate)
{
   if (0 != invalidItemTemplate)
   {
      ControlTemplate::ResetControlTemplateData(*invalidItemTemplate, true);
   }
}


void ListWidget2D::Finalize()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D:FInalize instance=%p id=%d", this, GetListId()));

   if (!_cachedCloningBehavior.PointsToNull())
   {
      ControlTemplateCloningStrategy& cloningStrategy(_cachedCloningBehavior->GetCloningStrategy());
      cloningStrategy.Finalize();
      _listContentUpdater.Finalize();
      cloningStrategy.Flush(0, 0);
   }
   if (0 != GetFocusedNode())
   {
      GetFocusedNode()->SetLayouter(DefaultLayouter::GetInstance());
   }

   DisposeAppData(GetTemplateGroup());
   DisposeAppData(GetInvalidItemTemplate());

   //if not invisible by now, enforce notification
   UpdateEffectiveVisibiliy(true);
}


void ListWidget2D::OnChanged(Courier::UInt32 propertyId)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D:OnChanged instance=%p id=%d", this, GetListId()));

   Base::OnChanged(propertyId);

   _itemsInvalid = true;

   switch (propertyId)
   {
      case ListIdPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x", GetListId(), propertyId));
         _dataProviderInvalid = true;
         _ownerInfoRetriever.SetOwnerId(GetListId());
         _itemsInstanceContainer.Init(GetItemsNode());
         _listData.Clear();
         _dataReceived = false;
         _animationContainer.SetOwnerId(GetListId());
         break;

      case InvalidItemTemplatePropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x, value=%p", GetListId(), propertyId, GetInvalidItemTemplate()));
         _templateRetriever.SetInvalidItemTemplate(GetInvalidItemTemplate());
         break;

      case TemplateGroupPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x, value=%p", GetListId(), propertyId, GetTemplateGroup()));
         _templateRetriever.SetTemplateGroup(GetTemplateGroup());
         break;

      case StartIndexPropertyId:
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x, startIndex=%d, _startIndexInternallySet=%d", GetListId(), propertyId, GetStartIndex(), _startIndexInternallySet));

         if (!_startIndexInternallySet)
         {
            SetListMovement(ListMovementUnknown);
            _listContentUpdater.SetTargetStartIndex(GetStartIndex());
         }

         ValidateDataProvider();
         _startIndexInternallySet = false;
         break;
      }

      case PixelWiseAnimationTimePropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x, value=%d", GetListId(), propertyId, GetPixelWiseAnimationTime()));

         _listContentUpdater.SetPixelWiseAnimationTime(GetPixelWiseAnimationTime());
         break;

      case ShortPixelWiseAnimationTimePropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x, value=%d", GetListId(), propertyId, GetShortPixelWiseAnimationTime()));

         _listContentUpdater.SetShortPixelWiseAnimationTime(GetShortPixelWiseAnimationTime());
         break;

      case PixelWiseScrollingEnabledPropertyId:
      case ScrollingOrientationPropertyId:
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x, scrollingOrientation=%d", GetListId(), propertyId, GetScrollingOrientation()));
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x, pixelwiseScrolling=%d", GetListId(), propertyId, GetPixelWiseScrollingEnabled()));

         const ::Candera::ListScrollingOrientation& scrollingOrientation(GetScrollingOrientation());
         const PixelWiseScrollingEnabledType& pixelWiseScrollingEnabled(GetPixelWiseScrollingEnabled());

         _orientationOperator = OrientationOperator(scrollingOrientation, pixelWiseScrollingEnabled);
         _orthogonalOrientationOperator = OrientationOperator((scrollingOrientation == Vertical) ? Horizontal : Vertical, pixelWiseScrollingEnabled);

         _listContentUpdater.SetScrollingOrientation(scrollingOrientation);
         _listContentUpdater.SetPixelwise(pixelWiseScrollingEnabled);
      }
      break;

      case SwipingAccelerationPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x, value=%f", GetListId(), propertyId, GetSwipingAcceleration()));

         _swipingStrategy.SetSwipingAcceleration(GetSwipingAcceleration());
         break;

      case SwipingMaxDistancePropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x, value=%f", GetListId(), propertyId, GetSwipingMaxDistance()));

         _swipingStrategy.SetMaxSwipingDistance(GetSwipingMaxDistance());
         break;

      case SnapPropertyId:
         UpdateSnapper();
         break;

      case SnapHoldTimePropertyId:
         if (_snapper != 0)
         {
            _snapper->SetSnapHoldMaxTime(GetSnapHoldTime());
            break;
         }

      case SnapOffsetThresholdPropertyId:
         if (_snapper != 0)
         {
            _snapper->SetThreshold(GetSnapOffsetThreshold());
            break;
         }

      case LimitExceededUpAnimationTypePropertyId:
      case LimitExceededUpAnimationPropertyId:
      case LimitExceededUpAnimationTimePropertyId:
         InvalidateLimiterAnimation(ExceedUp);
         break;
      case LimitExceededDownAnimationTypePropertyId:
      case LimitExceededDownAnimationPropertyId:
      case LimitExceededDownAnimationTimePropertyId:
         InvalidateLimiterAnimation(ExceedDown);
         break;
      case LimitReachedUpAnimationTypePropertyId:
      case LimitReachedUpAnimationPropertyId:
      case LimitReachedUpAnimationTimePropertyId:
         InvalidateLimiterAnimation(ReachUp);
         break;
      case LimitReachedDownAnimationTypePropertyId:
      case LimitReachedDownAnimationPropertyId:
      case LimitReachedDownAnimationTimePropertyId:
         InvalidateLimiterAnimation(ReachDown);
         break;

      case NumberOfItemsPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x, value=%d", GetListId(), propertyId, GetNumberOfItems()));

         _listContentUpdater.RetainConfiguredNumberOfItems(GetNumberOfItems());
         break;

      case ScrollingTypePropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x, value=%d", GetListId(), propertyId, GetScrollingType()));

         _listContentUpdater.SetCircular(GetScrollingType() == CircularScrolling);
         break;

      case FixedPageScrollingPropertyId:
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x, value=%d", GetListId(), propertyId, GetFixedPageScrolling()));

         const FixedPageScrollingType& fixedPages(GetFixedPageScrolling());
         _listContentUpdater.SetFixedPages(fixedPages);
         _swipingStrategy.SetFixedPageScrolling(fixedPages);
         _swipingStrategy.BuildSwipingBehavior();
         if (_snapper != 0)
         {
            _snapper->SetPagewiseSnap(true);
         }

         break;
      }

      case FixedPageScrollingOffsetPropertyId:
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x, value=%f", GetListId(), propertyId, GetFixedPageScrollingOffset()));

         _swipingStrategy.SetFixedPageScrollingOffset(GetFixedPageScrollingOffset());
         _swipingStrategy.BuildSwipingBehavior();

         break;
      }

      case DynamicGridPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x, value=%d", GetListId(), propertyId, GetDynamicGrid()));

         _listContentUpdater.SetDynamicGrid(GetDynamicGrid());
         SetFixedPageScrolling(true);
         break;

      case CoverflowPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x, value=%d", GetListId(), propertyId, GetCoverflow()));

         _listContentUpdater.SetCoverflow(GetCoverflow());
         _itemsInvalid = true;
         InternalInvalidate();
         UpdateNormalizer();
         _swipingStrategy.SetCoverflow(true);
         break;

      case ItemsNodePropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x, value=%p", GetListId(), propertyId, GetItemsNode()));

         _animationContainer.SetItemsGroup(GetItemsNode());
         ValidateItemsGroup();
         break;

      case CustomAnimationsGroupNodePropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x, value=%p", GetListId(), propertyId, GetCustomAnimationsGroupNode()));

         _animationContainer.SetAnimationsGroup(GetCustomAnimationsGroupNode());
         break;

      case DynamicGridEmptyCellsPolicyPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x, value=%d", GetListId(), propertyId, GetDynamicGridEmptyCellsPolicy()));
         _listContentUpdater.SetDynamicGridEmptyCellsPolicy(GetDynamicGridEmptyCellsPolicy());
         break;
      case PreparedItemsBufferSizePropertyId:
      case PreparedItemsUpdateTriggerOffsetPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x", GetListId(), propertyId));
         UpdateCachedCloningBehavior();
         break;

      case UsableViewportPaddingPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x", GetListId(), propertyId));
         _listContentUpdater.SetPaddingArea(GetUsableViewportPadding());
         break;

      default:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnChanged listId=%d, propertyId=0x%08x", GetListId(), propertyId));
         break;
   }
}


WidgetGestureConfig ListWidget2D::getDefaultGestureConfig() const
{
   return WidgetGestureConfig(WidgetGestureConfig::Drag(true, 0, 0, 5),
                              WidgetGestureConfig::Swipe(true, static_cast<Int32>(GetSwipingVelocityThreshold() * 1000), static_cast<Int32>(GetSwipingDistanceThreshold())));
}


WidgetGestureConfig ListWidget2D::getGestureConfig() const
{
   WidgetGestureConfig gestureConfig(Base::getGestureConfig());

   //drag and swipe make no sense if all items are visible
   if ((GetController() == NULL) && !GetCoverflow() && GetDragAndSwipeOnNeed() && const_cast<ListWidget2D*>(this)->AreAllElementsVisible())
   {
      if (GetDrag() && gestureConfig.getDrag().Enabled)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "getGestureConfig() Disable drag because all items are visible %s", HMIBASE_TO_STRING_VW(this)));
         gestureConfig.set(WidgetGestureConfig::Drag(false));
      }

      if (GetSwipe() && gestureConfig.getSwipe().Enabled)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "getGestureConfig() Disable swipe because all items are visible %s", HMIBASE_TO_STRING_VW(this)));
         gestureConfig.set(WidgetGestureConfig::Swipe(false));
      }
   }

   return gestureConfig;
}


bool ListWidget2D::OnTouch(const Candera::Camera2D& camera, const Candera::Vector2& point)
{
   bool touched = IsEnabled() && (0 != GetNode()) && GetNode()->IsEffectiveRenderingEnabled() && Base::OnTouch(camera, point);
   if (touched)
   {
      _touchCamera = &camera;
   }

   return touched;
}


bool ListWidget2D::OnTouchMessage(const Courier::TouchMsg& /*msg*/)
{
#ifdef FLEX_LIST_TOUCH_TEST
   if (msg.GetXPos() < 20)
   {
      if (msg.GetState() == TouchMsgState::Down)
      {
         FlexListTestTouchMessages(true, *this);
      }
      return true;
   }
#endif
   return false;
}


bool ListWidget2D::IsDynamicGridPropertyVisible() const
{
   return _listContentUpdater.IsGrid();
}


bool ListWidget2D::IsFixedPageScrollingOffsetPropertiesVisible() const
{
   return GetFixedPageScrolling() || GetDynamicGrid();
}


bool ListWidget2D::AreFixedPageScrollingPropertiesVisible() const
{
   return !GetDynamicGrid();
}


void ListWidget2D::OnParentViewLoad(bool activate)
{
   BaseWidget2D::OnParentViewLoad(activate);
   if (activate)
   {
      UpdateInit();

      _requestPending = false;
      _dataProviderInvalid = true;
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: OnParentViewLoad"));

      if ((0 != GetTemplateGroup()) && (0 != GetTemplateGroup()->GetScene()))
      {
         GetTemplateGroup()->GetScene()->ValidateLayout();
      }
   }
}


bool ListWidget2D::OnMessage(const Message& msg)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D:OnMessage instance=%p id=%d, msgPtr=%p, msgId=0x%08X", this, GetListId(), &msg, msg.GetId()));

   bool msgConsumed = false;
   bool shouldForwardMsg = true;

   MsgDisposer disposer;

   const Message* translatedMsg = FlexListWidget2DGetMsgTranslatorStrategy()(msg, disposer);

   if (0 != translatedMsg)
   {
      if (Base::OnMessage(*translatedMsg) == true)
      {
         msgConsumed = true;
         shouldForwardMsg = false;
      }
      else
      {
         switch (translatedMsg->GetId())
         {
            case DragDropStatusUpdMsg::ID:
               ResetStatemachineToIdle();
               break;

            case ParentViewActivateEvent::ID:
            {
               const ParentViewActivateEvent* parentViewActivateEvent = message_cast<const ParentViewActivateEvent*>(translatedMsg);
               OnParentViewActivateEvent(parentViewActivateEvent, msgConsumed, shouldForwardMsg);
            }
            break;

            case ParentViewRenderingEnabledEvent::ID:
            {
               const ParentViewRenderingEnabledEvent* parentViewRenderingEnableEvent = message_cast<const ParentViewRenderingEnabledEvent*>(translatedMsg);
               OnParentViewRenderingEnabledEvent(parentViewRenderingEnableEvent, msgConsumed, shouldForwardMsg);
            }
            break;

            case ListChangeMsg::ID:
            {
               const ListChangeMsg* lstChangeMsg = message_cast<const ListChangeMsg*>(translatedMsg);
               OnListChangeMsg(lstChangeMsg, msgConsumed, shouldForwardMsg);
            }
            break;

            case ListAnimatedChangeReqMsg::ID:
            {
               const ListAnimatedChangeReqMsg* lstAnimatedChangeMsg = message_cast<const ListAnimatedChangeReqMsg*>(translatedMsg);
               OnListAnimatedChangeReqMsg(*lstAnimatedChangeMsg, msgConsumed, shouldForwardMsg);
            }
            break;

            case ListDateProviderResMsg::ID:
            {
               const ListDateProviderResMsg* listDataResultMsg = message_cast<const ListDateProviderResMsg*>(translatedMsg);
               OnListDateProviderResMsg(listDataResultMsg);
            }
            break;

            case ListDataProviderUpdMsg::ID:
            {
               const ListDataProviderUpdMsg* listDataUpdateMsg = message_cast<const ListDataProviderUpdMsg*>(translatedMsg);
               OnListDateProviderUpdMsg(listDataUpdateMsg);
            }
            break;

            case LockOutMsg::ID:
            {
               const LockOutMsg* listDataResultMsg = message_cast<const LockOutMsg*>(translatedMsg);
               OnLockOutMsg(listDataResultMsg);
            }
            break;

            case ListItemExpandReqMsg::ID:
            {
               const ListItemExpandReqMsg* expandReqMsg = message_cast<const ListItemExpandReqMsg*>(translatedMsg);
               OnListItemExpandReqMsg(expandReqMsg);
            }
            break;

            case ListCustomAnimationReqMsg::ID:
               _receivedListCustomAnimationReqMsg = true;
               _itemsInvalid = true;
               _delayItemValidation = false; // necessary to start the custom animation if data is not received yet
               break;

            default:
               break;
         }
         if (shouldForwardMsg && (!msgConsumed) && (translatedMsg->GetId() != Courier::TouchMsg::ID))
         {
            msgConsumed = _itemsInstanceContainer.OnMessage(*translatedMsg);
         }

         bool invalid(false);
         msgConsumed = msgConsumed || _animationContainer.ProcessMessage(*translatedMsg, invalid);

         if (!_cachedCloningBehavior.PointsToNull())
         {
            msgConsumed = msgConsumed || _cachedCloningBehavior->OnMessage(*translatedMsg);
         }

         if (invalid)
         {
            InternalInvalidate();
         }
      }
      if (0 != disposer)
      {
         disposer(translatedMsg);
      }
   }

   if ((!msgConsumed) && (!_currentCloningControlTemplateInstance.PointsToNull()))
   {
      msgConsumed = _currentCloningControlTemplateInstance->OnMessage(msg);
   }

   TryPostListStatusUpdMsg();

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D:OnMessage instance=%p id=%d, msgPtr=%p, msgId=0x%08X finished", this, GetListId(), &msg, msg.GetId()));

   return msgConsumed;
}


tSharedPtrListDataItem ListWidget2D::GetDataItem(Int32 index)
{
   if (_listData.IsDataAvailable())
   {
      if ((index >= Int32(_listData.GetStartIndex())) && (index < Int32(_listData.GetStartIndex() + _listData.GetWindowElementSize())))
      {
         return _listData[UInt32(index - Int32(_listData.GetStartIndex()))];
      }
   }
   else if (!_items.PointsToNull())
   {
      if ((index >= 0) && (index < Int32(_items->size())))
      {
         tSharedPtrIDataItem rawDataItem = (*_items)[UInt32(index)];
         return tSharedPtrListDataItem(Dynamic_Cast<ListDataItem*>(rawDataItem.GetPointerToSharedInstance()));
      }
   }
   return tSharedPtrListDataItem();
}


void ListWidget2D::InitWidget()
{
   Base::InitWidget();

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D:InitialPropertyStatus instance=%p id=%d", this, GetListId()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "  ListWidget2D:InitialPropertyStatusItemsNode= %p", GetItemsNode()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::FocusedNode= %p", GetFocusedNode()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::TemplateGroup= %p", GetTemplateGroup()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::InvalidItemTemplate= %p", GetInvalidItemTemplate()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::BufferSize= %d", GetBufferSize()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::UpdateTriggerOffset= %d", GetUpdateTriggerOffset()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::FocusedIndex= %d", GetFocusedIndex()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::StartIndex= %d", GetStartIndex()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::NumberOfItems= %d", GetNumberOfItems()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::ScrollbarAlwaysVisible= %d", GetScrollbarAlwaysVisible()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::PostListChanged= %d", GetPostListChanged()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::Coverflow= %d", GetCoverflow()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::FixedPageScrolling= %d", GetFixedPageScrolling()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::FixedPageScrollingOffset= %f", GetFixedPageScrollingOffset()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::AcceptImmediatePositioning= %d", GetAcceptImmediatePositioning()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::AcceptImmediatePositioningOnFirstApperance= %d", GetAcceptImmediatePositioningOnFirstApperance()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::AcceptImmediatePositioningOnPositionSet= %d", GetAcceptImmediatePositioningOnPositionSet()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::AcceptImmediatePositioningOnPageScroll= %d", GetAcceptImmediatePositioningOnPageScroll()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::AcceptImmediatePositioningOnItemScroll= %d", GetAcceptImmediatePositioningOnItemScroll()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::PixelWiseScrollingEnabled= %d", GetPixelWiseScrollingEnabled()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::PixelWiseAnimationTime= %d", GetPixelWiseAnimationTime()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::ShortPixelWiseAnimationTime= %d", GetShortPixelWiseAnimationTime()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::ScrollingOrientation= %d", GetScrollingOrientation()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::ScrollingType= %d", GetScrollingType()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::SwipingDistanceThreshold= %d", GetSwipingDistanceThreshold()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::SwipingTimerThreshold= %d", GetSwipingTimerThreshold()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::SwipingVelocityThreshold= %f", GetSwipingVelocityThreshold()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::SwipingAcceleration= %f", GetSwipingAcceleration()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::LimitExceededUpAnimationType= %d", GetLimitExceededUpAnimationType()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::LimitExceededUpAnimationTime= %d", GetLimitExceededUpAnimationTime()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::LimitExceededUpAnimation= %p", GetLimitExceededUpAnimation().GetPointerToSharedInstance()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::LimitExceededUpBounceAmplitude= (%f; %f)", GetLimitExceededUpBounceAmplitude().GetX(), GetLimitExceededUpBounceAmplitude().GetY()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::LimitExceededDownAnimationType= %d", GetLimitExceededDownAnimationType()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::LimitExceededDownAnimationTime= %d", GetLimitExceededDownAnimationTime()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::LimitExceededDownAnimation= %p", GetLimitExceededDownAnimation().GetPointerToSharedInstance()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::LimitExceededDownBounceAmplitude= (%f; %f)", GetLimitExceededDownBounceAmplitude().GetX(), GetLimitExceededDownBounceAmplitude().GetY()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::LimitReachedUpAnimationType= %d", GetLimitReachedUpAnimationType()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::LimitReachedUpAnimationTime= %d", GetLimitReachedUpAnimationTime()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::LimitReachedUpAnimation= %p", GetLimitReachedUpAnimation().GetPointerToSharedInstance()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::LimitReachedUpBounceAmplitude= (%f; %f)", GetLimitReachedUpBounceAmplitude().GetX(), GetLimitReachedUpBounceAmplitude().GetY()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::LimitReachedDownAnimationType= %d", GetLimitReachedDownAnimationType()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::LimitReachedDownAnimationTime= %d", GetLimitReachedDownAnimationTime()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::LimitReachedDownAnimation= %p", GetLimitReachedDownAnimation().GetPointerToSharedInstance()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "   ListWidget2D::Property::LimitReachedDownBounceAmplitude= (%f; %f)", GetLimitReachedDownBounceAmplitude().GetX(), GetLimitReachedDownBounceAmplitude().GetY()));

   _ownerInfoRetriever.SetOwner(this);
   _ownerInfoRetriever.SetOwnerId(GetListId());
   if (0 != GetTemplateGroup())
   {
      ControlTemplate::SetTemplateNode(*GetTemplateGroup(), true);
   }
   if (0 != GetInvalidItemTemplate())
   {
      ControlTemplate::SetTemplateNode(*GetInvalidItemTemplate(), true);
   }
   const ::Candera::ListScrollingOrientation& scrollingOrientation(GetScrollingOrientation());
   const PixelWiseScrollingEnabledType& pixelWiseScrollingEnabled(GetPixelWiseScrollingEnabled());

   _orientationOperator = OrientationOperator(scrollingOrientation, pixelWiseScrollingEnabled);
   _orthogonalOrientationOperator = OrientationOperator((scrollingOrientation == Vertical) ? Horizontal : Vertical, pixelWiseScrollingEnabled);

   _itemsInstanceContainer.Init(GetItemsNode());

   // scrolling animation not initialized yet, same issue as above, will be done as workaround in Update() method
   _templateRetriever.SetInvalidItemTemplate(GetInvalidItemTemplate());
   _templateRetriever.SetTemplateGroup(GetTemplateGroup());
   _templateRetriever.SetTemplateScrollAnimations(_animationArray);

   _contentProvider.SetData(&_listData);

   _listUpdaterListener.SetSizeContainer(&_sizesContainer);

   _listContentUpdater.SetListener(&_listUpdaterListener);
   _listContentUpdater.SetPixelWiseAnimationTime(GetPixelWiseAnimationTime());
   _listContentUpdater.SetShortPixelWiseAnimationTime(GetShortPixelWiseAnimationTime());
   _listContentUpdater.SetAnimationManagerListenr(this);
   _listContentUpdater.SetCircular(GetScrollingType() == CircularScrolling);
   _listContentUpdater.SetFixedPages(GetFixedPageScrolling());
   _listContentUpdater.SetDynamicGrid(GetDynamicGrid());
   _listContentUpdater.SetCoverflow(GetCoverflow());
   _listContentUpdater.SetPixelwise(pixelWiseScrollingEnabled);
   _listContentUpdater.SetScrollingOrientation(scrollingOrientation);
   _listContentUpdater.SetScrollAnimation(GetScrollAnimationInput());
   _listContentUpdater.SetListAlignment(GetListAlignment());
   _listContentUpdater.SetExpandTriggerManager(&_expandTriggerManager);
   _listContentUpdater.SetDynamicGridEmptyCellsPolicy(GetDynamicGridEmptyCellsPolicy());

   _listContentUpdater.Init(_swipingStrategy.GetSwipingStateChecker(),
                            _swipingStrategy.GetItemSizesContainer(),
                            this, GetListItemsNode(), GetNumberOfItems(), this);

   if ((0 != GetFocusedNode()) && (&_focusedLayouter != GetFocusedNode()->GetLayouter()))
   {
      _focusedLayouter.SetListWidget(this);
      GetFocusedNode()->SetLayouter(&_focusedLayouter);
   }

   _swipingStrategy.SetFixedPageScrolling(GetFixedPageScrolling());
   _swipingStrategy.SetFixedPageScrollingOffset(GetFixedPageScrollingOffset());
   _swipingStrategy.SetPagewiseLockOut(_lockoutManager.IsPagewiseLockout());
   _swipingStrategy.SetSwipingAcceleration(GetSwipingAcceleration());
   _swipingStrategy.SetMaxSwipingDistance(GetSwipingMaxDistance());
   _swipingStrategy.Init(this, &_listContentUpdater, this, this);

   _swipingStrategy.BuildSwipingBehavior();

   _swipePositionObserver.Init(&_listContentUpdater);

   Candera::enGestureDirection gestureDir = (Base::GetScrollingOrientation() == Candera::Horizontal) ? Candera::enHorizontal : Candera::enVertical;
   SetDragDirection(gestureDir);
   SetSwipeDirection(gestureDir);

   UpdatePlayer(ExceedUp);
   UpdatePlayer(ExceedDown);
   UpdatePlayer(ReachUp);
   UpdatePlayer(ReachDown);

   UpdateNormalizer();
   UpdateCachedCloningBehavior();
   ValidateItemsGroup();
}


void ListWidget2D::Update()
{
   _sizesContainer.Clear();

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D:Update instance=%p id=%d", this, GetListId()));
   Base::Update();

   UpdateEffectiveVisibiliy();

   UpdateInit();

   if ((0 == GetParentView()) || !_dataReceived)
   {
      ValidateDataProvider();
   }
   else
   {
#ifdef FLEX_LIST_TOUCH_TEST
      FlexListTestTouchMessages(false, *this);
#endif
      ListWidget2DBase::Update();

      UpdateViewportSize();

      _itemsInstanceContainer.ResetClippingRectangle();
      bool invalidate = false;
      bool invalidateLayout = false;

      PerformUpdate(invalidate, invalidateLayout);

      UpdateInvalidations(invalidate, invalidateLayout);
   }

   _itemsInstanceContainer.Update(_customAnimationsRunning);

   if (!_cachedCloningBehavior.PointsToNull())
   {
      _cachedCloningBehavior->Update();
   }

   _frameTime = 0;

   _updateCalled = true;

   _sizesContainer.Clear();
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D:Update instance=%p id=%d finished updating", this, GetListId()));
}


void ListWidget2D::UpdateInit()
{
   // this method is called from update because init is called too early
   // and composite properties are not yet set
   if (!_initialized)
   {
      UpdateSnapper();

      _templateRetriever.SetTemplateScrollAnimations(_animationArray);
      _listContentUpdater.SetScrollAnimation(GetScrollAnimationInput());

      if (GetParentView() != 0)
      {
         HideTemplates();
      }

      _initialized = true;
   }
}


void ListWidget2D::PerformUpdate(bool& invalidate, bool& invalidateLayout)
{
   ExpandAdjuster::AdjustData expandData;
   _expandTriggerManager.UpdateAnimation();
   bool expandAnimationRunning(UpdateExpandAnimations(expandData));

   _itemsInvalid = _itemsInvalid || expandAnimationRunning || _expandAnimationRunning || _customAnimationsRunning;

   _expandAnimationRunning = expandAnimationRunning;

   if (_expandAnimationRunning)
   {
      _listContentUpdater.SetPixelwise(true);
   }
   else
   {
      _listContentUpdater.SetPixelwise(GetPixelWiseScrollingEnabled());
   }

   if (!_delayItemValidation)
   {
      if (_itemsInvalid)
      {
         UpdateContent();
         invalidateLayout = _itemsInvalid;
         invalidate = _itemsInvalid;

         if (!_itemsInvalid)
         {
            SetListMovement(ListMovementFinished);
         }

         TryPostListChangedMsg();
      }

      ValidateDataProvider();
   }
   _delayItemValidation = false;
   _listContentUpdater.UpdateMaxPositions();

   _listContentUpdater.UpdateState();

   if (GetExpandAutoScroll())
   {
      if (_expandAdjuster.Adjust(expandData, _listContentUpdater.GetLastItemMargin()))
      {
         _itemsInvalid = true;
         InternalInvalidate();
      }
   }

   bool listMovementFinished(IsListMovementFinished());

   if (listMovementFinished)
   {
      if (!ImmediatePositioningSet())
      {
         CheckListLimits();
      }

      _oldMovementStatus = _movementStatus;
   }

   invalidate = _swipingStrategy.Update(GetNode(), GetItemsNode(), _clippingRectangle.GetSize()) || invalidate;

   if (AreLimiterAnimationsInvalid())
   {
      ValidateLimiterAnimations();
   }

   // the custom animation running status is taken before updating just to invalidate
   // another frame after they are stopped
   _customAnimationsRunning = _animationContainer.IsRunning();
   UpdateCustomAnimations();
   bool customAnimationsRunning(_animationContainer.IsRunning());

   if ((!customAnimationsRunning) && (_customAnimationsRunning))
   {
      _listContentUpdater.OnCustomAnimationsStopped();
   }

   invalidate = invalidate || IsBouncing();
   invalidate = invalidate || expandAnimationRunning;
   invalidate = invalidate || _customAnimationsRunning;

   invalidateLayout = invalidateLayout || _customAnimationsRunning || IsBouncing();

   TryPostListStatusUpdMsg();

   UpdateToIdle();

   if (_restoreContentUpdaterState)
   {
      _restoreContentUpdaterState = false;

      _listContentUpdater.SetShortPixelWiseAnimationTime(GetShortPixelWiseAnimationTime());
      const ::Candera::ListScrollingOrientation& scrollingOrientation(GetScrollingOrientation());
      const PixelWiseScrollingEnabledType& pixelWiseScrollingEnabled(GetPixelWiseScrollingEnabled());

      _orientationOperator = OrientationOperator(scrollingOrientation, pixelWiseScrollingEnabled);
      _orthogonalOrientationOperator = OrientationOperator((scrollingOrientation == Vertical) ? Horizontal : Vertical, pixelWiseScrollingEnabled);

      _listContentUpdater.SetPixelwise(pixelWiseScrollingEnabled);
      _listContentUpdater.SetMovementDelay(0);
   }

   if (listMovementFinished)
   {
      ListMovementFinishedUpdMsg* msg(COURIER_MESSAGE_NEW(ListMovementFinishedUpdMsg)(GetListId(), _listContentUpdater.GetFirstItemIndex()));
      if (0 != msg)
      {
         msg->Post();
      }
   }
   // To keep the _movementStatus consistent
   if ((!_itemsInvalid) && (_movementStatus != ListMovementFinished))
   {
      SetListMovement(ListMovementFinished);
   }
}


void ListWidget2D::UpdateInvalidations(bool invalidate, bool invalidateLayout)
{
   if (invalidate)
   {
      InternalInvalidate();
   }

   if (invalidateLayout)
   {
      InvalidateLayout();
   }
   else
   {
      //update clipping rectangle if layouting will not be performed
      UpdateClippingRectangle(GetItemsNode());
      UpdateNodeTouchableRectangle();
   }
}


void ListWidget2D::UpdateClippingRectangle(Candera::Node2D* itemsNode)
{
   if (0 != itemsNode)
   {
      const Node2D* parent(itemsNode->GetParent());
      if (0 != parent)
      {
         _clippingRectangle.SetPosition(parent->GetWorldPosition());
         Candera::Margin viewPortPadding(GetUsableViewportPadding());
         _touchableRectangle = Candera::Rectangle(_clippingRectangle.GetLeft() + viewPortPadding.GetLeft(), _clippingRectangle.GetTop() + viewPortPadding.GetTop(),
                               _clippingRectangle.GetWidth() - viewPortPadding.GetLeft() - viewPortPadding.GetRight(),
                               _clippingRectangle.GetHeight() - viewPortPadding.GetTop() - viewPortPadding.GetBottom());
         ClippingTraverser traverser(_clippingRectangle, _touchableRectangle);
         traverser.Traverse(*itemsNode);
         Node2D* node(GetNode());
         if (0 != node)
         {
            ClippingDynamicPropertyHost::SetTouchableRectangle(*node, _touchableRectangle);
            const Node2D* parent(node->GetParent());
            if (0 != parent)
            {
               Matrix3x2 parentTransform(parent->GetWorldTransform());
               if (parentTransform.GetDeterminant() != 0)
               {
                  parentTransform.Inverse();
                  Rectangle boundingRectangle(_clippingRectangle);
                  boundingRectangle.Transform(parentTransform);
                  node->SetBoundingRectangle(boundingRectangle);
               }
            }
         }
      }
   }
}


void ListWidget2D::UpdateNodeTouchableRectangle()
{
   Node2D* node(GetNode());
   if (0 != node)
   {
      ClippingDynamicPropertyHost::SetTouchableRectangle(*node, _touchableRectangle);
   }
}


void ListWidget2D::InternalInvalidate()
{
   FeatStd::Optional<Candera::Rectangle> optRect(_clippingRectangle);
   Invalidate(optRect);
}


void ListWidget2D::UpdateToIdle()
{
   bool resetToIdle = (_currentListStatus == ListScrollBar) || (_currentListStatus == ListEncoder) || (_currentListStatus == ListStatusUnknown);
   resetToIdle = resetToIdle && (!_lockoutManager.IsMoveAllowed(_changeType, _currentListChangeMsgSourceType));

   if (resetToIdle)
   {
      if (_currentListStatus != ListIdle)
      {
         OnListStateChanged(ListIdle);
      }
   }
}


void ListWidget2D::OnPreMeasure(Candera::Node2D& /*node*/, const Candera::Vector2& clientArea)
{
   // Not invalidated when view port changed due to bouncing
   bool invalidate = !IsBouncing() && (_orientationOperator.GetVectorComponent(clientArea) != _orientationOperator.GetVectorComponent(_clippingRectangle.GetSize()));
   if (invalidate)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D:OnPreMeasure Viewport size changed"));
      _itemsInvalid = true;
      InternalInvalidate();
   }

   _swipingStrategy.SetVisibleArea(clientArea);
   _clippingRectangle.SetSize(clientArea);
   _sizesContainer.SetViewPort(clientArea);
}


void ListWidget2D::UpdateContent()
{
   Candera::Node2D* templateGroup = GetTemplateGroup();
   const Vector2& clientArea = _clippingRectangle.GetSize();

   bool updateContent(_itemsInvalid);
   updateContent = updateContent && (templateGroup != GetFocusedNode());
   updateContent = updateContent && (GetShowItemsOnViewActivated() || _receivedListCustomAnimationReqMsg);
   updateContent = updateContent && (!_animationContainer.IsRunning());

   if (updateContent)
   {
      StartIndexType startIndex = GetStartIndex();

      FocusedIndexType focusedIndex = GetFocusedIndex();
      NumberOfItemsType numberOfItems = GetEffectiveNumberOfItems();

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::UpdateContent id=%d, startIndex=%d, viewport=(%f; %f)", GetListId(), startIndex, clientArea.GetX(), clientArea.GetY()));
      _itemsInvalid = _listContentUpdater.Update(startIndex, focusedIndex, numberOfItems, clientArea, _scrollingSpeed);
      _itemsInvalid = _itemsInvalid || (_itemSpeedAnimationManager.RequiresUpdate());
      const Int32 firstItemIndex(_listContentUpdater.GetFirstItemIndex());
      const Int32 lastItemIndex(_listContentUpdater.GetLastItemIndex());

      if (!_cachedCloningBehavior.PointsToNull())
      {
         _cachedCloningBehavior->OnPostItemsUpdated(firstItemIndex, lastItemIndex);
      }

      NotifyOnContentUpdated();

      if (_itemsInvalid)
      {
         CheckAndSetFocusedIndex(focusedIndex);
         SetStartIndexInternally(_listContentUpdater.GetIndexOfFirstCompletelyVisibleElement());

         InternalInvalidate();
      }
      else
      {
         if (!_cachedCloningBehavior.PointsToNull())
         {
            _cachedCloningBehavior->OnPostItemsUpdateFinished(firstItemIndex, lastItemIndex);
         }
      }
   }

   Node2D* itemsNode(GetItemsNode());
   if (0 != itemsNode)
   {
      itemsNode->SetRenderingEnabled(GetShowItemsOnViewActivated());
   }
}


void ListWidget2D::OnPostArrange(Candera::Node2D& node, const Candera::Rectangle& /*clientArea*/)
{
   UpdateClippingRectangle(&node);
   UpdateNodeTouchableRectangle();

   if (0 != _snapper && _snapper->IsActive())
   {
      _snapper->Snap(node, _sizesContainer, _touchableRectangle.GetSize(), _orientationOperator, _listContentUpdater.GetCurrentPosition(), _swipePositionObserver.GetPositionDelta());
      _snapper->SetActive(false);
   }
}


bool ListWidget2D::OnDragGesture(const hmibase::input::gesture::GestureEvent& gestureData)
{
   return _swipingStrategy.OnDragGesture(gestureData);
}


bool ListWidget2D::OnSwipeGesture(const hmibase::input::gesture::GestureEvent& gestureData)
{
   return _swipingStrategy.OnSwipeGesture(gestureData);
}


void ListWidget2D::OnGestureBegin(const Candera::Vector2& position)
{
   _swipingStrategy.BeginGesture(position);
   _swipePositionObserver.OnGestureBegin();
}


bool ListWidget2D::OnScrollStart(const Candera::Vector2& pos)
{
   if (_snapper != 0)
   {
      _snapper->SetActive(false && _swipingStrategy.ShouldSnap());
   }

   _swipingStrategy.StartScroll(pos);

   return !_swipingStrategy.IsIdleRequested();
}


bool ListWidget2D::OnScroll(const Candera::Float movementDistance, const bool forward)
{
   if (_snapper != 0)
   {
      _snapper->SetScrolling(true);
      _snapper->SetForward(forward);
      if (GetSnapHoldTime() > 0)
      {
         _snapper->StartHoldTimer();
      }
   }

   _changeType = ListTouchChange;
   _currentListChangeMsgSourceType = ListChangeMsgSourceInternal;
   _statusChanged = true;

   _lockoutManager.UpdateOperationDirection(forward);
   bool moveAllowed = _lockoutManager.IsMoveAllowed(_changeType, _currentListChangeMsgSourceType);
   if (moveAllowed)
   {
      SetListMovement(ListMovementScrolling);

      RequestImmediatePositioning(false);
      _listChangeType = ListChangeDown;
      UpdateScrollingSpeed();
      UpdateFocus(movementDistance);

      _itemsInvalid = _swipingStrategy.Scroll(movementDistance, forward) || _itemsInvalid;

      if (_itemsInvalid)
      {
         InternalInvalidate();
      }
   }
   else
   {
      _listContentUpdater.UpdateLocked();
   }

   return (!_swipingStrategy.IsIdleRequested() && moveAllowed);
}


bool ListWidget2D::OnScrollEnd(const Candera::Vector2& pos)
{
   if (_lockoutManager.IsMoveAllowed(_changeType, _currentListChangeMsgSourceType))
   {
      _swipingStrategy.EndScroll(pos);
      _swipePositionObserver.OnScrollEnd();
   }

   _listChangeType = ListChangeDown;
   UpdateScrollingSpeed();
   if (_snapper != 0)
   {
      _snapper->SetActive(true && _swipingStrategy.ShouldSnap());
      if (0 != GetItemsNode() && 0 != GetItemsNode()->GetParent())
      {
         if (IsTouchCameraValid())
         {
            const Candera::Vector2 pointInWorldSpace = Candera::Math2D::TransformViewportToScene(*_touchCamera, Candera::Math2D::TransformRenderTargetToViewport(*_touchCamera, pos));

            // transform pointInWorldSpace to parent space of the ItemsNode
            Candera::Matrix3x2 matrix(GetItemsNode()->GetParent()->GetWorldTransform());
            matrix.Inverse();
            const Candera::Vector2 pointInNodeSpace = matrix.Multiply(pointInWorldSpace);
            _snapper->SetTouchPos(_orientationOperator.GetVectorComponent(pointInNodeSpace));
            if (GetSnapHoldTime() > 0)
            {
               _snapper->StopHoldTimer();
            }
         }
         else
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: OnScrollEnd - TouchCamera not valid!"));
         }
      }
   }

   return !_swipingStrategy.IsIdleRequested();
}


bool ListWidget2D::OnSwipeStart(const Candera::Float estimatedDistance)
{
   if (_snapper != 0)
   {
      _snapper->SetActive(false && _swipingStrategy.ShouldSnap());
      if (GetSnapHoldTime() > 0)
      {
         _snapper->StopHoldTimer();
      }
   }
   _swipingStrategy.StartSwipe(estimatedDistance);

   return !_swipingStrategy.IsIdleRequested();
}


bool ListWidget2D::OnSwipe(const Candera::Float movementDistance, const bool forward)
{
   if (_snapper != 0)
   {
      _snapper->SetScrolling(false);
      _snapper->SetForward(forward);
   }
   _changeType = ListTouchChange;
   _currentListChangeMsgSourceType = ListChangeMsgSourceInternal;
   _statusChanged = true;

   _lockoutManager.UpdateOperationDirection(forward);

   bool moveAllowed = _lockoutManager.IsMoveAllowed(_changeType, _currentListChangeMsgSourceType);
   if (moveAllowed)
   {
      SetListMovement(ListMovementSwiping);

      RequestImmediatePositioning(false);
      _listChangeType = ListChangeDown;
      UpdateScrollingSpeed();
      UpdateFocus(movementDistance);

      _itemsInvalid = _swipingStrategy.Swipe(movementDistance, forward) || _itemsInvalid;

      if (_itemsInvalid)
      {
         InternalInvalidate();
         InvalidateLayout();
      }
   }
   else
   {
      _listContentUpdater.UpdateLocked();
   }

   return (!_swipingStrategy.IsIdleRequested() && moveAllowed);
}


bool ListWidget2D::OnSwipeEnd()
{
   if (_lockoutManager.IsMoveAllowed(_changeType, _currentListChangeMsgSourceType))
   {
      _swipingStrategy.EndSwipe();
      _swipePositionObserver.OnSwipeEnd();
   }
   if (_snapper != 0)
   {
      _snapper->SetActive(true && _swipingStrategy.ShouldSnap());
   }

   return !_swipingStrategy.IsIdleRequested();
}


UInt32 ListWidget2D::GetEffectiveTouchPriority() const
{
   return GetTouchPriority();
}


void ListWidget2D::ValidateDataProvider()
{
   if ((!_requestPending) && (0 != GetListId()))
   {
      Int32 visualStart(_listContentUpdater.GetIndexOfFirstCompletelyVisibleElement());

      const bool isCircularScrolling = IsCircular();
      bool updateDataProvider = false;
      _delayItemValidation = false;

      if (_dataProviderInvalid || !_listData.IsDataAvailable())
      {
         // no request is pending and we do not yet have a data provider or an invalid one
         // a delay of one frame gives the data provider the chance to send an update before the next frame is rendered
         updateDataProvider = true;
         _delayItemValidation = true;
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: ValidateDataProvider ListDateProvider request condition 1; List %d", GetListId()));
      }
      else if ((0 == _listData.GetStartIndex()) && (_listData.GetVirtualListSize() == _listData.GetWindowElementSize()))
      {
         // nothing should be done. All the data is already available
      }
      // If data provider's startIndex == 0 and the window size == list size we do not need to check for validity.
      else if (isCircularScrolling || (0 != _listData.GetStartIndex()) || (_listData.GetWindowElementSize() != _listData.GetVirtualListSize()))
      {
         // Certain values need to be signed, to keep code readable they are stored here.
         const Int32 updateTriggerOffset(GetUpdateTriggerOffset());
         const Int32 visibleItems(GetEffectiveNumberOfItems());
         Int32 visualEnd(visualStart + visibleItems);

         const Int32 dataVirtualSize(_listData.GetVirtualListSize());
         const Int32 dataWindowSize(_listData.GetWindowElementSize());
         const Int32 dataStart(_listData.GetStartIndex());
         const Int32 dataEnd(dataStart + dataWindowSize);

         if (isCircularScrolling)
         {
            const Int32 bufferSize(GetBufferSize());

            const Int32 visualStartAdjusted(visualStart - bufferSize);
            const Int32 visualEndAdjusted(visualStart + bufferSize);

            const bool dataContainsStart((dataStart <= 0) && (0 <= dataEnd));
            const bool dataContainsEnd((dataStart <= dataVirtualSize) && (dataVirtualSize <= dataEnd));

            const bool visualContainsStart((visualStartAdjusted <= 0) && (0 <= visualEndAdjusted));
            const bool visualContainsEnd((visualStartAdjusted <= dataVirtualSize) && (dataVirtualSize <= visualEndAdjusted));

            if (dataContainsStart && visualContainsEnd)
            {
               visualStart -= dataVirtualSize;
               visualEnd -= dataVirtualSize;
            }
            else if (dataContainsEnd && visualContainsStart)
            {
               visualStart += dataVirtualSize;
               visualEnd += dataVirtualSize;
            }
         }

         if (visualStart < dataStart)
         {
            // the top item window trigger area of the list has been reached and
            // the item window is not yet at the top of the virtual list
            // a delay of one frame gives the data provider the chance to send an update before the next frame is rendered
            updateDataProvider = true;
            _delayItemValidation = true;
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: ValidateDataProvider ListDateProvider request condition 2; List %d", GetListId()));
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: ValidateDataProvider ListDateProvider request list %d, startIndex = %d, provider start index = %d, provider window size = %d, list size = %d", GetListId(), visualStart, dataStart, dataWindowSize, dataVirtualSize));
         }
         else if (((dataStart > 0) || isCircularScrolling) && (dataStart + updateTriggerOffset > visualStart))
         {
            // the top item window trigger area of the list has been reached and
            // the item window is not yet at the top of the virtual list.

            updateDataProvider = true;
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: ValidateDataProvider ListDateProvider request condition 5; List %d", GetListId()));
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: ValidateDataProvider ListDateProvider request list %d, startIndex = %d, provider start index = %d, provider window size = %d, list size = %d", GetListId(), visualStart, dataStart, dataWindowSize, dataVirtualSize));
         }
         else if ((visualEnd > dataEnd) && ((dataEnd < dataVirtualSize) || isCircularScrolling))
         {
            // the bottom of the list has been reached and we have no more items to show
            // a delay of one frame gives the data provider the chance to send an update before the next frame is rendered
            _delayItemValidation = true;
            updateDataProvider = true;
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: ValidateDataProvider ListDateProvider request condition 3; List %d", GetListId()));
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: ValidateDataProvider ListDateProvider request list %d, startIndex = %d, provider start index = %d, provider window size = %d, list size = %d", GetListId(), visualStart, dataStart, dataWindowSize, dataVirtualSize));
         }
         else if ((visualEnd + updateTriggerOffset > dataEnd) && ((dataEnd < dataVirtualSize) || isCircularScrolling))
         {
            // the bottom item window trigger area of the list has been reached and
            // the item window is not yet at the bottom of the virtual list
            updateDataProvider = true;
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: ValidateDataProvider ListDateProvider request condition 4; List %d", GetListId()));
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: ValidateDataProvider ListDateProvider request list %d, startIndex = %d, provider start index = %d, provider window size = %d, list size = %d", GetListId(), visualStart, dataStart, dataWindowSize, dataVirtualSize));
         }
         else
         {
            // nothing required
         }
      }

      if (updateDataProvider)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: ->> PostListDataRequestMsg"));
         PostListDataRequestMsg(visualStart);
      }
   }
}


void ListWidget2D::PostListDataRequestMsg(UInt32 startIndex)
{
   if ((0 != GetParentView()) && (GetParentView()->IsRenderingEnabled() || _activ == false))
   {
      if (_requestPending)
      {
         //do not send a new request.
         return;
      }

      UInt32 bufferSize = GetBufferSize();
      Int32 tmpStartIndex = (startIndex <= bufferSize) ? (0) : (Int32(startIndex) - Int32(bufferSize));
      UInt32 windowSize = GetEffectiveNumberOfItems() + bufferSize + bufferSize;
      UInt32 virtualListSize = (_listData.IsDataAvailable()) ? (_listData.GetVirtualListSize()) : ((!_items.PointsToNull()) ? _items->size() : 0);
      bool fetchingPastEnd = (IsCircular()) && ((bufferSize >= startIndex + GetEffectiveNumberOfItems()) || (startIndex >= (virtualListSize - bufferSize)));
      if (_dataProviderInvalid || !_listData.IsDataAvailable() || (tmpStartIndex != _listData.GetStartIndex()) || (windowSize != _listData.GetWindowElementSize()) || fetchingPastEnd)
      {
         bool cachedData = (GetCachedDataMessage() != 0);
         bool dataNeedsUpdate = _listData.PostListDataRequestMsg(GetListId(), startIndex, bufferSize, GetEffectiveNumberOfItems(), GetScrollingType(), cachedData, _requestPending);
         _dataProviderInvalid = dataNeedsUpdate;
         _activ = true;

         if (_requestPending)
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: ->> Message posted"));
         }
      }
   }
   else
   {
      if (_dataProviderInvalid && !_requestPending)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: ->> No parent view or rendering of view disabled. _dataProviderInvalid reset to false"));
      }
      _dataProviderInvalid = !_requestPending;
   }
}


bool ListWidget2D::IsScrollBarVisible()
{
   return GetScrollbarAlwaysVisible() || (GetFirstFullyVisiblePosition() > 0) || (!AreAllElementsVisible());
}


Candera::UInt32 ListWidget2D::GetPosition()
{
   return _listContentUpdater.GetFirstItemIndex();
}


Candera::UInt32 ListWidget2D::GetFirstFullyVisiblePosition()
{
   return _listContentUpdater.GetIndexOfFirstCompletelyVisibleElement();
}


Candera::UInt32 ListWidget2D::GetVisibleSize()
{
   return GetEffectiveNumberOfItems();
}


Candera::UInt32 ListWidget2D::GetCompleteSize()
{
   if (_listData.IsDataAvailable())
   {
      return _listData.GetVirtualListSize();
   }
   else if (!_items.PointsToNull())
   {
      return _items->size();
   }
   else
   {
      return 0;
   }
}


Candera::Widget2D* ListWidget2D::GetWidget()
{
   return this;
}


void ListWidget2D::OnStateChanged(SwipingState::Enum state)
{
   OnListStateChanged(ListStatusType(state));
}


void ListWidget2D::OnListStateChanged(ListStatusType state)
{
   if (_currentListStatus != state)
   {
      if (state == ListIdle)
      {
         _lockoutManager.UpdateOperationDirection(ListStatusDirectionUnavailable);
      }

      _currentListStatus = state;
      _statusChanged = true;

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnListStateChanged, listStatus=%u", _currentListStatus));

      InternalInvalidate();
   }
}


const ::Candera::ListScrollingOrientation& ListWidget2D::GetScrollingOrientation() const
{
   return ListWidget2DBase::GetScrollingOrientation();
}


bool ListWidget2D::IsNumberOfItemsVisible() const
{
   return !GetPixelWiseScrollingEnabled();
}


bool ListWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const ListWidget2D* original = CLONEABLE_WIDGET_CAST<const ListWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetItemsNode(controlTemplateMap.ResolveNodeClone(original->GetItemsNode()));
      SetFocusedNode(controlTemplateMap.ResolveNodeClone(original->GetFocusedNode()));
      SetTemplateGroup(controlTemplateMap.ResolveNodeClone(original->GetTemplateGroup()));
      SetInvalidItemTemplate(controlTemplateMap.ResolveNodeClone(original->GetInvalidItemTemplate()));
      SetListId(original->GetListId());
      SetBufferSize(original->GetBufferSize());
      SetUpdateTriggerOffset(original->GetUpdateTriggerOffset());
      SetFocusedIndex(original->GetFocusedIndex());
      SetNumberOfItems(original->GetNumberOfItems());
      SetScrollbarAlwaysVisible(original->GetScrollbarAlwaysVisible());
      SetDragAndSwipeOnNeed(original->GetDragAndSwipeOnNeed());

      SetPixelWiseScrollingEnabled(original->GetPixelWiseScrollingEnabled());
      SetScrollingOrientation(original->GetScrollingOrientation());
      SetSnap(original->GetSnap());

      _TODO("Review all list widget properties and add here those which make sense for clones")

      if (ControlTemplateBinding::IsItemsBindable(*this))
      {
         FeatStd::Int bindingIndex = ControlTemplateBinding::GetItemsBindingIndex(*this);
         if (bindingIndex > 0)
         {
            tSharedPtrIDataItem rawDataItem = (*ControlTemplateBinding::GetDataContext(*this))[bindingIndex];
            tSharedPtrListDataItem dataItem(Candera::Dynamic_Cast<ListDataItem*>(rawDataItem.GetPointerToSharedInstance()));

            if (!dataItem.PointsToNull())
            {
               if (dataItem->type() == ListDataItem::IntegerValue)
               {
                  SetListId(static_cast<Candera::UInt32>(*dataItem));
               }
               else if (dataItem->type() == ListDataItem::VectorValue)
               {
                  _items = dataItem;
                  _contentProvider.SetItems(dataItem);
               }
            }
         }
      }

      if (ControlTemplateBinding::IsSelectedBindable(*this))
      {
         Candera::Int32 value = ControlTemplateBinding::GetSelectedValue(*this);
         SetStartIndex(static_cast<StartIndexType>(value));
      }
      else
      {
         SetStartIndex(original->GetStartIndex());
      }

      cloned = true;
   }
   return cloned;
}


void ListWidget2D::OnLimitExceededUp()
{
   PlayAnimation(ExceedUp, _currentLimitAnimationPlayer[ExceedUp].player);
}


void ListWidget2D::OnLimitExceededDown()
{
   PlayAnimation(ExceedDown, _currentLimitAnimationPlayer[ExceedDown].player);
}


void ListWidget2D::OnLimitReachedUp()
{
   PlayAnimation(ReachUp, _currentLimitAnimationPlayer[ReachUp].player);
}


void ListWidget2D::OnLimitReachedDown()
{
   PlayAnimation(ReachDown, _currentLimitAnimationPlayer[ReachDown].player);
}


void ListWidget2D::PlayAnimation(LimitAnimationPlayerId id, Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayer> player)
{
   if (!player.PointsToNull())
   {
      if (!player->IsEnabled())
      {
         _currentLimitAnimationPlayerId = id;

         //Note: Checks if animation player was already added is done inside AddPlayer.
         static_cast<void>(GetAnimationTimeDispatcher()->AddPlayer(player));

         static_cast<void>(player->AddAnimationPlayerListener(this));
         static_cast<void>(player->Start());

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D:PlayAnimation instance=%p listId=%d, animationId=%d", this, GetListId(), id));

         if (id < LimitAnimationPlayerCount)
         {
            UInt32 animationTime = GetAnimationTime(id);
            if (animationTime != 0)
            {
               Float speedfactor = static_cast<Float>(player->GetSequenceDurationMs()) / static_cast<Float>(animationTime);
               player->SetSpeedFactor(speedfactor);
            }
         }
      }
   }
}


void ListWidget2D::OnPastEnd(Candera::Animation::AnimationPlayerBase* animationPlayer, Candera::Int32 /*completedIterationsCount*/)
{
   if (0 != animationPlayer)
   {
      animationPlayer->RemoveAnimationPlayerListener(this);
   }

   if (GetAnimationTimeDispatcher() != 0)
   {
      static_cast<void>(GetAnimationTimeDispatcher()->RemovePlayer(Candera::MemoryManagement::SharedPointer<Animation::AnimationPlayerBase>(animationPlayer)));
   }

   _currentLimitAnimationPlayerId = LimitAnimationPlayerCount;
}


void ListWidget2D::OnStopAnimation(Animation::AnimationPlayerBase* animationPlayer)
{
   if (GetAnimationTimeDispatcher() != 0)
   {
      static_cast<void>(GetAnimationTimeDispatcher()->RemovePlayer(Candera::MemoryManagement::SharedPointer<Animation::AnimationPlayerBase>(animationPlayer)));
   }

   _currentLimitAnimationPlayerId = LimitAnimationPlayerCount;
}


bool ListWidget2D::IsAnimationRunning(Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayer> player) const
{
   bool result = false;
   if (!player.PointsToNull())
   {
      result = player->IsEnabled();
   }

   return result;
}


void ListWidget2D::UpdatePlayer(LimitAnimationPlayerId id)
{
   LimitAnimationType type = NoAnimation;
   switch (id)
   {
      case ExceedUp:
         type = GetLimitExceededUpAnimationType();
         break;
      case ExceedDown:
         type = GetLimitExceededDownAnimationType();
         break;
      case ReachUp:
         type = GetLimitReachedUpAnimationType();
         break;
      case ReachDown:
         type = GetLimitReachedDownAnimationType();
         break;
      default:
         break;
   }

   if (id < ListWidget2D::LimitAnimationPlayerCount)
   {
      if (_currentLimitAnimationPlayer[id].type != type)
      {
         //Stop animation if not done already.
         if (!_currentLimitAnimationPlayer[id].player.PointsToNull())
         {
            if (_currentLimitAnimationPlayer[id].player->IsEnabled())
            {
               static_cast<void>(_currentLimitAnimationPlayer[id].player->Stop());
            }
         }
      }

      _currentLimitAnimationPlayer[id].type = type;
      bool created = false;
      switch (_currentLimitAnimationPlayer[id].type)
      {
         case MarginBounceAnimation:
            if (GetNode() != 0)
            {
               if (GetNode()->GetParent() != 0)
               {
                  switch (id)
                  {
                     case ExceedUp:
                        _currentLimitAnimationPlayer[id].player =
                           FlexListMarginBounceAnimationFactory::Create(GetNode()->GetParent(), GetAnimationTime(id), GetLimitExceededUpBounceAmplitude());
                        created = true;
                        break;
                     case ReachUp:
                        _currentLimitAnimationPlayer[id].player =
                           FlexListMarginBounceAnimationFactory::Create(GetNode()->GetParent(), GetAnimationTime(id), GetLimitReachedUpBounceAmplitude());
                        created = true;
                        break;
                     case ExceedDown:
                        _currentLimitAnimationPlayer[id].player =
                           FlexListMarginBounceAnimationFactory::Create(GetNode()->GetParent(), GetAnimationTime(id), GetLimitExceededDownBounceAmplitude());
                        created = true;
                        break;
                     case ReachDown:
                        _currentLimitAnimationPlayer[id].player =
                           FlexListMarginBounceAnimationFactory::Create(GetNode()->GetParent(), GetAnimationTime(id), GetLimitReachedDownBounceAmplitude());
                        created = true;
                        break;
                     default:
                        break;
                  }
               }
            }
            if (!created)
            {
               _currentLimitAnimationPlayer[id].player = SharedPointer<Animation::AnimationPlayer>(0);
            }
            break;
         case CustomAnimation:
            _currentLimitAnimationPlayer[id].player = GetCustomAnimationPlayer(id);
            break;
         case NoAnimation:
         default:
            _currentLimitAnimationPlayer[id].player = SharedPointer<Animation::AnimationPlayer>(0);
            break;
      }

      _currentLimitAnimationPlayer[id].invalid = false;
   }
}


Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayer> ListWidget2D::GetCustomAnimationPlayer(LimitAnimationPlayerId id)
{
   Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayer> result(0);
   switch (id)
   {
      case ExceedUp:
         result = GetLimitExceededUpAnimation();
         break;
      case ExceedDown:
         result = GetLimitExceededDownAnimation();
         break;
      case ReachUp:
         result = GetLimitReachedUpAnimation();
         break;
      case ReachDown:
         result = GetLimitReachedDownAnimation();
         break;
      default:
         break;
   }

   return result;
}


UInt32 ListWidget2D::GetAnimationTime(LimitAnimationPlayerId id)
{
   UInt32 result(0);
   switch (id)
   {
      case ExceedUp:
         result = GetLimitExceededUpAnimationTime();
         break;
      case ExceedDown:
         result = GetLimitExceededDownAnimationTime();
         break;
      case ReachUp:
         result = GetLimitReachedUpAnimationTime();
         break;
      case ReachDown:
         result = GetLimitReachedDownAnimationTime();
         break;
      default:
         break;
   }

   return result;
}


void ListWidget2D::ValidateLimiterAnimations()
{
   for (UInt32 i = 0; i < UInt32(LimitAnimationPlayerCount); i++)
   {
      if (IsLimiterAnimationInvalid(static_cast<LimitAnimationPlayerId>(i)))
      {
         UpdatePlayer(static_cast<LimitAnimationPlayerId>(i));
      }
   }
   _limiterAnimationsInvalid = false;
}


void ListWidget2D::UpdateFocus(Candera::Float movementDistance)
{
   const Int32 focusedIndex(GetFocusedIndex());
   const Int32 startIndex(static_cast<Int32>(_listContentUpdater.GetIndexOfFirstCompletelyVisibleElement()));
   const Int32 numberOfItems(static_cast<Int32>(GetEffectiveNumberOfItems()));
   Int32 newFocus = focusedIndex;

   if ((startIndex <= focusedIndex) && (focusedIndex < (startIndex + numberOfItems)))
   {
      // the focus item is still visible
   }
   else
   {
      if (movementDistance > 0)
      {
         newFocus = startIndex + numberOfItems - 1;
      }
      else
      {
         // movementDistance <= 0
         newFocus = startIndex;
      }

      if (focusedIndex != newFocus)
      {
         CheckAndSetFocusedIndex(newFocus);
         _itemsInvalid = true;
         InternalInvalidate();
      }
   }
}


void ListWidget2D::UpdateStartIndex(ListWidget2DBase::StartIndexType startIndex)
{
   CheckAndSetStartIndex(startIndex);
   _itemsInvalid = true;
   InternalInvalidate();
}


FeatStd::UInt32 ListWidget2D::GetTimeMs()
{
   if (0 == _frameTime)
   {
      _frameTime = hmibase::utils::Ticker::getTickCountMsec();
   }
   return _frameTime;
}


Candera::UInt32 ListWidget2D::GetMaxPosition()
{
   return _listContentUpdater.GetMaxPosition();
}


FeatStd::Int32 ListWidget2D::GetEffectiveNumberOfItems() const
{
   return _listContentUpdater.GetNumberOfCompletelyVisibleItems();
}


void ListWidget2D::SetListener(FlexScrollableListener* listener)
{
   _listener = listener;
   _swipingStrategy.SetOverscrollerListener(listener);
}


void ListWidget2D::NotifyOnPositionReached()
{
   if (0 != _listener)
   {
      _listener->OnPositionReached();
   }
}


void ListWidget2D::NotifyOnContentUpdated()
{
   if (0 != _listener)
   {
      _listener->OnContentUpdated();
   }
}


void ListWidget2D::OnAnimationTimeDispatcherChanged()
{
   Animation::AnimationTimeDispatcher::SharedPointer animationTimeDispatcher(GetAnimationTimeDispatcher());

   _listContentUpdater.SetTimeDispatcher(animationTimeDispatcher);
   _expandTriggerManager.SetTimeDispatcher(animationTimeDispatcher);
}


void ListWidget2D::OnAnimationFinished()
{
   NotifyOnPositionReached();

   // track the list operation
   if ((_currentListStatus == ListEncoder) || (_currentListStatus == ListScrollBar) || (_currentListStatus == ListStatusUnknown))
   {
      OnListStateChanged(ListIdle);
   }

   _restoreContentUpdaterState = true;
}


bool ListWidget2D::AreAllElementsVisible()
{
   return (!_listContentUpdater.IsContentAdded()) || (GetCompleteSize() <= GetVisibleSize());
}


class UpdateTraverser : public Candera::TreeTraverser2D
{
      FEATSTD_LINT_CURRENT_SCOPE(1712, "No default constructor needed.")

   public:
      UpdateTraverser(ViewScene2D& viewScene2D) :
         _viewScene2D(viewScene2D)
      {
      }

      virtual ~UpdateTraverser()
      {
      }

   protected:
      virtual TraverserAction ProcessNode(Node2D& node)
      {
         WidgetBaseEnumerator en(ControlTemplate::EnumerateAssociatedWidgets(_viewScene2D, node));

         while (en.MoveNext())
         {
            WidgetBase* widget(en.Current());
            if (0 != widget)
            {
               widget->Update();
            }
         }
         return ProceedTraversing;
      }

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(UpdateTraverser);

      ViewScene2D& _viewScene2D;
};


void ListWidget2D::OnDataChanged(StartIndexType requestedStartIndex)
{
   if (GetDynamicGrid())
   {
      SetStartIndex(0);
   }

   UpdateTemplateWidgets();

   _listContentUpdater.RefreshOnDataChanged();

   if ((_movementStatus != ListMovementSwiping) && (_movementStatus != ListMovementScrolling))
   {
      UpdateContent();
   }

   const Int32 startIndex(static_cast<Int32>(_listContentUpdater.GetIndexOfFirstCompletelyVisibleElement()));
   if ((requestedStartIndex >= 0) && (requestedStartIndex != startIndex))
   {
      ListChangeMsg posSet(GetListId(), ListChangeSet, requestedStartIndex);
      OnMessage(posSet);
   }
}


void ListWidget2D::UpdateTemplateWidgets()
{
   if (!_updateCalled)
   {
      View* parentView = GetParentView();
      if (0 != parentView)
      {
         ViewScene2D* viewScene2D(parentView->ToViewScene2D());
         if (0 != viewScene2D)
         {
            UpdateTraverser traverser(*viewScene2D);
            Node2D* invalidItemTemplate = GetInvalidItemTemplate();
            Node2D* templateGroup = GetTemplateGroup();
            if (0 != invalidItemTemplate)
            {
               traverser.Traverse(*invalidItemTemplate);
            }
            if (0 != templateGroup)
            {
               traverser.Traverse(*templateGroup);
            }
         }
      }
   }
}


void ListWidget2D::OnParentViewRenderingEnabled(bool enable)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2d %p shown=%d\n", this, enable));

   if (enable)
   {
      RetrieveDataFromCache();
      _itemsInvalid = true;
   }

   _firstSetAfterRenderingEnabled = enable;
   RequestImmediatePositioning(enable);

   _receivedListCustomAnimationReqMsg = false;

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2d %p shown=%d finished handler\n", this, enable));
}


void ListWidget2D::RequestImmediatePositioning(bool active)
{
   _immediatePositioningRequested = active;
}


void ListWidget2D::UpdateScrollingSpeed()
{
   bool sliderSet(_listChangeType == ::ListChangeSet);

   _scrollingSpeed = ListContentUpdater::Slow;

   if (sliderSet)
   {
      _scrollingSpeed = ListContentUpdater::Fast;
   }

   if (ImmediatePositioningSet())
   {
      _scrollingSpeed = ListContentUpdater::Immediate;
   }

   if (sliderSet)
   {
      _firstSetAfterRenderingEnabled = false;
   }
}


void ListWidget2D::SetStartIndexInternally(StartIndexType startIndex)
{
   _startIndexInternallySet = (startIndex != GetStartIndex());
   CheckAndSetStartIndex(startIndex);
}


void ListWidget2D::HideNode(Candera::Node2D* node) const
{
   if (0 != node)
   {
      if (node->IsRenderingEnabled())
      {
         node->SetRenderingEnabled(false);
         Layouter::SetCollapsible(*node, true);
      }
   }
}


ListDataProviderCache* ListWidget2D::sListDataProviderCache = 0;

void ListWidget2D::SetListDataProviderCache(ListDataProviderCache* listDataProviderCache)
{
   sListDataProviderCache = listDataProviderCache;
}


void ListWidget2D::RetrieveDataFromCache()
{
   Courier::Message* lMsg = GetCachedDataMessage();
   if (0 != lMsg)
   {
      OnMessage(*lMsg);
   }
}


Courier::Message* ListWidget2D::GetCachedDataMessage()
{
   Courier::Message* msg = 0;

   if (0 != sListDataProviderCache)
   {
      Courier::MessageReferrer msgRef;
      if (sListDataProviderCache->checkForCachedData(GetListId(), msgRef))
      {
         msg = msgRef.GetMessage();
      }
   }

   return msg;
}


const PagesInfo& ListWidget2D::GetPagesInfo() const
{
   return _listContentUpdater.GetPagesInfo();
}


static ListMovementStatusType GetMovement(ListChangeType changeType)
{
   ListMovementStatusType movementType(ListMovementFinished);

   switch (changeType)
   {
      case ListChangeUp:
      case ListChangeDown:
         movementType = ListMovementItem;
         break;
      case ListChangePageUp:
      case ListChangePageDown:
         movementType = ListMovementPage;
         break;
      case ListChangeSet:
         movementType = ListMovementSet;
         break;
      default:
         movementType = ListMovementFinished;
         break;
   }

   return movementType;
}


static ListStatusDirectionType GetDirection(const ListChangeMsg& changeMsg, ListWidget2DBase::StartIndexType startIndex)
{
   ListStatusDirectionType direction(ListStatusDirectionUnavailable);

   //track operation direction
   switch (changeMsg.GetListChangeType())
   {
      case ListChangeDown:
      case ListChangePageDown:
         direction = ListStatusDirectionNext;
         break;
      case ListChangeUp:
      case ListChangePageUp:
         direction = ListStatusDirectionPrev;
         break;

      case ListChangeSet:
      {
         ListWidget2DBase::StartIndexType requestedStartIndex(changeMsg.GetValue());
         direction = (requestedStartIndex < startIndex) ? ListStatusDirectionPrev : ListStatusDirectionNext;
         break;
      }

      default:
         direction = ListStatusDirectionUnavailable;
         break;
   }

   return direction;
}


static ListGeneralChangeType GetChangeType(const ListChangeMsg& changeMsg)
{
   ListGeneralChangeType changeType(ListItemChange);

   switch (changeMsg.GetListChangeType())
   {
      case ListChangeDown:
      case ListChangeUp:
         changeType = ListItemChange;
         break;

      case ListChangePageDown:
      case ListChangePageUp:
         changeType = ListPageChange;
         break;

      case ListChangeSet:
         changeType = ListSetChange;
         break;

      default:
         break;
   }

   return changeType;
}


void ListWidget2D::OnListChangeMsg(const ListChangeMsg* lstChangeMsg, bool& msgConsumed, bool& shouldForwardMsg)
{
   if (ShouldProcessListChangeMsg() && (0 != lstChangeMsg) && ((lstChangeMsg->GetListId() == 0) || (lstChangeMsg->GetListId() == GetListId())))
   {
      _lockoutManager.UpdateOperationDirection(GetDirection(*lstChangeMsg, GetStartIndex()));
      _statusChanged = true;

      _changeType = GetChangeType(*lstChangeMsg);
      _currentListChangeMsgSourceType = lstChangeMsg->GetListChangeMsgSource();
      const bool moveAllowed(_lockoutManager.IsMoveAllowed(_changeType, _currentListChangeMsgSourceType));

      if (moveAllowed)
      {
         ResetStatemachineToIdle();
      }

      UpdateListState(lstChangeMsg->GetListChangeMsgSource());

      if (moveAllowed)
      {
         ListChangeMsg msg(lstChangeMsg->GetListId(), lstChangeMsg->GetListChangeType(), lstChangeMsg->GetValue(), lstChangeMsg->GetListChangeMsgSource(), lstChangeMsg->GetImmediatePositioning());
         _lockoutManager.AlterListChangeMsg(msg);

         ProcessListChangeMsg(msg, msgConsumed, shouldForwardMsg);
      }
      else
      {
         _listContentUpdater.UpdateLocked();
      }
   }
}


void ListWidget2D::OnListAnimatedChangeReqMsg(const ListAnimatedChangeReqMsg& lstChangeMsg, bool& msgConsumed, bool& shouldForwardMsg)
{
   View* parentView(GetParentView());
   if (0 != parentView)
   {
      IViewHandler* viewHandler(parentView->GetViewHandler());
      if (0 != viewHandler)
      {
         ListChangeMsg changeMsg(lstChangeMsg.GetListId(), lstChangeMsg.GetListChangeType(), lstChangeMsg.GetValue(), lstChangeMsg.GetListChangeMsgSource(), lstChangeMsg.GetImmediatePositioning());
         OnListChangeMsg(&changeMsg, msgConsumed, shouldForwardMsg);

         ListMovementDetails details(lstChangeMsg.GetMovementDetails());

         _scrollingSpeed = ListContentUpdater::Fast;
         Int32 duration = details._duration;
         if (duration <= 0)
         {
            duration = GetPixelWiseAnimationTime();
         }
         _listContentUpdater.SetShortPixelWiseAnimationTime(duration);

         const ::Candera::ListScrollingOrientation& scrollingOrientation(GetScrollingOrientation());
         const PixelWiseScrollingEnabledType& pixelWiseScrollingEnabled(details._pixelwise);
         _orientationOperator = OrientationOperator(scrollingOrientation, pixelWiseScrollingEnabled);
         _orthogonalOrientationOperator = OrientationOperator((scrollingOrientation == Vertical) ? Horizontal : Vertical, pixelWiseScrollingEnabled);

         _listContentUpdater.SetPixelwise(pixelWiseScrollingEnabled);
         _listContentUpdater.SetMovementDelay(details._delay);

         viewHandler->ExecuteAnimationAction(lstChangeMsg.GetExternalAnimationAction(), lstChangeMsg.GetViewId(), lstChangeMsg.GetCompositePath(), lstChangeMsg.GetAnimationId(), lstChangeMsg.GetAnimationProperties());
      }
   }
}


void ListWidget2D::ProcessListChangeMsg(const ListChangeMsg& lstChangeMsg, bool& msgConsumed, bool& shouldForwardMsg)
{
   _listChangeType = lstChangeMsg.GetListChangeType();
   Courier::Int32 const& value(lstChangeMsg.GetValue());

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: ListChangeMsg listId=%u, action=%u, value=%u", GetListId(), _listChangeType, value));

   msgConsumed = FlexListWidget2DGetFocusStrategy()(*this, lstChangeMsg, _forceMovement);
   _forceMovement = false;

   SetListMovement(GetMovement(_listChangeType));

   _immediatePositioning = lstChangeMsg.GetImmediatePositioning();

   UpdateScrollingSpeed();

   shouldForwardMsg = false;
}


void ListWidget2D::OnListDateProviderResMsg(const ListDateProviderResMsg* listDataResultMsg)
{
   if ((!listDataResultMsg->GetListDateProvider().PointsToNull()) && (GetListId() == listDataResultMsg->GetListDateProvider()->listId()))
   {
      if (etg_bIsTraceActive(TR_CLASS_HMI_LISTPROVIDER, ETG_LEVEL_USER_4))
      {
         ETG_TRACE_USR4_CLS_DCL((TR_CLASS_HMI_LISTPROVIDER, APP_TRACECLASS_ID(),
                                 "ListDateProviderResMsg response received for listId=%d, startIndex=%d virtualListSize=%d, setIndex=%d  items=%d cached=%d seqNr=%u",
                                 listDataResultMsg->GetListDateProvider()->listId(),
                                 listDataResultMsg->GetListDateProvider()->startIndex(),
                                 listDataResultMsg->GetListDateProvider()->virtualListSize(),
                                 listDataResultMsg->GetListDateProvider()->getListChangeSetIndex(),
                                 listDataResultMsg->GetListDateProvider()->listSize(),
                                 listDataResultMsg->GetListDateProvider()->isCacheEnabled() ? 1 : 0,
                                 listDataResultMsg->GetListDateProvider()->getSequenceNumber()
                                ));

         std::string strListProviderContent = listDataResultMsg->GetListDateProvider()->dumpContent().c_str();
         ETG_TRACE_USR4_CLS_DCL((TR_CLASS_HMI_LISTPROVIDER, APP_TRACECLASS_ID(), "%s", strListProviderContent.c_str()));
      }
      else
      {
         ETG_TRACE_COMP_DCL((APP_TRACECLASS_ID(),
                             "ListDateProviderResMsg response received for listId=%d, startIndex=%d virtualListSize=%d, setIndex=%d  items=%d cached=%d seqNr=%u",
                             listDataResultMsg->GetListDateProvider()->listId(),
                             listDataResultMsg->GetListDateProvider()->startIndex(),
                             listDataResultMsg->GetListDateProvider()->virtualListSize(),
                             listDataResultMsg->GetListDateProvider()->getListChangeSetIndex(),
                             listDataResultMsg->GetListDateProvider()->listSize(),
                             listDataResultMsg->GetListDateProvider()->isCacheEnabled() ? 1 : 0,
                             listDataResultMsg->GetListDateProvider()->getSequenceNumber()
                            ));
      }

      _listData.OnListDataResultMsg(*listDataResultMsg);

      /// behavior when list empty req for a new list
      //_requestPending = false;
      //_itemsInvalid = true;
      //_dataProviderInvalid = !_listData.IsDataAvailable();

      _dataReceived = _listData.IsDataAvailable();
      _dataProviderInvalid = !_dataReceived;

      if (_dataReceived)
      {
         _requestPending = false;
      }
      _itemsInvalid = true;

      FeatStd::Int32 idx = listDataResultMsg->GetListDateProvider()->getListChangeSetIndex();
      OnDataChanged(idx);

      InternalInvalidate();
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: set _dataProviderInvalid to: %s", _dataProviderInvalid ? "true" : "false"));
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: ListDateProviderResMsg received in List %d", GetListId()));

      // Scene transition : inform the mediator via view controller, that data is available and view can be shown
      if (0 != GetParentView())
      {
         Courier::ViewController* viewController = GetParentView()->GetViewController();
         if (0 != viewController)
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: Fire TransitionMediatorOnDataCompleteMsg listId=%u", GetListId()));
            TransitionMediatorOnDataCompleteMsg msg(GetListId());
            viewController->OnMessage(msg);
         }
      }

      PostListContentUpdMsg();
   }
}


void ListWidget2D::OnListDateProviderUpdMsg(const ListDataProviderUpdMsg* listDataUpateMsg)
{
   if ((!listDataUpateMsg->GetUpdater().PointsToNull()) && (GetListId() == listDataUpateMsg->GetUpdater()->listId()))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Data provider updater received for ListId=%u", GetListId()));

      ListDataProviderUpdater::ChangedItemsType changedItems;
      _listData.OnListDataUpdateMsg(*listDataUpateMsg, changedItems);

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Updated %u items for ListId=%u", changedItems.size(), GetListId()));

      if (!changedItems.empty())
      {
         OnDataChanged(-1);
         _itemsInvalid = true;

         InternalInvalidate();
         Update();

         PostListContentUpdMsg();
      }
   }
}


void ListWidget2D::OnParentViewRenderingEnabledEvent(const ParentViewRenderingEnabledEvent* parentViewRenderingEnableEvent, bool& msgConsumed, bool& shouldForwardMsg)
{
   bool _currentParentViewRenderingEnabled = parentViewRenderingEnableEvent->GetEnabled();
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: %p ParentViewRenderingEnabledEvent LISTID=%d, ENABLED=%s", this, GetListId(), _currentParentViewRenderingEnabled ? "On" : "off"));
   msgConsumed = true;
   shouldForwardMsg = false;
}


void ListWidget2D::OnParentViewActivateEvent(const ParentViewActivateEvent* parentViewActivateEvent, bool& msgConsumed, bool& shouldForwardMsg)
{
   bool activated = parentViewActivateEvent->GetActivated();
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: %p ParentViewActivateEvent LISTID=%d, ACTIVATE=%s", this, GetListId(), activated ? "On" : "off"));
   if (activated && !_activ)
   {
      _requestPending = false;
      _dataProviderInvalid = true;
      ValidateDataProvider();
      _itemsInvalid = true;
      InternalInvalidate();
   }
   _activ = activated;
   msgConsumed = true;
   shouldForwardMsg = false;
}


bool ListWidget2D::IsCircular()
{
   return GetScrollingType() == CircularScrolling;
}


void ListWidget2D::OnLockOutMsg(const LockOutMsg* lockOutMsg)
{
   const UInt32& receiverId(lockOutMsg->GetReceiverId());

   if ((0 != lockOutMsg) && ((receiverId == 0) || (receiverId == GetListId())))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::OnLockOutMsg, listId=%u", GetListId()));

      _lockoutManager.ProcessLockOutMsg(*lockOutMsg);

      const bool pagewiseLockout(_lockoutManager.IsPagewiseLockout());
      _swipingStrategy.SetPagewiseLockOut(pagewiseLockout);
      _listContentUpdater.SetPageWiseLockOut(pagewiseLockout);

      if (0 != _snapper)
      {
         _snapper->SetPagewiseSnap(pagewiseLockout);
      }

      if (lockOutMsg->GetLockOut() == LockOutFixMovement)
      {
         _swipingStrategy.SetLockOutFixMovementOffset(GetLockOutFixMovementOffset());
      }
      else
      {
         _swipingStrategy.SetLockOutFixMovementOffset(0);
      }
   }

   _statusChanged = true;

   _swipingStrategy.BuildSwipingBehavior();
}


void ListWidget2D::TryPostListStatusUpdMsg()
{
   if (_statusChanged)
   {
      ListStatusUpdMsg* msg = COURIER_MESSAGE_NEW(ListStatusUpdMsg)(ListStatusType(_currentListStatus), GetListId(), (0 != GetParentView()) ? GetParentView()->GetId() : ViewId(), Courier::Identifier(GetLegacyName()), _lockoutManager.GetCurrentDirection(), _lockoutManager.IsAnyLockout());

      bool msgPosted((0 != msg) && (msg->Post()));

      if (!msgPosted)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D %p id=%d PostListStatusUpdMsg unsuccessful", this, GetListId()));
      }

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: Post ListStatusUpdMsg listId=%u, listStatus=%u", GetListId(), _currentListStatus));

      // if the message was not posted, the status remains marked as changed; otherwise is reset
      _statusChanged = (!msgPosted);
   }
}


void ListWidget2D::OnListItemExpandReqMsg(const ListItemExpandReqMsg* expandReqMsg)
{
   if ((0 != expandReqMsg) && (expandReqMsg->GetListId() == GetListId()))
   {
      Int32 absoluteItemIndex(expandReqMsg->GetItemIndex());
      Int32 itemIndex(absoluteItemIndex - _listContentUpdater.GetFirstItemIndex());
      bool forward(expandReqMsg->GetForward());
      const tSharedPtrListDataItem& item(_contentProvider.GetItem(absoluteItemIndex));
      if ((itemIndex >= 0) && _expandTriggerManager.MarkToStart(itemIndex, forward))
      {
         _itemsInvalid = true;
         InternalInvalidate();
         if (!item.PointsToNull())
         {
            item->SetCollapsed(!forward);
         }
      }
      else if (itemIndex < 0)
      {
         if (!item.PointsToNull())
         {
            item->SetCollapsed(!forward);
         }
      }

      _listContentUpdater.RefreshOnListItemExpandReqMsg();
   }
}


void ListWidget2D::TryPostListChangedMsg()
{
   if (GetPostListChanged() && (GetParentView() != NULL))
   {
      Int32 first(_listContentUpdater.GetFirstItemIndex());
      const UInt32 maxPosition(GetMaxPosition());

      if (first < 0)
      {
         first += maxPosition;
      }

      if ((_firstElementIndex != first) || (_oldMovementStatus != ListMovementFinished && _movementStatus == ListMovementFinished))
      {
         // _listData.UpdateSetListChangeSetIndex(first);
         ListChangedUpdMsg* msg = COURIER_MESSAGE_NEW(ListChangedUpdMsg)(GetListId(), _movementStatus, IsCircular(), first, first, (GetStartIndex() == first), 0, maxPosition, _listContentUpdater.GetDataSequenceNumber());
         if (0 != msg)
         {
            //If GetPostListChanged is true, the list will post a ListChangedUpdMsg message every time the first visible index (startIndex) changes.
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: Post ListChangedUpdMsg listId=%u, movementStatus=%u, circular=%u, firstVisibleIndex=%u, minPos=%u, maxPos=%u, seqNr=%u", GetListId(), _movementStatus, IsCircular(), first, 0, maxPosition, _listContentUpdater.GetDataSequenceNumber()));

            static_cast<void>(msg->Post());
         }

         _firstElementIndex = first;
      }
   }
}


void ListWidget2D::PostListContentUpdMsg()
{
   ListContentUpdMsg* msg = COURIER_MESSAGE_NEW(ListContentUpdMsg)(GetListId(), GetPosition(), GetMaxPosition(), _listContentUpdater.GetDataSequenceNumber());
   if (msg != 0)
   {
      msg->Post();
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: Post ListContentUpdateMsg listId=%u, firstVisibleIndex=%u, maxFirstVisibleIndex=%u, seqNr=%u", GetListId(), GetPosition(), GetMaxPosition(), _listContentUpdater.GetDataSequenceNumber()));
   }
}


void ListWidget2D::UpdateEffectiveVisibiliy(bool forceInvisible)
{
   //get the node the list is bound to
   ListStatusVisibilityType newVisibility = ListInVisible;

   Node2D* node = GetNode();
   if (0 != node)
   {
      newVisibility = node->IsEffectiveRenderingEnabled() ? ListVisible : ListInVisible;
   }

   if (forceInvisible)
   {
      newVisibility = ListInVisible;
   }
   //check if the effective visibility changed
   if (newVisibility != _currentEffectiveVisible)
   {
      _currentEffectiveVisible = newVisibility;
      //notify that the effective visibility changed
      ListVisibiltyUpdMsg* msg = COURIER_MESSAGE_NEW(ListVisibiltyUpdMsg)(_currentEffectiveVisible, GetListId(), (0 != GetParentView()) ? GetParentView()->GetId() : ViewId(), Courier::Identifier(GetLegacyName()));
      if (0 != msg)
      {
         static_cast<void>(msg->Post());
      }
   }
}


void ListWidget2D::UpdateListState(ListChangeMsgSourceType source)
{
   ListStatusType state = ListIdle;
   switch (source)
   {
      case ListChangeMsgSourceEncoder:
         state = ListEncoder;
         break;
      case ListChangeMsgSourceScrollBar:
         state = ListScrollBar;
         break;
      case ListChangeMsgSourceUnknown:
         state = ListStatusUnknown;
         break;
      default:
         break;
   }
   OnListStateChanged(state);
}


bool ListWidget2D::UpdateExpandAnimations(ExpandAdjuster::AdjustData& expandData)
{
   bool animationsRunning(false);
   const Animation::AnimationTimeDispatcher::SharedPointer& animationTimeDispatcher(GetAnimationTimeDispatcher());

   if (!animationTimeDispatcher.PointsToNull())
   {
      Vector<ControlTemplateInstancePtr>& instances(_itemsInstanceContainer.GetInstances());
      bool lastIsRunning(false);
      bool lastIsForward(false);
      Int32 lastRunningIndex(-1);
      const bool lastViewport(_listContentUpdater.GetLastItemIndex() == (_contentProvider.GetVirtualListSize() - 1));

      for (Vector<ControlTemplateInstancePtr>::Iterator it(instances.Begin()); it != instances.End(); ++it)
      {
         ControlTemplateInstancePtr& instance(*it);
         if (!instance.PointsToNull())
         {
            Node2D* node(instance->GetItemRootNode());
            if ((0 != node) && (node->GetParent() != 0))
            {
               lastIsRunning = false;
               const Int32 instanceIndex(static_cast<Int32>(instance->GetIndex()));

               AnimationsContainer& expandAnimations(instance->GetExpandAnimationsManager());
               expandAnimations.SetWorldTime(animationTimeDispatcher->GetWorldTimeMs());
               expandAnimations.DispatchWorldTime();

               lastIsRunning = expandAnimations.AreRunning() || instance->TestExpandPlaybackFinish();
               animationsRunning = animationsRunning || lastIsRunning;
               tSharedPtrListDataItem data(Dynamic_Cast<tSharedPtrListDataItem>(ControlTemplate::GetDataContext(*node, true)));

               if (lastIsRunning)
               {
                  if (!data.PointsToNull())
                  {
                     lastIsForward = !data->IsCollapsed();
                  }

                  lastRunningIndex = instanceIndex;
               }
            }
         }
      }

      const bool newExpandStartFrame(animationsRunning &&                                                // When current frame has running expand animation and
                                     ((!_expandAnimationRunning)                                                                     // Previous frame had 0 expand animations
                                      ||                                                                                              // or
                                      (_expandAnimationRunning && (lastRunningIndex != _expandAdjuster.GetAdjustingItemIndex()))));   // Another expand of an item closer tho the end started while first one is not finished

      expandData = { lastViewport, lastIsRunning, lastIsForward, lastRunningIndex };

      if (animationsRunning && GetExpandAutoScroll())
      {
         _expandAdjuster.PrepareAdjustment(lastViewport);
      }

      if (newExpandStartFrame)
      {
         // First expand frame of last item in the view
         // Check the end margin of expanded item for a initial margin reference and index of the last item
         _expandAdjuster.ExpandStarted(expandData, _listContentUpdater.GetLastItemMargin());
      }
   }
   return animationsRunning;
}


bool ListWidget2D::UpdateCustomAnimations()
{
   bool running(false);

   bool visible(IsEffectiveVisible());

   if (_dataReceived && visible)
   {
      const Animation::AnimationTimeDispatcher::SharedPointer& animationTimeDispatcher(GetAnimationTimeDispatcher());

      if ((!animationTimeDispatcher.PointsToNull()))
      {
         _animationContainer.Update(GetTimeMs());

         running = _animationContainer.IsRunning();
      }

      if ((!GetShowItemsOnViewActivated()) && _receivedListCustomAnimationReqMsg)
      {
         Node2D* itemsNode(GetItemsNode());
         if (0 != itemsNode)
         {
            itemsNode->SetRenderingEnabled(true);
         }
      }
   }
   return running;
}


bool ListWidget2D::IsEffectiveVisible() const
{
   const View* parentView(GetParentView());
   bool visible((0 != parentView) && (parentView->IsRenderingEnabled()));

   return visible;
}


void ListWidget2D::CheckListLimits()
{
   if (IsCircular())
   {
      return;
   }

   Int32 firstVisibleIndex = GetStartIndex();
   Int32 lastFirstVisibleIndex = GetCompleteSize() - GetVisibleSize();
   bool bounce = (lastFirstVisibleIndex > 0);

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D:CheckListLimits instance=%p listId=%d, firstVisibleIndex=%d, lastFirstVisibleIndex=%d", this, GetListId(), firstVisibleIndex, lastFirstVisibleIndex));

   if (bounce && (firstVisibleIndex == 0))
   {
      OnLimitReachedUp();
   }

   //adjust start to not go before first item
   else if (firstVisibleIndex < 0)
   {
      if (bounce)
      {
         OnLimitExceededUp();
      }
      SetStartIndexInternally(0);
   }
   else if (bounce && (firstVisibleIndex == lastFirstVisibleIndex))
   {
      OnLimitReachedDown();
   }
   //adjust focus to not go after the last item
   else if (firstVisibleIndex > lastFirstVisibleIndex)
   {
      if (bounce)
      {
         OnLimitExceededDown();
      }
      SetStartIndexInternally(lastFirstVisibleIndex);
   }
   else
   {
      //nothing to do
   }
}


void ListWidget2D::ResetStatemachineToIdle()
{
   _swipingStrategy.Reset();
}


void ListWidget2D::SetListMovement(ListMovementStatusType listMovement)
{
   if (listMovement != _movementStatus)
   {
      _oldMovementStatus = _movementStatus;
      _movementStatus = listMovement;
   }
}


bool ListWidget2D::IsListMovementFinished() const
{
   return ((_oldMovementStatus != _movementStatus) && (_movementStatus == ListMovementFinished));
}


void ListWidget2D::ForceMovement()
{
   _forceMovement = true;
}


bool ListWidget2D::ImmediatePositioningSet() const
{
   bool sliderSet(_listChangeType == ::ListChangeSet);
   bool pageChange((_listChangeType == ::ListChangePageDown) || (_listChangeType == ::ListChangePageUp));
   bool itemChange((_listChangeType == ::ListChangeDown) || (_listChangeType == ::ListChangeUp));

   const bool acceptImmediatePositioning = GetAcceptImmediatePositioning();
   bool immediate = false;
   if ((acceptImmediatePositioning && _immediatePositioningRequested))
   {
      immediate = immediate || (sliderSet && _firstSetAfterRenderingEnabled && GetAcceptImmediatePositioningOnFirstApperance());
      immediate = immediate || (sliderSet && GetAcceptImmediatePositioningOnPositionSet());
      immediate = immediate || (pageChange && GetAcceptImmediatePositioningOnPageScroll());
      immediate = immediate || (itemChange && GetAcceptImmediatePositioningOnItemScroll());
      immediate = immediate || _lockoutManager.AcceptImmediatePositioning();
      immediate = immediate || _immediatePositioning;
   }

   return immediate;
}


void ListWidget2D::UpdateSnapper()
{
   if (_snapper != 0)
   {
      FEATSTD_DELETE(_snapper);
      _snapper = 0;
   }

   switch (GetSnap())
   {
      case ListSnapType::SnapToCenter:
         _snapper = (HasScrollAnimations()) ? FEATSTD_NEW(AnimatedCenterItemSnapper)(_listContentUpdater) : FEATSTD_NEW(CenterItemSnapper)(_listContentUpdater);
         break;

      case ListSnapType::SnapToEdge:
         _snapper = FEATSTD_NEW(EdgeItemSnapper)(_listContentUpdater);
         break;

      default:
         _snapper = FEATSTD_NEW(DefaultItemSnapper)(_listContentUpdater);
         break;
   }

   if (_snapper != 0)
   {
      _snapper->SetPagewiseSnap(GetFixedPageScrolling() || _lockoutManager.IsPagewiseLockout());
      _snapper->SetThreshold(GetSnapOffsetThreshold());
      _snapper->SetSnapHoldMaxTime(GetSnapHoldTime());
      _snapper->SetNormalizer(_normalizer);
   }
}


void ListWidget2D::UpdateNormalizer()
{
   if (0 != _normalizer)
   {
      FEATSTD_DELETE(_normalizer);
      _normalizer = 0;
   }

   if (GetCoverflow())
   {
      _normalizer = FEATSTD_NEW(CoverflowNormalizer);
   }
   else
   {
      _normalizer = FEATSTD_NEW(DefaultNormalizer);
   }

   _listContentUpdater.SetNormalizer(_normalizer);
   if (0 != _snapper)
   {
      _snapper->SetNormalizer(_normalizer);
   }

   _swipingStrategy.SetNormalizer(_normalizer);
   _expandAdjuster.SetNormalizer(_normalizer);
}


bool ListWidget2D::HasScrollAnimations() const
{
   bool scrollAnimationFound = false;
   FeatStd::UInt8 i = 0;
   while (!scrollAnimationFound && i < _animationArray.GetCount())
   {
      scrollAnimationFound = (ListItemAnimationDynamicPropertyHost::GetListItemAnimationType(_animationArray.Get(i)) == LIAScroll);
      i++;
   }

   return scrollAnimationFound;
}


void ListWidget2D::HideTemplates()
{
   HideNode(GetTemplateGroup());
   HideNode(GetInvalidItemTemplate());
   HideNode(GetCustomAnimationsGroupNode());
}


bool ListWidget2D::IsTouchCameraValid()
{
   bool cameraValid = false;
   if (_touchCamera != 0 && GetParentView() != 0)
   {
      Courier::ViewScene2D* viewScene2D = GetParentView()->ToViewScene2D();
      if (viewScene2D != 0)
      {
         cameraValid = viewScene2D->FindCamera2D(_touchCamera->GetName()) == _touchCamera;
      }
   }

   return cameraValid;
}


bool ListWidget2D::IsBouncing() const
{
   return (_currentLimitAnimationPlayerId < LimitAnimationPlayerCount) && IsAnimationRunning(_currentLimitAnimationPlayer[_currentLimitAnimationPlayerId].player);
}


void ListWidget2D::ValidateItemsGroup() const
{
   const Node2D* itemsNode(GetItemsNode());

   if (0 == itemsNode)
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ListWidget2D:ValidateItemsGroup instance=%p id=%d - items node is null", this, GetListId()));
   }
   else
   {
      const Node2D* parent(itemsNode->GetParent());
      if (0 == parent)
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ListWidget2D:ValidateItemsGroup instance=%p id=%d - parent of items node is null", this, GetListId()));
      }
      else
      {
         Layouter* layouter(parent->GetLayouter());
         if ((0 == layouter) || (DefaultLayouter::GetInstance() == layouter))
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ListWidget2D:ValidateItemsGroup instance=%p id=%d - parent of items node must have a layouter set and it must be different than the DefaultLayouter", this, GetListId()));
         }
      }
   }
}


void ListWidget2D::UpdateCachedCloningBehavior()
{
   const PreparedItemsBufferSizeType& preparedItemsBufferSize(GetPreparedItemsBufferSize());

   if (preparedItemsBufferSize > 0)
   {
      _cachedCloningBehavior = PrepareClonedItemCachingBehavior::SharedPointer(FEATSTD_NEW(PrepareClonedItemCachingBehavior)(_ownerInfoRetriever, _contentProvider, _templateRetriever, _ownerInfoRetriever.GetOwner(), _ownerInfoRetriever.GetOwnerId(), _sizesContainer, preparedItemsBufferSize, GetPreparedItemsUpdateTriggerOffset()));
   }
   else
   {
      _cachedCloningBehavior = DefaultClonedItemCachingBehavior::SharedPointer(FEATSTD_NEW(DefaultClonedItemCachingBehavior)(_ownerInfoRetriever));
   }

   if (!_cachedCloningBehavior.PointsToNull())
   {
      _itemsInstanceContainer.SetCloningStrategy(&(_cachedCloningBehavior->GetCloningStrategy()));
   }
}


void ListWidget2D::OnNodeChanged()
{
   Candera::Node2D* node(GetNode());
   if (0 != node)
   {
      ControlTemplate::AddAssociatedWidget(*node, this);
   }
}


bool ListWidget2D::ShouldProcessListChangeMsg()
{
   SwipingStateChecker* checker(_swipingStrategy.GetSwipingStateChecker());
   bool oldShould(((0 != checker) && (checker->IsIdle())) || (_listContentUpdater.GetLastItemIndex() >= _listData.GetVirtualListSize()));
   bool shouldProcess((0 != checker) && (checker->IsIdle()));

   IgnoreListChangeMsgType ignore(GetIgnoreListChangeMsg());

   switch (ignore)
   {
      case IgnoreListChangeMsgType::WhileTouching:
         shouldProcess = ((0 != checker) && (!checker->IsScrolling()));
         break;

      case IgnoreListChangeMsgType::WhileTouchingOrSwiping:
         shouldProcess = ((0 != checker) && (checker->IsIdle()));
         break;

      case IgnoreListChangeMsgType::Never:
      default:
         shouldProcess = true;
         break;
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D::ShouldProcessListChangeMsg instance=%p id=%d oldShould=%d should=%d", this, GetListId(), oldShould, shouldProcess));

   return shouldProcess;
}


void ListWidget2D::UpdateViewportSize()
{
   Node2D* invalidTemplate(GetInvalidItemTemplate());
   Node2D* itemsContainerNode(GetItemsNode());
   if (GetAutoViewportSize() && (0 != invalidTemplate) && (0 != itemsContainerNode))
   {
      const Float size(_orientationOperator.GetVectorComponent(_sizesContainer.MeasureClientSize(invalidTemplate)));
      const Float numberOfItems(static_cast<Float>(GetNumberOfItems()));

      const Int32 newViewportSize(static_cast<Int32>(size * numberOfItems));

      Vector2 viewportSize(Layouter::GetSize(*itemsContainerNode));
      _orientationOperator.SetVectorComponent(viewportSize, newViewportSize);

      Layouter::SetSize(*itemsContainerNode, viewportSize);
   }
}


namespace FeatStd {
template<> ::FeatStd::UInt32 StringBufferAppender< ::ListMovementDetails >::Append(::FeatStd::StringBuffer& stringBuffer, ::ListMovementDetails const& object)
{
   ::FeatStd::UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append("::ListMovementDetails {");
   tcharCount += stringBuffer.Append("pixelwise = ");
   tcharCount += stringBuffer.AppendObject(object._pixelwise);
   tcharCount += stringBuffer.Append(", ");
   tcharCount += stringBuffer.Append("delay = ");
   tcharCount += stringBuffer.AppendObject(object._delay);
   tcharCount += stringBuffer.Append(", ");
   tcharCount += stringBuffer.Append("duration = ");
   tcharCount += stringBuffer.AppendObject(object._duration);
   tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


}
