/* ***************************************************************************************
* FILE:          ListProviderEventInfo.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ListProviderEventInfo is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "ListProviderEventInfo.h"

#include "Courier/Platform/CriticalSection.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ListProviderEventInfo.cpp.trc.h"
#endif

// FEATSTD_LOG_SET_REALM(WidgetLogRealm::WIDGET_LIST);

typedef std::map< unsigned, ListProviderEventInfo> ButtonEventMap;

static ButtonEventMap  _buttonEventMap;
class SectionLocker {
   public:
      SectionLocker()
      {
         getLock().Obtain();
      }
      ~SectionLocker()
      {
         getLock().Release();
      }
   private:
      Courier::Platform::CriticalSection& getLock()
      {
         static Courier::Platform::CriticalSection criticalSection;
         return criticalSection;
      }
};

ListProviderEventInfo::ListProviderEventInfo()
   : _listId(0), _hdlRow(0), _hdlCol(0), _flags(0)
{
   setInValid();
}


ListProviderEventInfo::ListProviderEventInfo(unsigned listId, ServersDataItemHdlRow hdlRow, ServersDataItemHdlCol hdlCol, ServersDataItemFlags flags)
   : _listId(listId), _hdlRow(hdlRow), _hdlCol(hdlCol), _flags(flags)
{
}


unsigned ListProviderEventInfo::getListId() const
{
   return _listId;
}


ServersDataItemHdlRow ListProviderEventInfo::getHdlRow() const
{
   return _hdlRow;
}


ServersDataItemHdlCol ListProviderEventInfo::getHdlCol() const
{
   return _hdlCol;
}


ServersDataItemFlags ListProviderEventInfo::getFlags() const
{
   return _flags;
}


bool ListProviderEventInfo::isValid() const
{
   return (_listId > 0);
}


void ListProviderEventInfo::setInValid()
{
   _hdlRow = 0;
   _hdlCol = 0;
   _listId = 0;
   _flags = 0;
}


void ListProviderEventInfo::SetItemIdentifier(const Courier::Identifier& buttonId, const ListProviderEventInfo ev)
{
   SectionLocker locker;
   _buttonEventMap[buttonId.HashValue()] = ev;
}


void ListProviderEventInfo::ClearItemIdentifier(const Courier::Identifier& buttonId)
{
   if (buttonId.IsValid())
   {
      SectionLocker locker;
      ButtonEventMap::iterator it = _buttonEventMap.find(buttonId.HashValue());
      if (it != _buttonEventMap.end())
      {
         _buttonEventMap.erase(it);
      }
   }
}


bool ListProviderEventInfo::GetItemIdentifierInfo(const Courier::Identifier& buttonId, ListProviderEventInfo& info, unsigned listIdToMatch)
{
   SectionLocker locker;
   ButtonEventMap::iterator it = _buttonEventMap.find(buttonId.HashValue());
   if (it == _buttonEventMap.end())
   {
      // ETG_TRACE_USR4(("ListProviderEventInfo::GetButtonEventInfo !! NOT FOUND !! HASH=%x  size=%d",
      //  buttonId.HashValue(), _buttonEventMap.size() ));
      return false;
   }
   ETG_TRACE_USR4(("ListProviderEventInfo::GetButtonEventInfo found HASH=%x ", buttonId.HashValue()));
   info = ListProviderEventInfo(it->second);
   if (listIdToMatch < 0xFFFF)
   {
      return (info.getListId() == listIdToMatch);
   }
   return true;
}


const char* ListProviderEventInfo::DebugDumpEntry(int i)
{
   static char tempbuf[60];

   ButtonEventMap::iterator it;
   int ii = 0;
   SectionLocker locker;
   int maxcount = _buttonEventMap.size();
   for (it = _buttonEventMap.begin(); it != _buttonEventMap.end(); ++it)
   {
      if (i == ii)
      {
         ListProviderEventInfo info = ListProviderEventInfo(it->second);
#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
         static_cast<void>(SNPRINTF(tempbuf, sizeof(tempbuf), " IteratorMAP[%d of %d] PDL_L%u_P1%ld_P2%ld_F%lx",
                                    ii, maxcount,
                                    info.getListId(), info.getHdlRow(), info.getHdlCol(), info.getFlags()));
#else
         static_cast<void>(SNPRINTF(tempbuf, sizeof(tempbuf), " IteratorMAP[%d of %d] PDL_L%u_P1%ld_P2%ld_F%x",
                                    ii, maxcount,
                                    info.getListId(), info.getHdlRow(), info.getHdlCol(), info.getFlags()));
#endif
         return tempbuf;
      }
      ii++;
   }
   return NULL;
}
