/* ***************************************************************************************
* FILE:          ListDataWrapper.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ListDataWrapper is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef LISTDATAWRAPPER_H
#define LISTDATAWRAPPER_H

#include <Candera/Candera.h>
#include "Widgets/2D/List/generated/ListMessages.h"
#include "Widgets/2D/List/Data/ListDataProvider.h"
#include "Widgets/2D/Common/FlexScrollable.h"

/**
 *  Provides a wrapper for handling data requests and access to the
 *  list data provider to trick the FlexListWidget2D into circular scrolling.
 */
class ListDataWrapper
{
   public:
      ListDataWrapper();
      virtual ~ListDataWrapper();

      /**
       * Reset the ListDataWrapper to its initial state.
       */
      void Clear();

      /**
       * Modifies startIndex and window size
       * @return true if the data needs updated; false otherwise
       */
      bool PostListDataRequestMsg(Candera::UInt32 listId, Candera::UInt32 startIndex, Candera::UInt32 bufferSize, Candera::UInt32 numberOfItems, Candera::ListScrollingType scrollingType, bool isDataCached, bool& requestPending);

      /**
       *  process a new ListDateProviderResMsg and retains the incoming data provider
       */
      void OnListDataResultMsg(const ListDateProviderResMsg& listDateProviderResMsg);
      void OnListDataUpdateMsg(const ListDataProviderUpdMsg& listDateProviderUpdMsg, ListDataProviderUpdater::ChangedItemsType& changedItems);

      /**
       *  Returns true if there is data available.
       *  @return true if the data provider is not null.
       */
      bool IsDataAvailable() const;

      Candera::Int32 GetStartIndex() const;

      /**
      *  Fetches the number of elements in the window, i.e. number of data items in the data provider.
      */
      Candera::UInt32 GetWindowElementSize() const;

      /**
      *  Fetches the list ID.
      */
      Candera::UInt32 GetListId() const;

      /**
       *  The original virtual list size.
       *  The wrapper increases the virtual list size to trick the flex list widget, nevertheless the original size may be required.
       */
      Candera::UInt32 GetUnmodifiedListSize() const;

      /**
       *  For circular scrolling lists this value will differ from the original (unmodified) list size.
       */
      Candera::UInt32 GetVirtualListSize() const;

      /* Sequence number which uniquely identifies each instantiated list data provider. */
      FeatStd::SizeType GetSequenceNumber() const;

      void UpdateSetListChangeSetIndex(Candera::Int32);

      tSharedPtrListDataItem& operator[](size_t index);

      const tSharedPtrListDataItem& operator[](size_t index) const;

   private:
      static tSharedPtrDataProvider s_emptyProvider; ///< A fallback empty provider if none is available yet.
      static tSharedPtrListDataItem s_emptyItem; ///< An empty item

      /* parameters required to correctly modify indices and the list request messages.*/
      Candera::UInt32 m_numberOfItems;
      Candera::UInt32 m_bufferSize;
      Candera::ListScrollingType m_scrollingType;

      /* internal values */
      Candera::UInt32 m_virtualListSize; ///< The modified virtual list size.
      Candera::Int32 m_startIndex; ///< The modified startIndex.
      tSharedPtrDataProvider m_dataProvider; ///< The data provider we are referring to.

      bool IsIndexValid(size_t index) const;

      void AdjustParametersNonCircular(FeatStd::Int32& startIndex, FeatStd::UInt32& windowSize) const;

      void AdjustParametersCircular(FeatStd::Int32& startIndex, FeatStd::UInt32& windowSize) const;
};


#endif // LISTDATAWRAPPER_H
