/* ***************************************************************************************
* FILE:          ListDataProviderBuilder.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ListDataProviderBuilder is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include <FeatStd/Config.h>
#include <FeatStd/Util/WarningMacros.h>
#include <Candera/Macros.h>

#include "ListDataProviderBuilder.h"


ListDataProviderBuilder::ListItem::ListItem(const ListDataProviderBuilder& builder,
      ServersDataItemHdlRow hdlRow,
      ServersDataItemHdlCol hdlCol,
      DataItemContext dataItemContext,
      ServersDataItemFlags flags,
      Candera::Int32 indexToUpdate)
   : _builder(builder), _hdlRow(hdlRow), _dataItemContext(dataItemContext), _data(), _indexToUpdate(indexToUpdate)
{
   _data.push_back(ListDataItem::newIdentifierItem(_builder.GetListId(), hdlRow, hdlCol, flags));
}


ListDataProviderBuilder::ListItem& ListDataProviderBuilder::ListItem::AddId(ServersDataItemHdlRow hdlRow,
      ServersDataItemHdlCol hdlCol,
      ServersDataItemFlags flags)
{
   _data.push_back(ListDataItem::newIdentifierItem(_builder.GetListId(), hdlRow, hdlCol, flags));
   return *this;
}


ListDataProviderBuilder::ListItem& ListDataProviderBuilder::ListItem::AddData(Candera::UInt32 intValue)
{
   _data.push_back(ListDataItem::newDataItem(intValue, 0, "", GetFirstItemFlags()));
   return *this;
}


ListDataProviderBuilder::ListItem& ListDataProviderBuilder::ListItem::AddData(const Candera::String& strValue)
{
   _data.push_back(ListDataItem::newDataItem(strValue, 0, "", GetFirstItemFlags()));
   return *this;
}


ListDataProviderBuilder::ListItem& ListDataProviderBuilder::ListItem::AddData(const std::vector<tSharedPtrListDataItem>& vecValue)
{
   _data.push_back(ListDataItem::newDataItem(vecValue, 0, "", GetFirstItemFlags()));
   return *this;
}


ListDataProviderBuilder::ListItem& ListDataProviderBuilder::ListItem::AddDataBindingUpdaterPtr(Courier::IDataBindingUpdater* dataBindingUpdater)
{
   return AddDataBindingUpdaterPtrImpl(dataBindingUpdater);
}


ListDataProviderBuilder::ListItem& ListDataProviderBuilder::ListItem::AddDataBindingUpdaterPtrImpl(Courier::IDataBindingUpdater* dataBindingUpdater)
{
   _data.push_back(ListDataItem::newDataItem(dataBindingUpdater, 0, "", GetFirstItemFlags()));
   return *this;
}


ServersDataItemFlags ListDataProviderBuilder::ListItem::GetFirstItemFlags() const
{
   ServersDataItemFlags flags = Unused;

   if (_data.size() > 0)
   {
      flags = _data[0]->getFlags();
   }

   return flags;
}


tSharedPtrListDataItem ListDataProviderBuilder::ListItem::CreateDataItem() const
{
   DataItemContext dataItemContext = _dataItemContext;
   if ((dataItemContext == NULL) || (dataItemContext[0] == '\0'))
   {
      dataItemContext = _builder.GetDefaultDataItemContext();
   }

   return ListDataItem::newDataItem(_data, _hdlRow, dataItemContext);
}


ListDataProviderBuilder::ListDataProviderBuilder(unsigned int listId, DataItemContext defaultDataItemContext)
   : _listId(listId), _defaultDataItemContext(defaultDataItemContext), _items()
{
}


ListDataProviderBuilder::~ListDataProviderBuilder()
{
   for (std::vector<ListItem*>::iterator it = _items.begin(); it != _items.end(); ++it)
   {
      FEATSTD_DELETE(*it);
   }
   _defaultDataItemContext = NULL;
}


ListDataProviderBuilder::ListItem& ListDataProviderBuilder::AddItem(ServersDataItemHdlRow hdlRow,
      ServersDataItemHdlCol hdlCol,
      DataItemContext dataItemContext,
      ServersDataItemFlags flags,
      Candera::Int32 indexToUpdate)
{
   CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, "Reference to item is returned.");
   ListItem* item = FEATSTD_NEW(ListItem)(*this, hdlRow, hdlCol, dataItemContext, flags, indexToUpdate);
   if (item == NULL)
   {
      //no memory to allocate a dynamic item, return a static preallocated one

      CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(774, "always debug assert if no memory");
      FEATSTD_DEBUG_ASSERT(!"Out Off Memory");
      static ListDataProviderBuilder dummyBuilder(0);
      static ListItem dummyItem(dummyBuilder, 0);
      return dummyItem;
   }
   _items.push_back(item);
   return *item;
}


ListDataProviderBuilder::ListItem& ListDataProviderBuilder::AddExpandableItem(ServersDataItemHdlRow hdlRow,
      unsigned int expandableListId,
      DataItemContext dataItemContext,
      ServersDataItemFlags extraFlags,
      Candera::Int32 indexToUpdate)
{
   return AddItem(hdlRow, static_cast<ServersDataItemHdlCol>(expandableListId), dataItemContext, extraFlags | ListItem::Expandable, indexToUpdate);
}


tSharedPtrListDataItemVector ListDataProviderBuilder::CreateDataItemVector()
{
   tSharedPtrListDataItemVector dataItemVector;
   size_t size = _items.size();
   for (size_t i = 0; i < size; ++i)
   {
      dataItemVector.push_back(_items.at(i)->CreateDataItem());
   }

   return dataItemVector;
}


tSharedPtrDataProvider ListDataProviderBuilder::CreateDataProvider(int startIndex, unsigned int virtualListSize)
{
   if (virtualListSize == 0)
   {
      virtualListSize = static_cast<unsigned int>(startIndex + static_cast<int>(_items.size()));
   }

   tSharedPtrDataProvider dataProvider = ListDataProvider::newBackEndInterface(_listId, startIndex, _items.size(), virtualListSize);

   for (size_t i = 0; i < _items.size() && i < virtualListSize; ++i)
   {
      (*dataProvider)[i] = _items.at(i)->CreateDataItem();
   }

   return dataProvider;
}


tSharedPtrDataProviderUpdater ListDataProviderBuilder::CreateDataProviderUpdater()
{
   tSharedPtrDataProviderUpdater updater = ListDataProviderUpdater::create(_listId);

   for (std::vector<ListItem*>::const_iterator it = _items.begin(); it != _items.end(); ++it)
   {
      const ListItem* item = *it;
      if (item != NULL)
      {
         (*updater).addItem(item->GetIndexToUpdate(), item->CreateDataItem());
      }
   }

   return updater;
}


///////////////////////////////////////////////////////////////////////

ListDateProviderReqItemEnumerator::ListDateProviderReqItemEnumerator(int32_t startIndex, uint32_t windowElementSize, uint32_t virtualListSize, bool circular)
{
   init(startIndex, windowElementSize, virtualListSize, circular);
}


ListDateProviderReqItemEnumerator::ListDateProviderReqItemEnumerator(const ListDateProviderReqMsg& oMsg, uint32_t virtualListSize)
{
   init(oMsg.GetStartIndex(), oMsg.GetWindowElementSize(), virtualListSize, oMsg.GetCircular());
}


void ListDateProviderReqItemEnumerator::init(int32_t startIndex, uint32_t windowElementSize, uint32_t virtualListSize, bool circular)
{
   _maxitems = virtualListSize;
   _startIndex = startIndex;

   if ((_startIndex < 0) && (_maxitems < windowElementSize))
   {
      // check for lists, where content is less windowsize
      _startIndex = 0;
      _lastIndex = _maxitems;
   }
   else
   {
      if (windowElementSize > 0)
      {
         _lastIndex = _startIndex + static_cast<int>(windowElementSize);
      }
      else
      {
         _lastIndex = _startIndex + _maxitems;
      }
      if (!circular)
      {
         // adjust for non circular scrolling
         if (_startIndex < 0)
         {
            _startIndex = 0;
         }
         if (_lastIndex > static_cast<int>(_maxitems))
         {
            _lastIndex = _maxitems;
         }
      }
   }
   _itemIndex = _startIndex;
}


bool ListDateProviderReqItemEnumerator::getItemIndex(uint32_t& index)
{
   if ((_maxitems > 0) && (_itemIndex < _lastIndex))
   {
      int32_t itemIndex = ((_itemIndex + _maxitems) % _maxitems);
      _itemIndex++;
      index = static_cast<uint32_t>(itemIndex);
      return true;
   }
   return false;
}
