/* ***************************************************************************************
* FILE:          ListDataItem.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ListDataItem is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"


#include "ListProviderEventInfo.h"
#include "ListDataProviderBuilder.h"
#include "ListDataItem.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ListDataItem.cpp.trc.h"
#endif

FEATSTD_RTTI_DEFINITION(ListDataItem, IDataItem);

tSharedPtrListDataItem ListDataItem::newDataItem(Candera::UInt32 intValue, ServersDataItemHdlRow hdl /*= 0*/, DataItemContext dataItemContext /*= ""*/, ServersDataItemFlags flags)
{
   return tSharedPtrListDataItem(FEATSTD_NEW(ListDataItem)(intValue, dataItemContext, hdl, flags));
}


tSharedPtrListDataItem ListDataItem::newDataItem(const Candera::String& strValue, ServersDataItemHdlRow hdl /*= 0*/, DataItemContext dataItemContext /*= ""*/, ServersDataItemFlags flags)
{
   return tSharedPtrListDataItem(FEATSTD_NEW(ListDataItem)(strValue, dataItemContext, hdl, flags));
}


tSharedPtrListDataItem ListDataItem::newDataItem(const std::vector<tSharedPtrListDataItem> vecValue, ServersDataItemHdlRow hdl /*= 0*/, DataItemContext dataItemContext /*= ""*/, ServersDataItemFlags flags)
{
   return tSharedPtrListDataItem(FEATSTD_NEW(ListDataItem)(vecValue, dataItemContext, hdl, flags));
}


tSharedPtrListDataItem ListDataItem::newDataItem(Courier::IDataBindingUpdater* value, ServersDataItemHdlRow hdl /*= 0*/, DataItemContext dataItemContext /*= ""*/, ServersDataItemFlags flags)
{
   return tSharedPtrListDataItem(FEATSTD_NEW(ListDataItem)(value, dataItemContext, hdl, flags));
}


ListDataItem::~ListDataItem()
{
   //fet2hi:
   //ListItem id format was changed from a string containing ListId+Row+Column+PointerAddress to a string containing only ListId+Row+Column
   //This is necessary for the buttons to retain their name (and focus state) when new data was received.
   //So we can not remove the mapping between the ButtonName.Hash and ListId+Row+Column when old data is destroyed.
   //ListProviderEventInfo::ClearItemIdentifier(_ident);

   switch (_type)
   {
      case StringValue:
         FEATSTD_DELETE(_data._stringValue);
         _data._stringValue = NULL;
         break;

      case VectorValue:
         FEATSTD_DELETE(_data._vecValue);
         _data._vecValue = NULL;
         break;

      case DataBindingUpdater:
         FEATSTD_DELETE(_data._dataBindingUpdater);
         _data._dataBindingUpdater = NULL;
         break;

      case IntegerValue:
         break;

      default:
         break;
   }

   _dataItemContext = 0;
}


tSharedPtrListDataItem ListDataItem::newIdentifierItem(unsigned listId, ServersDataItemHdlRow hdlRow, ServersDataItemHdlCol hdlCol, ServersDataItemFlags flags)
{
   return tSharedPtrListDataItem(FEATSTD_NEW(ListDataItem)(listId, hdlRow, hdlCol, flags, 0));
}


ListDataItem::operator Candera::Int32() const
{
   if (_type == IntegerValue)
   {
      return _data._intValue;
   }

   switch (_type)
   {
      case StringValue:
         FEATSTD_LINT_NEXT_EXPRESSION(774, "Intended debug assert");
         FEATSTD_DEBUG_ASSERT(false); // StringValue is not allowed in Flexlist
         break;

      case VectorValue:
         FEATSTD_LINT_NEXT_EXPRESSION(774, "Intended debug assert");
         FEATSTD_DEBUG_ASSERT(false); // Vector is not allowed in Flexlist
         break;

      case DataBindingUpdater:
         FEATSTD_LINT_NEXT_EXPRESSION(774, "Intended debug assert");
         FEATSTD_DEBUG_ASSERT(false); // Databinding is not allowed in Flexlist
         break;

      default:
         FEATSTD_LINT_NEXT_EXPRESSION(774, "Intended debug assert");
         FEATSTD_DEBUG_ASSERT(false); // This type is not allowed in Flexlist
         break;
   }

   return 0;
}


IDataItem& ListDataItem::operator=(Candera::Int32 value)
{
   _intModified = true;

   FEATSTD_DEBUG_ASSERT(_type == IntegerValue);
   if (_type == IntegerValue)
   {
      _data._intValue = value;
   }

   return *this;

   FEATSTD_LINT_NEXT_EXPRESSION(1539, "This assignment operator only assigns an integer value, no other values are set");
}


ListDataItem::operator Candera::String() const
{
   switch (_type)
   {
      case IntegerValue:
      {
         char buf[50];
         FeatStd::Internal::String::StringPrintf(buf, sizeof(buf), "%d", _data._intValue);
         return Candera::String(buf);
      }

      case StringValue:
      {
         FEATSTD_DEBUG_ASSERT(_data._stringValue);
         return *(_data._stringValue);
      }

      default:
         break;
   }

   return "not implemented";
}


bool ListDataItem::isVector() const
{
   return (_type == VectorValue);
}


size_t ListDataItem::size() const
{
   if (_type == VectorValue)
   {
      return (*_data._vecValue).size();
   }

   return 0;
}


tSharedPtrIDataItem ListDataItem::operator[](size_t index) const
{
   if (_type == VectorValue)
   {
      return (*_data._vecValue)[index];
   }

   return tSharedPtrListDataItem();
}


bool ListDataItem::updateDataBindings(bool force)
{
   bool updated(false);

   switch (_type)
   {
      case DataBindingUpdater:
      {
         if (_data._dataBindingUpdater != NULL)
         {
            updated = _data._dataBindingUpdater->Update(force);
         }

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListDataItem::updateDataBindings dataBindingUpdater instance = %p, updated = %d", _data._dataBindingUpdater, updated));
         break;
      }

      case VectorValue:
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListDataItem::updateDataBindings vector instance = %p", _data._vecValue));
         if (_data._vecValue != NULL)
         {
            for (std::vector<tSharedPtrListDataItem>::iterator it = _data._vecValue->begin(); it != _data._vecValue->end(); ++it)
            {
               if (!(*it).PointsToNull())
               {
                  updated = (*it)->updateDataBindings(force) || updated;
               }
               else
               {
                  ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "DataItem: DataItem contained misconfigured data (null pointer for updateDataBindings detected dataItemContext = '%s')!", _dataItemContext));
               }
            }
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListDataItem::updateDataBindings vector size = %d, updated = %d", _data._vecValue->size(), updated));
         }
         break;
      }

      case IntegerValue:
         updated = _intModified;
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListDataItem::updateDataBindings integer value = %d, updated = %d", _data._intValue, updated));
         break;

      case StringValue:
      {
         Candera::String str = *(_data._stringValue);
         updated = _stringModified;
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListDataItem::updateDataBindings string value = %p (\"%15s\"), updated = %d", _data._stringValue, str.GetCString(), updated));
         break;
      }

      default:
         break;
   }

   _intModified = false;
   _stringModified = false;

   return updated;
}


void ListDataItem::dumpContent(int index, int layer) const
{
   if (layer == 0)
   {
      ETG_TRACE_USR1_CLS_DCL((TR_CLASS_HMI_LISTPROVIDER, APP_TRACECLASS_ID(), "--- Index[%d] Handle[%u]  Template: %20.20s---", index, _hdlRow, dataItemContext()));
   }
   switch (_type)
   {
      case StringValue:
      {
         Candera::String str = *(_data._stringValue);
         ETG_TRACE_USR1_CLS_DCL((TR_CLASS_HMI_LISTPROVIDER, APP_TRACECLASS_ID(), " TYPE: STRING value='%s'", str.GetCString()));
      }
      break;

      case IntegerValue:
      {
         char buf[50];
         FeatStd::Internal::String::StringPrintf(buf, sizeof(buf), "%d", _data._intValue);
         ETG_TRACE_USR1_CLS_DCL((TR_CLASS_HMI_LISTPROVIDER, APP_TRACECLASS_ID(), " TYPE: INT value='%s'", buf));
      }
      break;

      case DataBindingUpdater:
      {
         ETG_TRACE_USR1_CLS_DCL((TR_CLASS_HMI_LISTPROVIDER, APP_TRACECLASS_ID(), " TYPE: DataBindingUpdater instance = %p", _data._dataBindingUpdater));
      }
      break;

      case VectorValue:
         ETG_TRACE_USR1_CLS_DCL((TR_CLASS_HMI_LISTPROVIDER, APP_TRACECLASS_ID(), " TYPE: VECTOR, enumerate items in layer %d, handle=%u ...", layer + 1, _hdlRow));
         if (_data._vecValue != NULL)
         {
            ETG_TRACE_USR1_CLS_DCL((TR_CLASS_HMI_LISTPROVIDER, APP_TRACECLASS_ID(), " --->"));
            for (std::vector<tSharedPtrListDataItem>::iterator it = _data._vecValue->begin(); it != _data._vecValue->end(); ++it)
            {
               if (!(*it).PointsToNull())
               {
                  (*it)->dumpContent(index, layer + 1);
               }
            }
            ETG_TRACE_USR1_CLS_DCL((TR_CLASS_HMI_LISTPROVIDER, APP_TRACECLASS_ID(), " <---"));
         }
         break;

      default:
         break;
   }
}


Courier::IDataBindingUpdater* ListDataItem::dataBindingUpdater() const
{
   return (_type == DataBindingUpdater) ? _data._dataBindingUpdater : NULL;
}


DataItemContext ListDataItem::dataItemContext() const
{
   return _dataItemContext;
}


ListDataItem::Type ListDataItem::type() const
{
   return _type;
}


tSharedPtrListDataItem ListDataItem::cloneAs(DataItemContext dataItemContext) const
{
   return tSharedPtrListDataItem(FEATSTD_NEW(ListDataItem)(dataItemContext, *this));
}


ServersDataItemFlags ListDataItem::getFlags() const
{
   return _flags;
}


void ListDataItem::setFlags(ServersDataItemFlags fl)
{
   _flags |= fl;
}


void ListDataItem::setHdlRow(ServersDataItemHdlRow index)
{
   _hdlRow = index;
}


void ListDataItem::setHdlCol(ServersDataItemHdlCol index)
{
   _hdlCol = index;
}


ServersDataItemHdlRow ListDataItem::getHdlRow() const
{
   return _hdlRow;
}


ServersDataItemHdlCol ListDataItem::getHdlCol() const
{
   return _hdlCol;
}


Courier::Identifier ListDataItem::getIdent() const
{
   return _ident;
}


ListDataItem::ListDataItem(Courier::IDataBindingUpdater* value, DataItemContext dataItemContext, ServersDataItemHdlRow hdlRow, ServersDataItemFlags flags) :
   _type(DataBindingUpdater),
   _isCollapsed(false),
   _intModified(false),
   _stringModified(false)
{
   _data._dataBindingUpdater = value;
   _dataItemContext = dataItemContext;
   _hdlRow = hdlRow;
   _hdlCol = 0;
   _flags = flags;
}


ListDataItem::ListDataItem(Candera::UInt32 intValue, DataItemContext dataItemContext, ServersDataItemHdlRow hdlRow, ServersDataItemFlags flags) :
   _type(IntegerValue),
   _isCollapsed(false),
   _intModified(true),
   _stringModified(false)
{
   _data._intValue = intValue;
   _dataItemContext = dataItemContext;
   _hdlRow = hdlRow;
   _hdlCol = 0;
   _flags = flags;
}


ListDataItem::ListDataItem(const Candera::String& strValue, DataItemContext dataItemContext, ServersDataItemHdlRow hdlRow, ServersDataItemFlags flags) :
   _type(StringValue),
   _isCollapsed(false),
   _intModified(false),
   _stringModified(true)
{
   _data._stringValue = FEATSTD_NEW(Candera::String)(strValue);
   _dataItemContext = dataItemContext;
   _hdlRow = hdlRow;
   _hdlCol = 0;
   _flags = flags;
}


ListDataItem::ListDataItem(const std::vector<tSharedPtrListDataItem>& vecValue, DataItemContext dataItemContext, ServersDataItemHdlRow hdlRow, ServersDataItemFlags flags) :
   _type(VectorValue),
   _isCollapsed(false),
   _intModified(false),
   _stringModified(false)
{
   _data._vecValue = FEATSTD_NEW(std::vector<tSharedPtrListDataItem>)(vecValue);
   _dataItemContext = dataItemContext;
   _hdlRow = hdlRow;
   _hdlCol = 0;
   _flags = flags;
}


ListDataItem::ListDataItem(unsigned listId, ServersDataItemHdlRow hdlRow, ServersDataItemHdlCol hdlCol, ServersDataItemFlags flags, int) :
   _type(StringValue),
   _isCollapsed(false),
   _intModified(false),
   _stringModified(true)
{
#ifdef WIN32
#ifndef snprintf
#define snprintf _snprintf
#endif
#endif
   char buf[40];
   //fet2hi: We need a more consistent name of the buttons which is maintained even if new data is received, so the pointer address is not part anymore from the generated id.
   //snprintf(buf, sizeof(buf), "PDL_L%ld_P1%ld_P2%ld_ADR%04x", listId, hdlRow, (int)hdlCol, this);
#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
   snprintf(buf, sizeof(buf), "ListItem_%u_%lu_%lu", listId, hdlRow, hdlCol);
#else
   snprintf(buf, sizeof(buf), "ListItem_%u_%u_%u", listId, hdlRow, hdlCol);
#endif
   _ident = Courier::Identifier(buf);
   ListProviderEventInfo::SetItemIdentifier(_ident, ListProviderEventInfo(listId, hdlRow, hdlCol, flags));
   _data._stringValue = FEATSTD_NEW(Candera::String)(buf);
   _dataItemContext = "";
   _hdlRow = hdlRow;
   _hdlCol = hdlCol;
   _flags = flags;
}


ListDataItem::ListDataItem(DataItemContext dataItemContext, const ListDataItem& item) :
   _data(item._data),
   _dataItemContext(dataItemContext),
   _type(item._type),
   _flags(item._flags),
   _hdlRow(item._hdlRow),
   _hdlCol(item._hdlCol),
   _ident(item._ident),
   _isCollapsed(false),
   _intModified(false),
   _stringModified(true)
{
   switch (_type)
   {
      case StringValue:
         _data._stringValue = FEATSTD_NEW(Candera::String)(*_data._stringValue);
         break;
      case VectorValue:
         _data._vecValue = FEATSTD_NEW(std::vector<tSharedPtrListDataItem>)(*_data._vecValue);
         break;
      case DataBindingUpdater:
         //cloning of the data binding updater not supported
         _data._dataBindingUpdater = NULL;
         break;
      case IntegerValue:
         break;
   }
}


static Candera::Int32 ListDataItemCompareVectors(const std::vector<tSharedPtrListDataItem>& first, const std::vector<tSharedPtrListDataItem>& second)
{
   std::vector<tSharedPtrListDataItem>::const_iterator firstIt = first.begin();
   std::vector<tSharedPtrListDataItem>::const_iterator secondIt = second.begin();
   while ((firstIt != first.end()) && (secondIt != second.end()))
   {
      if (0 == firstIt->GetPointerToSharedInstance())
      {
         return -1;
      }
      if (0 == secondIt->GetPointerToSharedInstance())
      {
         return 1;
      }
      if (firstIt->GetPointerToSharedInstance() != secondIt->GetPointerToSharedInstance())
      {
         Candera::Int32 result = (*firstIt)->compare(**secondIt);
         if (0 != result)
         {
            return result;
         }
      }
      ++firstIt;
      ++secondIt;
   }
   if (firstIt == first.end())
   {
      if (secondIt == second.end())
      {
         return 0;
      }
      return -1;
   }
   return 0;
}


Candera::Int32 ListDataItem::compare(const IDataItem& item)
{
   if (this == &item)
   {
      return 0;
   }
   const ListDataItem* listDataItem = Candera::Dynamic_Cast<const ListDataItem*>(&item);
   if (0 != listDataItem)
   {
      if (listDataItem->_type != _type)
      {
         return _type < listDataItem->_type ? -1 : 1;
      }
      if (listDataItem->_dataItemContext != _dataItemContext)
      {
         return _dataItemContext < listDataItem->_dataItemContext ? -1 : 1;
      }
      if (listDataItem->_flags != _flags)
      {
         return _flags < listDataItem->_flags ? -1 : 1;
      }
      if (listDataItem->_hdlRow != _hdlRow)
      {
         return _hdlRow < listDataItem->_hdlRow ? -1 : 1;
      }
      if (listDataItem->_hdlCol != _hdlCol)
      {
         return _hdlCol < listDataItem->_hdlCol ? -1 : 1;
      }
      if (listDataItem->_ident != _ident)
      {
         return _ident < listDataItem->_ident ? -1 : 1;
      }
      switch (_type)
      {
         case ListDataItem::IntegerValue:
            if ((HasFlags(ListDataProviderBuilder::ListItem::FullComparison)) && (_data._intValue != listDataItem->_data._intValue))
            {
               return (_data._intValue < listDataItem->_data._intValue) ? -1 : 1;
            }
            return 0;
         case ListDataItem::StringValue:
            if (HasFlags(ListDataProviderBuilder::ListItem::FullComparison))
            {
               if (listDataItem->_data._stringValue == _data._stringValue)
               {
                  return 0;
               }
               if (0 == _data._stringValue)
               {
                  return -1;
               }
               if (0 == listDataItem->_data._stringValue)
               {
                  return 1;
               }
               {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                  FeatStd::Internal::CriticalSectionLocker lock((*_data._stringValue).GetCriticalSection());
                  FeatStd::Internal::CriticalSectionLocker otherLock((*listDataItem->_data._stringValue).GetCriticalSection());
#endif
                  return Candera::StringPlatform::CompareStrings((*_data._stringValue).GetCString(), (*listDataItem->_data._stringValue).GetCString());
               }
            }
            return 0;
         case ListDataItem::VectorValue:
            if (listDataItem->_data._vecValue == _data._vecValue)
            {
               return 0;
            }
            if (0 == _data._vecValue)
            {
               return -1;
            }
            if (0 == listDataItem->_data._vecValue)
            {
               return 1;
            }
            return ListDataItemCompareVectors(*listDataItem->_data._vecValue, *_data._vecValue);
         case ListDataItem::DataBindingUpdater:
            if (HasFlags(ListDataProviderBuilder::ListItem::FullComparison))
            {
               if (listDataItem->_data._dataBindingUpdater == _data._dataBindingUpdater)
               {
                  return 0;
               }
               if (0 == _data._dataBindingUpdater)
               {
                  return -1;
               }
               if (0 == listDataItem->_data._dataBindingUpdater)
               {
                  return 1;
               }
               return listDataItem->_data._dataBindingUpdater->Compare(*_data._dataBindingUpdater);
            }
            return 0;
      }
   }
   return IDataItem::compare(item);
}


bool ListDataItem::HasFlags(ServersDataItemFlags flags)
{
   return (_flags & flags) == flags;
}
