/* ***************************************************************************************
* FILE:          DataBindingUpdater.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DataBindingUpdater is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#if !defined(DataBindingUpdater_H)
#define DataBindingUpdater_H

#include "Courier/Version.h"
#include "Courier/Platform/MessageFactory.h"

#include "IDataBindingUpdater.h"

#include "Courier/DataBinding/ModelBindingSourceMap.h"

namespace Courier {

template<typename T> class DataBindingUpdater : public IDataBindingUpdater
{
   public:
      typedef typename T::Data ValueType;

      DataBindingUpdater() : mValueContainer()
      {
         mValueContainer.MarkAllItemsModified();
      }

      DataBindingUpdater(const DataItemContainer<T>& valueContainer) : mValueContainer(valueContainer)
      {
         mValueContainer.MarkAllItemsModified();
      }

      virtual bool Update(bool force)
      {
         bool updated = false;

         if (force)
         {
            mValueContainer.MarkAllItemsModified();
         }

         if (mValueContainer.HasModifiedItems())
         {
            Courier::DataItemMsg<T>* msg = COURIER_MESSAGE_NEW(::Courier::DataItemMsg<T>)(mValueContainer);
            //takes care to delete the message
            ::Courier::MessageReferrer msgRef(msg);
            if (msg != 0)
            {
               ::Courier::Internal::ModelBindingSource* bs = ::Courier::Internal::ModelBindingSourceMap::Locate(
                        ::Courier::ComponentType::View,
                        ::Courier::Internal::DataBinding::DataItemAccessor::Root(msg->GetItemKey())
#if((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5))
                        , Candera::AbstractNodePointer()
#endif
                     );

               if (bs != 0)
               {
                  updated = true;
                  bs->OnDataItemMsg(msg);
                  bs->ReleaseDataItemMsgReference();
               }

               mValueContainer.ResetAllItemsModified();
            }
         }

         return updated;
      }

      /**
       * The values modified by this DataItemCOntainer must be marked as modified using the
       * DataItemContainer::MarkItemModified method.
       */
      DataItemContainer<T>& GetValueContainer()
      {
         return mValueContainer;
      }

      virtual Candera::Int32 Compare(const IDataBindingUpdater& dataBindingUpdater)
      {
#ifdef DATABINDING_UPDATER_ALWAYS_EQUAL
         return 0;
#else
         Candera::Int32 result = IDataBindingUpdater::Compare(dataBindingUpdater);
         if (0 != result)
         {
            const DataBindingUpdater<T>* tDataBindingUpdater = dynamic_cast< const DataBindingUpdater< T > * >(&dataBindingUpdater);
            if (0 != tDataBindingUpdater)
            {
               return FeatStd::Internal::Memory::Compare(&mValueContainer, &(tDataBindingUpdater->mValueContainer), sizeof(T));
            }
         }
         return result;
#endif
      }

   private:
      DataItemContainer<T> mValueContainer;
};


}

#endif
