/* ***************************************************************************************
* FILE:          MaxIndexManager.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  MaxIndexManager is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include <Candera/System/Mathematics/Math.h>
#include <stdio.h>

#include "MaxIndexManager.h"


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/MaxIndexManager.cpp.trc.h"
#endif

using namespace FeatStd;
using namespace Candera;

MaxIndexManager::MaxIndexManager() :
   _maxPosition(0.0F),
   _accurateMaxPosition(false),
   _freshReset(true),
   _fixedPage(false),
   _circular(false),
   _indexOfLastFixedPage(-1)
{
}


void MaxIndexManager::UpdateMaxIndex(const MaxIndexData& maxPosData)
{
   if (!_accurateMaxPosition)
   {
      if (_circular)
      {
         _maxPosition = static_cast<Float>(maxPosData._totalNumberOfElements);
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "MaxIndexManager::UpdateMaxIndex circular maxPosition=%f", _maxPosition));
      }
      else if (_fixedPage)
      {
         if (_indexOfLastFixedPage >= 0)
         {
            _maxPosition = static_cast<Float>(_indexOfLastFixedPage);
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "MaxIndexManager::UpdateMaxIndex fixedPage maxPosition=%f", _maxPosition));
            _accurateMaxPosition = !maxPosData._invalidItemsAdded;
         }
      }
      else if ((maxPosData._pixelwiseEnabled || (0 == maxPosData._configuredNumberOfItems)) && (!maxPosData._coverflow))
      {
         if (maxPosData._endReached)
         {
            Float maxPosition = maxPosData._startIndex;

            if (maxPosData._lastItemCompletelyVisible)
            {
               const Float correction = (maxPosData._firstItemCompletelyVisible) ? 0.0F : Float(maxPosData._numberOfItemsInFirstStructure);

               _maxPosition = maxPosition + correction;
               ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "MaxIndexManager::UpdateMaxIndex lastItemCompletelyVisible maxPosition=%f", _maxPosition));
               _accurateMaxPosition = !maxPosData._invalidItemsAdded;
            }
            else
            {
               Float correction = Float(maxPosData._numberOfItemsInFirstStructure);

               if (maxPosData._firstItemVisibleSize < maxPosData._lastItemInvisibleSize)
               {
                  // needed if the visible part of the first item is smaller than the last invisible part
                  correction += Float(maxPosData._numberOfItemsInFirstStructure);
               }

               _maxPosition = maxPosition + correction;
               ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "MaxIndexManager::UpdateMaxIndex other1 maxPosition=%f", _maxPosition));
            }
         }
         else
         {
            const UInt32 previousVisibleItems = _freshReset ? maxPosData._currentVisibleItems : maxPosData._previousVisibleItems;
            const Float averageNumberOfVisibleItems = Math::Floor(Float(maxPosData._currentVisibleItems + previousVisibleItems + 1) * 0.5F);

            Float maxPos = Math::Floor((Float(maxPosData._totalNumberOfElements) - averageNumberOfVisibleItems));

            if (maxPosData._numberOfItemsInFirstStructure != 0)
            {
               Int32 maxPosModulo = Int32(maxPos) % maxPosData._numberOfItemsInFirstStructure;
               if (maxPosModulo != 0)
               {
                  maxPos += (maxPosData._numberOfItemsInFirstStructure - maxPosModulo);
               }
            }

            _maxPosition = maxPos;
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "MaxIndexManager::UpdateMaxIndex other2 maxPosition=%f", _maxPosition));
         }
      }
      else
      {
         const Float totalNumberOfElements(static_cast<Float>(maxPosData._totalNumberOfElements));
         const Int32 visibleElements(maxPosData._configuredNumberOfItems * maxPosData._numberOfItemsInFirstStructure);
         const Float visibleElementsFloat(static_cast<Float>(visibleElements));

         Float maxPos = Math::Floor(totalNumberOfElements - visibleElementsFloat);

         if (maxPosData._numberOfItemsInFirstStructure != 0)
         {
            Int32 maxPosModulo = static_cast<Int32>(maxPos) % maxPosData._numberOfItemsInFirstStructure;
            if (maxPosModulo != 0)
            {
               maxPos += (maxPosData._numberOfItemsInFirstStructure - maxPosModulo);
            }
         }

         const Float correction1 = ((maxPosData._startIndex <= maxPos) || maxPosData._coverflow) ? 0.0F : Float(maxPosData._numberOfItemsInFirstStructure);

         const Float correction2 = (maxPosData._lastItemCompletelyVisible) ? 0.0F : Float(maxPosData._numberOfItemsInFirstStructure);
         _accurateMaxPosition = (!maxPosData._invalidItemsAdded) && maxPosData._endReached && maxPosData._lastItemCompletelyVisible;

         _maxPosition = maxPos + correction1 + correction2;
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "MaxIndexManager::UpdateMaxIndex other3 maxPosition=%f", _maxPosition));
      }
   }
}


FeatStd::Float MaxIndexManager::GetMaxIndex() const
{
   return _maxPosition;
}


bool MaxIndexManager::GetAccurateMaxPosition() const
{
   return _accurateMaxPosition;
}


void MaxIndexManager::SetFixedPages(bool val)
{
   _fixedPage = val;
}


void MaxIndexManager::SetIndexOfLastFixedPage(FeatStd::Int32 val)
{
   _indexOfLastFixedPage = val;
}


void MaxIndexManager::SetCircular(bool val)
{
   _circular = val;
}


void MaxIndexManager::ResetMaxPosition(Int32 size, bool force/* = false*/)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "MaxIndexManager::ResetMaxPosition maxPosition=%f, size=%d, force=%d ", _maxPosition, size, force));

   if (force || Math::FloatAlmostEqual(_maxPosition, 0.0F))
   {
      _maxPosition = (_circular) ? Float(size) : Float(size - 1);
      if (_maxPosition < 0.0F)
      {
         _maxPosition = 0.0F;
      }

      if (!_fixedPage)
      {
         _indexOfLastFixedPage = -1;
      }

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "MaxIndexManager::ResetMaxPosition maxPosition=%f", _maxPosition));
      _accurateMaxPosition = false;
      _freshReset = true;
   }
}


MaxIndexManager::MaxIndexData::MaxIndexData(
   const bool invalidItemsAdded,
   const bool coverflow,
   const bool pixelwiseEnabled,
   const bool endReached,
   const bool lastItemCompletelyVisible,
   const bool firstItemCompletelyVisible,
   const Int32 numberOfItemsInFirstStructure,
   const FeatStd::Float startIndex,
   const FeatStd::UInt32 previousVisibleItems,
   const FeatStd::UInt32 currentVisibleItems,
   const FeatStd::Int32 totalNumberOfElements,
   const FeatStd::Float firstItemVisibleSize,
   const FeatStd::Float lastItemInvisibleSize,
   const FeatStd::Int32 configuredNumberOfItems) :
   _invalidItemsAdded(invalidItemsAdded),
   _coverflow(coverflow),
   _pixelwiseEnabled(pixelwiseEnabled),
   _endReached(endReached),
   _lastItemCompletelyVisible(lastItemCompletelyVisible),
   _firstItemCompletelyVisible(firstItemCompletelyVisible),
   _numberOfItemsInFirstStructure(numberOfItemsInFirstStructure),
   _startIndex(startIndex),
   _previousVisibleItems(previousVisibleItems),
   _currentVisibleItems(currentVisibleItems),
   _totalNumberOfElements(totalNumberOfElements),
   _firstItemVisibleSize(firstItemVisibleSize),
   _lastItemInvisibleSize(lastItemInvisibleSize),
   _configuredNumberOfItems(configuredNumberOfItems)
{
}
