/* ***************************************************************************************
* FILE:          ListContentUpdater.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ListContentUpdater is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef List_Content_Updater_H
#define List_Content_Updater_H

#include "Widgets/2D/List/generated/ListWidget2DBase.h"
#include "Widgets/2D/ControlTemplate/ControlTemplateInstanceContainer.h"

#include "Widgets/2D/List/Content/ContentProvider.h"
#include "Widgets/2D/List/Data/ListDataWrapper.h"
#include "Widgets/2D/List/Content/TimeDispatcher.h"
#include "Widgets/2D/Common/EventHub.h"
#include "Widgets/2D/List/Content/ITemplateRetriever.h"

class ListContentUpdater
{
   public:
      class ContentUpdateEvent : public EventHub::Event
      {
         public:
            virtual ~ContentUpdateEvent()
            {
            }

            const FeatStd::UInt32& GetListId() const
            {
               return _listId;
            }
            FeatStd::UInt32 GetStartIndex() const
            {
               return _startIndex;
            }
            FeatStd::UInt32 GetOldStartIndex() const
            {
               return _oldStartIndex;
            }

         private:
            FEATSTD_MAKE_CLASS_UNCOPYABLE(ContentUpdateEvent);
            friend class ListContentUpdater;

            ContentUpdateEvent(const FeatStd::UInt32& listId, FeatStd::UInt32 startIndex, FeatStd::UInt32 oldStartIndex) :
               _listId(listId), _startIndex(startIndex), _oldStartIndex(oldStartIndex)
            { }

            const FeatStd::UInt32& _listId;
            FeatStd::UInt32 _startIndex;
            FeatStd::UInt32 _oldStartIndex;
      };

      enum ScrollingSpeed
      {
         Slow,
         Fast,
         Immediate
      };

      ListContentUpdater(ControlTemplateInstanceContainer& itemsInstanceContainer,
                         ITemplateRetriever& templateRetriever,
                         const ContentProvider& contentProvider);
      virtual ~ListContentUpdater();

      typedef void (*ContentUpdateCallback)(ListWidget2DBase::ListIdType listId, bool success);
      static void SetContentUpdateCallback(ContentUpdateCallback callback);

      void SetScrollingOrientation(::Candera::ListScrollingOrientation val);
      Candera::ListScrollingOrientation GetScrollingOrientation() const
      {
         return _scrollingOrientation;
      }

      void SetScrollAnimation(ListWidget2DBase::ScrollAnimationInputType animationInput);

      void Init(Candera::Node2D* itemsGroup,
                const ListWidget2DBase::NumberOfItemsType& configuredNumberOfItems);

      void RetainConfiguredNumberOfItems(const ListWidget2DBase::NumberOfItemsType& configuredNumberOfItems);

      Candera::Node2D* GetFocusedItem() const;

      /**
       * update list content
       * @return true if the items are invalid after update; false otherwise
       */
      bool Update(ListWidget2DBase::StartIndexType& startIndex, ListWidget2DBase::FocusedIndexType& focusedIndex,
                  ListWidget2DBase::NumberOfItemsType& numberOfItems,
                  const Candera::Vector2& visibleAreaInPixels,
                  ScrollingSpeed scrollingSpeed);

      /**
      * @return the max position in order to fit all the elements
      */
      virtual FeatStd::UInt32 GetMaxPosition() const = 0;

      /**
      * @ return the current position of the list.
      */
      virtual FeatStd::Float GetCurrentPosition() const = 0;

      virtual void SetTargetStartIndex(ListWidget2DBase::StartIndexType startIndex) = 0;

      bool IsContentAdded() const;

      virtual void RefreshOnDataChanged() = 0;

      virtual void Finalize();

      void UpdateState() const;

      FeatStd::SizeType GetDataSequenceNumber() const
      {
         return _dataSequenceNumber;
      }

      void UpdateLocked() const;

   protected:
      virtual void DisposeContent();

      const ITemplateRetriever& _templateRetriever;
      ListWidget2DBase::TemplateScrollAnimationType _templateScrollAnimation;
      ListWidget2DBase::ScrollAnimationInputType _scrollAnimationInput;
      Candera::Node2D* _focusedItem;
      ControlTemplateInstanceContainer& _itemsInstanceContainer;
      FeatStd::SizeType _dataSequenceNumber;

      const ContentProvider& GetContentProvider() const;
      Candera::Node2D* GetItemsGroup() const;
      Candera::UInt32 GetConfiguredNumberOfItems() const;

      void NotifyContentUpdated(FeatStd::UInt32 startIndex, FeatStd::UInt32 oldStartIndex) const;
      static ContentUpdateCallback& GetContentUpdateCallback();

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(ListContentUpdater);

      virtual bool UpdateListContent(ListWidget2DBase::StartIndexType& startIndex, ListWidget2DBase::FocusedIndexType& focusedIndex,
                                     ListWidget2DBase::NumberOfItemsType& numberOfItems,
                                     const Candera::Vector2& visibleAreaInPixels,
                                     ScrollingSpeed scrollingSpeed) = 0;

      virtual void OnBeforeTemplateGroupChanged() { }
      virtual void OnAfterTemplateGroupChanged() { }
      virtual void OnBeforeInvalidItemTemplateChanged() { }
      virtual void OnAfterInvalidItemTemplateChanged() { }

      bool _contentAdded;
      Candera::Node2D* _itemsGroup;
      Candera::UInt32 _configuredNumberOfItems;
      ::Candera::ListScrollingOrientation _scrollingOrientation;
      const ContentProvider& _contentProvider;
      Candera::Margin _margin;
};


#endif
