/* ***************************************************************************************
* FILE:          ListContentUpdater.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ListContentUpdater is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "ListContentUpdater.h"
#include "Widgets/2D/Focus/FocusGroupWidget2D.h"
#include "Widgets/2D/WidgetFinder2D.h"

using namespace Candera;
using namespace Courier;

ListContentUpdater::ContentUpdateCallback& ListContentUpdater::GetContentUpdateCallback()
{
   static ListContentUpdater::ContentUpdateCallback callback = NULL;
   return callback;
}


void ListContentUpdater::SetContentUpdateCallback(ContentUpdateCallback callback)
{
   GetContentUpdateCallback() = callback;
}


ListContentUpdater::ListContentUpdater(ControlTemplateInstanceContainer& itemsInstanceContainer,
                                       ITemplateRetriever& templateRetriever,
                                       const ContentProvider& contentProvider) :
   _itemsGroup(0),
   _focusedItem(0),
   _dataSequenceNumber(0),
   _configuredNumberOfItems(0),
   _scrollingOrientation(Vertical),
   _contentAdded(false),
   _templateRetriever(templateRetriever),
   _scrollAnimationInput(Candera::ScrollAnimationInput::PixelPositionBased),
   _itemsInstanceContainer(itemsInstanceContainer),
   _contentProvider(contentProvider)
{
}


ListContentUpdater::~ListContentUpdater()
{
   _focusedItem = 0;
   _itemsGroup = 0;
}


Candera::Node2D* ListContentUpdater::GetItemsGroup() const
{
   return _itemsGroup;
}


Candera::UInt32 ListContentUpdater::GetConfiguredNumberOfItems() const
{
   return _configuredNumberOfItems;
}


void ListContentUpdater::SetScrollingOrientation(::Candera::ListScrollingOrientation val)
{
   _scrollingOrientation = val;
}


void ListContentUpdater::SetScrollAnimation(Candera::ScrollAnimationInput::Enum animationInput)
{
   _scrollAnimationInput = animationInput;
}


void ListContentUpdater::Init(Candera::Node2D* itemsGroup,
                              const ListWidget2DBase::NumberOfItemsType& configuredNumberOfItems)
{
   _itemsGroup = itemsGroup;

   RetainConfiguredNumberOfItems(configuredNumberOfItems);
}


void ListContentUpdater::RetainConfiguredNumberOfItems(const ListWidget2DBase::NumberOfItemsType& configuredNumberOfItems)
{
   _configuredNumberOfItems = configuredNumberOfItems;
}


void ListContentUpdater::UpdateLocked() const
{
   if (_contentProvider.HasItems() && (GetContentUpdateCallback() != NULL))
   {
      GetContentUpdateCallback()(_contentProvider.GetListId(), false);
   }
}


void ListContentUpdater::DisposeContent()
{
   _focusedItem = 0;
   _itemsInstanceContainer.DisposeContent();
}


Candera::Node2D* ListContentUpdater::GetFocusedItem() const
{
   return _focusedItem;
}


bool ListContentUpdater::Update(ListWidget2DBase::StartIndexType& startIndex, ListWidget2DBase::FocusedIndexType& focusedIndex, ListWidget2DBase::NumberOfItemsType& numberOfItems, const Candera::Vector2& visibleAreaInPixels, ScrollingSpeed scrollingSpeed)
{
   bool updated(false);
   if (visibleAreaInPixels.GetX() > 0 && visibleAreaInPixels.GetY() > 0)
   {
      _contentAdded = _contentProvider.HasItems();
      _dataSequenceNumber = _contentProvider.GetSequenceNumber();

      updated = UpdateListContent(startIndex, focusedIndex, numberOfItems, visibleAreaInPixels, scrollingSpeed);
   }

   Node2D* itemsGroup(GetItemsGroup());
   if (0 != itemsGroup)
   {
      _margin = Layouter::GetMargin(*itemsGroup);
   }

   return updated;
}


bool ListContentUpdater::IsContentAdded() const
{
   return _contentAdded;
}


const ContentProvider& ListContentUpdater::GetContentProvider() const
{
   return _contentProvider;
}


void ListContentUpdater::NotifyContentUpdated(FeatStd::UInt32 startIndex, FeatStd::UInt32 oldStartIndex) const
{
   //synchronous notification after widgets have been added/removed for a potential client like focus manager
   if (_contentProvider.HasItems())
   {
      EventHub::Notify(ContentUpdateEvent(_contentProvider.GetListId(), startIndex, oldStartIndex));
   }
}


void ListContentUpdater::Finalize()
{
   DisposeContent();
}


void ListContentUpdater::UpdateState() const
{
   Node2D* itemsGroup(GetItemsGroup());
   if (0 != itemsGroup)
   {
      Layouter::SetMargin(*itemsGroup, _margin);
   }
}
