/* ***************************************************************************************
* FILE:          ListContentAdder.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ListContentAdder is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include <Widgets/2D/List/Content/AddChecker.h>
#include <Widgets/2D/List/Content/PageInfoCollector.h>
#include "Widgets/2D/ControlTemplate/ControlTemplate.h"
#include "Widgets/2D/List/Content/Adders/OrthoGridAddingStrategy.h"
#include "Widgets/2D/List/Content/Adders/GridAddingStrategy.h"
#include "Widgets/2D/List/Content/Adders/StackAddingStrategy.h"
#include <Candera/Engine2D/Layout/GridLayouter.h>
#include <Widgets/2D/ControlTemplate/ItemLayouter.h>
#include "Widgets/2D/List/Content/SwipingListContentUpdaterListener.h"

#include "ListContentAdder.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ListContentAdder.cpp.trc.h"
#endif

using namespace Candera;

ListContentAdder::ListContentAdder(OrientationOperator& directOrientationOperator,
                                   OrientationOperator& orthogonalOrientationOperator,
                                   ControlTemplateInstanceContainer& itemsInstanceContainer,
                                   const ITemplateRetriever& templateRetriever,
                                   const ContentProvider& contentProvider,
                                   const ItemSpeedAnimationManager& itemSpeedAnimationManager,
                                   AddingDirection addingDirection) :

   _orientationOperator(directOrientationOperator),
   _orthogonalOrientationOperator(orthogonalOrientationOperator),
   _itemsInstanceContainer(itemsInstanceContainer),
   _templateRetriever(templateRetriever),
   _contentProvider(contentProvider),
   _addingDirection(addingDirection),
   _listItemAdder(0),
   _addingStrategy(0),
   _listener(0),
   _expandTriggerManager(0),
   _numberOfCompleteVisibleItems(0),
   _oldVirtualListSize(0),
   _elementsShownOnlyOnce(false),
   _maxIndexManager(0),
   _listContentInfo(),
   _correctCurrentPosition(false),
   _invalidItemsAdded(false),
   _reversePositionCorrection(0),
   _pixelwise(false),
   _endReached(false),
   _itemSpeedAnimationManager(itemSpeedAnimationManager)
{
}


ListContentAdder::~ListContentAdder()
{
   _listener = 0;
   _maxIndexManager = 0;
   _expandTriggerManager = 0;
}


void ListContentAdder::Set(const ItemAdder::SharedPointer& listItemAdder,
                           const AddingStrategy::SharedPointer& addingStrategy,
                           MaxIndexManager* maxIndexManager,
                           SwipingListContentUpdaterListener* listener,
                           ExpandTriggerManager* expandTriggerManager)
{
   _listItemAdder = listItemAdder;
   _addingStrategy = addingStrategy;
   _maxIndexManager = maxIndexManager;
   _listener = listener;
   _expandTriggerManager = expandTriggerManager;
}


Int32 ListContentAdder::AddContent(FeatStd::Float position, Candera::Vector2 listArea, const ListContentInfo& listInfo, bool pixelwise)
{
   ResetMembers();

   _listContentInfo = listInfo;
   _listArea = listArea;
   _pixelwise = pixelwise;

   const Int32 size(_contentProvider.GetVirtualListSize());
   FeatStd::UInt32 newStartIndex(0);

   if ((!_listItemAdder.PointsToNull()) && (!_addingStrategy.PointsToNull()) && _contentProvider.HasItems() && (_templateRetriever.IsValid()) && (size > 0))
   {
      const UInt32 previousVisibleItems = _numberOfCompleteVisibleItems;

      // Calculate firstItemIndex
      const Candera::Float correctedPosition(_listItemAdder->BeginAdding(listArea, position));
      const Candera::Float currentPosition(correctedPosition < _listContentInfo.positionLimit ? (correctedPosition < 0.0F ? 0.0F : correctedPosition) : _listContentInfo.positionLimit);
      const Int32 firstItemIndex(_addingStrategy->CalculateStartIndex(_templateRetriever, _contentProvider, currentPosition) % size);
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListContentAdder::AddListContent starting firstItemIndex=%d correctedPosition=%f", firstItemIndex, correctedPosition));

      //numberOfTemplatesFitting
      Candera::Node2D* invalidTemplate(_templateRetriever.GetTemplate(tSharedPtrListDataItem(0)));
      const Float visibleSize(_orientationOperator.GetVectorComponent(listArea));
      const Float originalTemplateSize(_listItemAdder->GetItemSize(*invalidTemplate));
      const Float templateSize(Math::FloatAlmostEqual(originalTemplateSize, 0.0F) ? 10.0F : originalTemplateSize);
      const Int32 numberOfTemplatesFitting((Int32(visibleSize / templateSize) + Int32(1)) * Int32(_listItemAdder->GetScrollingIncrement()));

      if (_maxIndexManager != 0)
      {
         _maxIndexManager->ResetMaxPosition(size - numberOfTemplatesFitting);
      }

      newStartIndex = firstItemIndex;

      FeatStd::Float indexStartPosition = Math::Floor(position) - position + 1;
      if (!_pixelwise)
      {
         indexStartPosition = Math::Floor(indexStartPosition);
      }
      FeatStd::Float indexPosition = indexStartPosition;

      FeatStd::Int32 newLastItemIndex = firstItemIndex;
      if (_addingDirection == ListContentAdder::Forward || _addingDirection == ListContentAdder::ForwardAndReverse)
      {
         newLastItemIndex = ForwardAdd(firstItemIndex, indexStartPosition);
      }

      _endReached = (!IsCircularNeeded()) && ((newLastItemIndex + 1) == size);

      if (_addingDirection == ListContentAdder::Reverse || _addingDirection == ListContentAdder::ForwardAndReverse)
      {
         newStartIndex = ReverseAdd(newStartIndex, indexStartPosition);
      }

      _listItemAdder->FinishAdding();
      _listItemAdder->SetReverseMode(false);

      _numberOfCompleteVisibleItems = _listItemAdder->GetNumberOfCompleteVisibleItems();
   }

   _oldVirtualListSize = size;

   return newStartIndex;
}


void ListContentAdder::ResetMembers()
{
   _correctCurrentPosition = false;
   _invalidItemsAdded = false;
   _reversePositionCorrection = 0.0F;
   _endReached = false;
}


bool ListContentAdder::IsCircularNeeded() const
{
   return _listContentInfo.isCircular && (_elementsShownOnlyOnce || _listContentInfo.isCoverflow);
}


Candera::Node2D* ListContentAdder::AddItem(FeatStd::UInt32 index, Candera::Float indexPosition, bool& vacantVisibleArea, bool& invalidTemplate)
{
   Candera::Node2D* itemNode = 0;
   if ((!_listItemAdder.PointsToNull()))
   {
      Candera::Node2D* templateNode = _templateRetriever.GetTemplate(_contentProvider.GetItem(index));
      invalidTemplate = invalidTemplate || _templateRetriever.IsInvalidTemplate(templateNode);

      if (ExtractItemNode(index, itemNode))
      {
         ControlTemplateInstance* templateInstance = ControlTemplate::GetControlTemplateInstance(*itemNode);
         if (0 != templateInstance)
         {
            const Candera::Float itemPosition(_listItemAdder->GetPostAddSizeHint(*itemNode, _listArea));

            OnPreItemAdd(templateInstance, indexPosition, itemPosition);

            vacantVisibleArea = _listItemAdder->AddItem(*templateNode, *itemNode);

            static_cast<void>(itemNode->Upload(ScopeMask(), Candera::Node2D::Deep));
         }
      }
   }

   return itemNode;
}


Candera::Node2D* ListContentAdder::AddPixelWiseItem(Candera::Float position, FeatStd::UInt32 index, bool& vacantVisibleArea, bool& invalidTemplate)
{
   if (!_listItemAdder.PointsToNull())
   {
      _listItemAdder->SetReverseMode(false);
   }
   return AddItem(index, position, vacantVisibleArea, invalidTemplate);
}


Candera::Node2D* ListContentAdder::ReverseAddPixelWiseItem(Candera::Float position, FeatStd::UInt32 index, bool& vacantVisibleArea, bool& invalidTemplate)
{
   if (!_listItemAdder.PointsToNull())
   {
      _listItemAdder->SetReverseMode(true);
   }
   return AddItem(index, position, vacantVisibleArea, invalidTemplate);
}


void ListContentAdder::OnPreItemAdd(ControlTemplateInstance* templateInstance, FeatStd::Float indexPosition, Candera::Float itemPosition)
{
   if (0 != templateInstance)
   {
//      if (!_privateTimeDispatcher.PointsToNull() && (0 != _expandTriggerManager))
      if (0 != _expandTriggerManager)
      {
         _expandTriggerManager->UpdateItemState(templateInstance);
      }

      if (_listContentInfo.scrollAnimationInput == Candera::ScrollAnimationInput::IndexBased)
      {
         templateInstance->PositionAnimate(indexPosition);
      }
      else if (_listContentInfo.scrollAnimationInput == Candera::ScrollAnimationInput::PixelPositionBased)
      {
         templateInstance->PositionAnimate(itemPosition);
      }

      templateInstance->SpeedAnimate(_itemSpeedAnimationManager.GetCurrentSize());

      if (!_listItemAdder.PointsToNull())
      {
         const Int32 addedItems(_listItemAdder->GetNumberOfAddedItems());
         if (0 != _listener)
         {
            _listener->OnPreItemAdd(addedItems);
         }
      }
   }
}


FeatStd::Int32 ListContentAdder::ForwardAdd(FeatStd::UInt32 firstItemIdx, FeatStd::Float indexStartPosition)
{
   Int32 i(firstItemIdx);
   bool _elementsShownOnlyOnce = true;
   bool vacantVisibleArea(true);
   bool nextItemZeroSize(false);
   Float nextItemSize(0.0F);
   UInt32 listSize(_contentProvider.GetVirtualListSize());
   bool shouldAdd(listSize > 0);
   const Int32 adjustedSize(_listContentInfo.isCircular ? listSize + _contentProvider.GetWindowElementSize() : listSize);
   Candera::Node2D* invalidTemplate(_templateRetriever.GetTemplate(tSharedPtrListDataItem(0)));

   AddChecker addChecker(_pixelwise, _listContentInfo.configuredNumberOfItems);

   FeatStd::Float indexPosition = indexStartPosition;

   for (; shouldAdd && (i < adjustedSize); ++i)
   {
      // Insert item
      Candera::Node2D* itemNode(AddPixelWiseItem(indexPosition, i, vacantVisibleArea, _invalidItemsAdded));

      indexPosition++;

      _elementsShownOnlyOnce = (((i + 1) % listSize) != firstItemIdx);
      if ((!vacantVisibleArea) && ((i + 1) < adjustedSize))
      {
         // Check if next item size is 0
         shouldAdd = GetItemSize(i + 1, nextItemSize);
         nextItemZeroSize = (nextItemSize == 0);
      }
      else
      {
         shouldAdd = true;
         nextItemZeroSize = false;
      }

      shouldAdd = shouldAdd && addChecker.ShouldAdd(*_listItemAdder, vacantVisibleArea, nextItemZeroSize) && _elementsShownOnlyOnce;
   }

   _listItemAdder->FinishDirectAdding();

   return (i - 1);
}


FeatStd::Int32 ListContentAdder::ReverseAdd(FeatStd::UInt32 startIndex, FeatStd::Float indexStartPosition)
{
   bool shouldAdd(true);
   const UInt32 scrollingIncrement(_listItemAdder->GetScrollingIncrement());
   bool vacantVisibleArea(true);
   Candera::Node2D* invalidTemplate(_templateRetriever.GetTemplate(tSharedPtrListDataItem(0)));

   // no items should be added if the actual size of the added items is greater or equal to the viewport size
   shouldAdd = shouldAdd && (_listItemAdder->GetSizeOfAddedElements() < _orientationOperator.GetVectorComponent(_listArea));
   // no new items should be added if at least one item one has its size 0
   shouldAdd = shouldAdd && (_listItemAdder->GetZeroSizedItems() == 0);
   // Check if items got removed and have to be refilled or the max position was not yet calculated correctly
   shouldAdd = shouldAdd && ((_addingDirection == Reverse) || (_endReached && (!_listContentInfo.isFixedPages) && (!_listContentInfo.isCoverflow)));
   shouldAdd = shouldAdd && ((_addingDirection == Reverse) || (_oldVirtualListSize > _contentProvider.GetVirtualListSize()) || (_maxIndexManager != 0 && !_maxIndexManager->GetAccurateMaxPosition() && (!_listContentInfo.isGrid)));

   UInt32 itemsToAdd(0);
   Int32 newStartIndex = startIndex;

   if (shouldAdd)
   {
      if (!_pixelwise && 0 != _listContentInfo.configuredNumberOfItems)
      {
         FeatStd::UInt32 maxNumberOfItems = _listContentInfo.configuredNumberOfItems * scrollingIncrement;
         FeatStd::UInt32 addedItems = static_cast<FeatStd::UInt32>(_listItemAdder->GetNumberOfAddedItems());

         FeatStd::UInt32 emptyItems = scrollingIncrement - (addedItems % scrollingIncrement);
         if (emptyItems == scrollingIncrement)
         {
            emptyItems = 0;
         }
         itemsToAdd = maxNumberOfItems - addedItems - emptyItems;
         shouldAdd = (itemsToAdd > 0);
      }
      else
      {
         shouldAdd = vacantVisibleArea;
      }
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListContentAdder::AddListContent before reverse adding shouldAdd=%d itemsToAdd=%d", shouldAdd, itemsToAdd));

   Int32 i = startIndex - 1;
   if (shouldAdd)
   {
      FeatStd::Float indexPosition = indexStartPosition;

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListContentAdder::AddListContent reverse adding itemsToAdd=%d", itemsToAdd));

      _listItemAdder->SetReverseMode(true);
      _listItemAdder->BeginReverseAdding();

      for (; shouldAdd && (i >= 0); --i)
      {
         indexPosition += i;
         // Insert item
         Candera::Node2D* itemNode(ReverseAddPixelWiseItem(indexPosition, i, vacantVisibleArea, _invalidItemsAdded));
         if (itemsToAdd > 0)
         {
            itemsToAdd--;
         }
         _reversePositionCorrection -= 1.0F;

         shouldAdd = ((_pixelwise || (0 == _listContentInfo.configuredNumberOfItems) ? vacantVisibleArea : itemsToAdd > 0));
      }

      newStartIndex = i + 1;

      _listItemAdder->FinishReverseAdding();

      if (_addingDirection == ForwardAndReverse)
      {
         _correctCurrentPosition = true;
      }
   }

   return newStartIndex;
}


bool ListContentAdder::ExtractItemNode(const FeatStd::Int32 index, Candera::Node2D*& node)
{
   const tSharedPtrListDataItem& itemData = _contentProvider.GetItem(index);
   node = _itemsInstanceContainer.AddControlTemplateClone(_templateRetriever, itemData, index);

   if (node != 0)
   {
      node->SetLayouter(_listContentInfo.itemLayouter);
      return true;
   }
   else
   {
      return false;
   }
}


bool ListContentAdder::GetItemSize(const FeatStd::Int32 itemIndex, FeatStd::Float& size)
{
   Candera::Node2D* node;
   if (ExtractItemNode(itemIndex, node))
   {
      size = _listItemAdder->GetItemSize(*node);
      return true;
   }
   else
   {
      size = 0.0F;
      return false;
   }
}
