/* ***************************************************************************************
* FILE:          FlexPixelWiseAnimationManager.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FlexPixelWiseAnimationManager is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef Flex_Pixel_Wise_Animation_Manager_H
#define Flex_Pixel_Wise_Animation_Manager_H

#include "FlexPixelWiseAnimationManagerListener.h"

#include "Widgets/2D/List/generated/ListWidget2DBase.h"
#include "TimeDispatcher.h"

class FlexPixelWiseAnimationManager
{
   public:
      FlexPixelWiseAnimationManager();
      virtual ~FlexPixelWiseAnimationManager();

      void Init(TimeDispatcher* timeDispatcher);

      Candera::Float GetPosition() const;
      void SetPosition(Candera::Float position);

      void SetAnimationTime(Candera::UInt32 val)
      {
         _animationTime = val;
      }

      void SetDelay(Candera::UInt32 val)
      {
         _delay = val;
      }

      void SetFinishCurrentAnimation(bool flag)
      {
         _finishCurrentAnimation = flag;
      }

      /**
       * Entry point into animation.
       * @param targetPosition target position in normalized space.
       * @return true if list items need to be redrawn.
       */
      bool RunAnimationIfNeeded(Candera::Float targetPosition);

      /**
       *  Sets a new position without animating. Used to jump from one end of the list to the other end.
       *  @param newPosition the new position to set.
       */
      void RemapPosition(Candera::Float newPosition);

      void SetListener(FlexPixelWiseAnimationManagerListener* listener);

   protected:
      void NotifyOnAnimationFinished();

   private:

      /**
       *  Starts a new animation to the given target position.
       *  @param targetPosition target position in normalized space.
       */
      void StartAnimation(Candera::Float targetPosition);

      /**
       *  Runs the current animation to the given target position.
       *  @param targetPosition target position in normalized space.
       */
      void RunCurrentAnimation(Candera::Float targetPosition);

      /**
       *  Finish the current animation at the given target index.
       *  @param targetIndex target position in normalized space.
       */
      void FinishAnimation(Candera::Float targetPosition);

      /**
       *  Calculates a velocity based on the current position, target position and animation time.
       */
      void UpdateVelocity(Candera::Float targetPosition);

      /**
       *  Update current position and last update time.
       *  @param currentTime
       */
      void AnimatePixelWiseItemEntrance(Candera::UInt32 elapsedAnimationTime);

      FeatStd::UInt32 GetCurrentTime() const;

      Candera::UInt32 _animationTime;                                ///< The total time the animation is allowed to take. Used to calculate velocity.
      Candera::UInt32 _delay;                                        ///< The delay to start the animation.

      bool _isAnimationRunning;                                      ///< True if an animation is active.
      bool _finishCurrentAnimation;                                  ///< True if the current animation has to be finished before a new one is started
      Candera::Float _startAnimationPosition;                        ///< Last known target position in normalized space (where the last animation finished).
      Candera::Float _endAnimationPosition;                          ///< Current target position we are animating to.
      Candera::UInt32 _animationStartingTime;                        ///< The time the current animation started.
      Candera::UInt32 _lastUpdateTime;                               ///< The last time the animation state was updated.
      Candera::Float _startPosition;                                 ///< the position value at startup
      Candera::Float _position;                                      ///< Current position in normalized space.
      Candera::Float _animationVelocity;                             ///< The velocity with which to change the position.
      TimeDispatcher* _timeDispatcher;                               ///< object responsible for retrieving the current time
      FlexPixelWiseAnimationManagerListener* _listener;              ///< listener that is notified on various events
};


#endif
