/* ***************************************************************************************
* FILE:          FlexPixelWiseAnimationManager.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FlexPixelWiseAnimationManager is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "FlexPixelWiseAnimationManager.h"

using namespace Candera;
using namespace Courier;

FlexPixelWiseAnimationManager::FlexPixelWiseAnimationManager() :
   _animationTime(0),
   _delay(0),
   _isAnimationRunning(false),
   _finishCurrentAnimation(true),
   _startAnimationPosition(0),
   _endAnimationPosition(0),
   _animationStartingTime(0),
   _lastUpdateTime(0),
   _startPosition(0.0),
   _position(0.0),
   _animationVelocity(0),
   _timeDispatcher(0),
   _listener(0)
{
}


FlexPixelWiseAnimationManager::~FlexPixelWiseAnimationManager()
{
   _listener = 0;
   _timeDispatcher = 0;
}


void FlexPixelWiseAnimationManager::Init(TimeDispatcher* timeDispatcher)
{
   _timeDispatcher = timeDispatcher;
}


Candera::Float FlexPixelWiseAnimationManager::GetPosition() const
{
   return _position;
}


void FlexPixelWiseAnimationManager::SetPosition(Candera::Float position)
{
   _position = position;
   _endAnimationPosition = position;
}


void FlexPixelWiseAnimationManager::RemapPosition(Candera::Float newPosition)
{
   _position = newPosition;
   _startAnimationPosition = newPosition;
   _endAnimationPosition = newPosition;
   _lastUpdateTime = _animationStartingTime + _animationTime;

   _isAnimationRunning = false;
   NotifyOnAnimationFinished();
}


void FlexPixelWiseAnimationManager::SetListener(FlexPixelWiseAnimationManagerListener* listener)
{
   _listener = listener;
}


void FlexPixelWiseAnimationManager::NotifyOnAnimationFinished()
{
   if (0 != _listener)
   {
      _listener->OnAnimationFinished();
   }
}


bool FlexPixelWiseAnimationManager::RunAnimationIfNeeded(Candera::Float targetPosition)
{
   bool invalidationNeeded = _isAnimationRunning;

   if (!Math::FloatAlmostEqual(_endAnimationPosition, targetPosition))
   {
      // finish the current started animation
      if (_finishCurrentAnimation)
      {
         RunCurrentAnimation(_endAnimationPosition);
      }
      else
      {
         _finishCurrentAnimation = true;
      }

      // start a new animation and animate the first frame
      StartAnimation(targetPosition);
      RunCurrentAnimation(targetPosition);
   }
   else if (!Math::FloatAlmostEqual(_position, _endAnimationPosition))
   {
      RunCurrentAnimation(targetPosition);
   }
   else
   {
      FinishAnimation(targetPosition);
   }

   return invalidationNeeded || _isAnimationRunning;
}


void FlexPixelWiseAnimationManager::StartAnimation(Candera::Float targetPosition)
{
   UpdateVelocity(targetPosition);
   _endAnimationPosition = targetPosition;
   _animationStartingTime = GetCurrentTime();
   _lastUpdateTime = _animationStartingTime;
   _isAnimationRunning = true;
   _startPosition = _position;
}


void FlexPixelWiseAnimationManager::RunCurrentAnimation(Candera::Float targetPosition)
{
   if (_isAnimationRunning)
   {
      UInt32 currentTime = GetCurrentTime();
      UInt32 timeElapsed = (currentTime - _animationStartingTime);

      if (timeElapsed >= _delay)
      {
         UInt32 elapsedAnimationTime = timeElapsed - _delay;
         if (elapsedAnimationTime < _animationTime)
         {
            AnimatePixelWiseItemEntrance(elapsedAnimationTime);
         }
         else
         {
            FinishAnimation(targetPosition);
         }
      }

      _lastUpdateTime = currentTime;
   }
}


void FlexPixelWiseAnimationManager::FinishAnimation(Candera::Float targetPosition)
{
   _lastUpdateTime = _animationStartingTime + _animationTime + _delay;
   _position = _endAnimationPosition;

   _startAnimationPosition = targetPosition;
   _isAnimationRunning = false;

   NotifyOnAnimationFinished();
}


void FlexPixelWiseAnimationManager::UpdateVelocity(Candera::Float targetPosition)
{
   if (_animationTime != 0)
   {
      Float scrollingDistance = targetPosition - _position;
      _animationVelocity = scrollingDistance / _animationTime;
   }
   else
   {
      _animationVelocity = 0.0F;
   }
}


void FlexPixelWiseAnimationManager::AnimatePixelWiseItemEntrance(UInt32 elapsedAnimationTime)
{
   const Float distance = _animationVelocity * elapsedAnimationTime;

   _position = _startPosition + distance;
}


FeatStd::UInt32 FlexPixelWiseAnimationManager::GetCurrentTime() const
{
   UInt32 now = 0;
   if (0 != _timeDispatcher)
   {
      now = _timeDispatcher->GetTimeMs();
   }

   return now;
}
