/* ***************************************************************************************
* FILE:          ListContentUpdater.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ListContentUpdater is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "ContentProvider.h"

using namespace FeatStd;

ContentProvider::ContentProvider() :
   _listData(0),
   _items(0)
{
}


ContentProvider::~ContentProvider()
{
   _listData = 0;
   _items = tSharedPtrListDataItem(0);
}


void ContentProvider::SetData(const ListDataWrapper* listData)
{
   _listData = listData;
}


void ContentProvider::SetItems(const tSharedPtrListDataItem& items)
{
   _items = items;
}


bool ContentProvider::HasItems() const
{
   return (!_items.PointsToNull()) || ((_listData != 0) && (_listData->IsDataAvailable()));
}


tSharedPtrListDataItem ContentProvider::GetItem(FeatStd::Int32 index) const
{
   tSharedPtrListDataItem item(0);

   if ((_listData != 0) && _listData->IsDataAvailable())
   {
      const Int32 virtualListSize = static_cast<Int32>(_listData->GetVirtualListSize());
      const Int32 startIndex = static_cast<Int32>(_listData->GetStartIndex());
      if (virtualListSize > 0)
      {
         const Int32 dataIndex = ((index - startIndex) + virtualListSize) % virtualListSize;
         item = (*_listData)[dataIndex];
      }
   }
   else if (_items != 0)
   {
      const Int32 itemsSize = static_cast<Int32>(_items->size());
      if ((_items != 0) && (index >= 0) && (index < itemsSize))
      {
         item = tSharedPtrListDataItem(Dynamic_Cast<ListDataItem*>((*_items)[index].GetPointerToSharedInstance()));
      }
   }

   return item;
}


FeatStd::Int32 ContentProvider::GetStartIndex() const
{
   Int32 startIndex(0);

   if ((_listData != 0) && _listData->IsDataAvailable())
   {
      startIndex = static_cast<Int32>(_listData->GetStartIndex());
   }
   else if (!_items.PointsToNull())
   {
      startIndex = 0;
   }
   else
   {
      FEATSTD_LINT_NEXT_EXPRESSION(774, "Intended debug assert");
      FEATSTD_DEBUG_ASSERT(false); // should never get here
   }

   return startIndex;
}


FeatStd::Int32 ContentProvider::GetWindowElementSize() const
{
   Int32 windowElementSize(0);

   if ((_listData != 0) && _listData->IsDataAvailable())
   {
      windowElementSize = static_cast<Int32>(_listData->GetWindowElementSize());
   }
   else if (!_items.PointsToNull())
   {
      windowElementSize = static_cast<Int32>(_items->size());
   }
   else
   {
      FEATSTD_LINT_NEXT_EXPRESSION(774, "Intended debug assert");
      FEATSTD_DEBUG_ASSERT(false); // should never get here
   }

   return windowElementSize;
}


FeatStd::Int32 ContentProvider::GetVirtualListSize() const
{
   Int32 virtualListSize(0);

   if ((_listData != 0) && _listData->IsDataAvailable())
   {
      virtualListSize = static_cast<Int32>(_listData->GetVirtualListSize());
   }
   else if (!_items.PointsToNull())
   {
      virtualListSize = static_cast<Int32>(_items->size());
   }
   else
   {
      FEATSTD_LINT_NEXT_EXPRESSION(774, "Intended debug assert");
      FEATSTD_DEBUG_ASSERT(false); // should never get here
   }

   return virtualListSize;
}


FeatStd::UInt32 ContentProvider::GetListId() const
{
   UInt32 id(0);

   if (_listData != 0)
   {
      id = _listData->GetListId();
   }

   return id;
}


FeatStd::SizeType ContentProvider::GetSequenceNumber() const
{
   return (_listData != 0) ? _listData->GetSequenceNumber() : 0;
}
