/* ***************************************************************************************
* FILE:          OrthoGridPageInfoCollection.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  OrthoGridPageInfoCollection is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "OrthoGridPageInfoCollection.h"
#include "Candera/System/Mathematics/Math.h"

using namespace FeatStd;

OrthoGridPageInfoCollection::OrthoGridPageInfoCollection() :
   _currentPageIndex(0),
   _minCellSize(Candera::Math::MaxFloat())
{
}


void OrthoGridPageInfoCollection::AddPage(const OrthoGridPageInfo& pageInfo)
{
   const FeatStd::Int32 startStructIndex(pageInfo.GetStartStructIndex());

   SizeType index(0);
   for (; (index < _pages.Size()) && (_pages[index].GetStartStructIndex() < startStructIndex); ++index);

   if (index == _pages.Size())
   {
      _pages.Add(pageInfo);
   }
   else
   {
      _pages.Insert(index, pageInfo);
   }
}


const OrthoGridPageInfo& OrthoGridPageInfoCollection::GetPageByStructIndex(FeatStd::Int32 structIndex) const
{
   static OrthoGridPageInfo empty;

   SizeType index(0);
   for (; (index < _pages.Size()) && (!_pages[index].ContainsStructIndex(structIndex)); ++index);

   if (index == _pages.Size())
   {
      return empty;
   }
   else
   {
      return _pages[index];
   }
}


const OrthoGridPageInfo& OrthoGridPageInfoCollection::GetPageByItemIndex(FeatStd::Int32 itemIndex) const
{
   static OrthoGridPageInfo empty;

   SizeType index(0);
   for (; (index < _pages.Size()) && (!_pages[index].ContainsItemIndex(itemIndex)); ++index);

   if (index == _pages.Size())
   {
      return empty;
   }
   else
   {
      return _pages[index];
   }
}


const FeatStd::Int32 OrthoGridPageInfoCollection::GetMaxStructStartIndex() const
{
   Int32 structIndex(0);
   if (!_pages.Empty())
   {
      structIndex = _pages[_pages.Size() - 1].GetStartStructIndex();
   }
   return structIndex;
}


bool OrthoGridPageInfoCollection::RetrieveStartIndexOfPage(const FeatStd::Int32 pageIndex, FeatStd::Int32& startIndex) const
{
   bool found(false);
   Int32 index(_currentPageIndex + pageIndex);
   startIndex = 0;

   if (index < 0)
   {
      index = 0;
   }
   else if (index >= _pages.Size())
   {
      index = _pages.Size() - 1;
   }
   else
   {
      found = true;
   }

   if (index < _pages.Size())
   {
      startIndex = _pages[index].GetStartItemIndex();
   }

   return found;
}


FeatStd::Int32 OrthoGridPageInfoCollection::GetStartIndexOfLastPage() const
{
   Int32 startItemIndex(0);
   if (_pages.Size() > 0)
   {
      startItemIndex = _pages[_pages.Size() - 1].GetStartItemIndex();
   }
   return startItemIndex;
}


void OrthoGridPageInfoCollection::MarkCurrentPage(Int32 structIndex)
{
   static OrthoGridPageInfo empty;

   SizeType index(0);
   for (; (index < _pages.Size()) && (!_pages[index].ContainsStructIndex(structIndex)); ++index);

   if (index < _pages.Size())
   {
      _currentPageIndex = index;
   }
}


void OrthoGridPageInfoCollection::AdjustMinCellSize(FeatStd::Float& val)
{
   if (_minCellSize == Candera::Math::MaxFloat())
   {
      _minCellSize = val;
   }
   else
   {
      Float halfSize(_minCellSize * 0.5F);
      while (val < halfSize)
      {
         _minCellSize = halfSize;
         halfSize = _minCellSize * 0.5F;
      }
   }

   val = _minCellSize;
}


void OrthoGridPageInfoCollection::Clear()
{
   _pages.Clear();
   _minCellSize = Candera::Math::MaxFloat();
   _currentPageIndex = 0;
}
