/* ***************************************************************************************
* FILE:          OrthoGridItemAdder.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  OrthoGridItemAdder is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "Widgets/2D/List/Layouting/FlexListLayouter.h"
#include "Candera/System/Mathematics/Math.h"
#include "FeatStd/Util/BinarySearch.h"
#include "OrthoCellResolver.h"
#include "OrthoItemArranger.h"

#include "widget2D_std_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/OrthoGridItemAdder.cpp.trc.h"
#endif

#include "OrthoGridItemAdder.h"

using namespace Candera;

CANDERA_RTTI_DEFINITION(OrthoGridItemAdder);


// OrthoGridItemAdder
OrthoGridItemAdder::OrthoGridItemAdder(const ITemplateSpanAnalyzer::SharedPointer& templateSpanAnalyzer, ISizeContainer& sizesContainer, Candera::Node2D* groupNode, OrthoGridPageInfoCollection& pagesInfo, const OrientationOperator& directOrientationOperator, const OrientationOperator& orthogonalOrientationOperator, Candera::UInt32 rows, Candera::UInt32 columns) :
   Base(sizesContainer, groupNode, directOrientationOperator, orthogonalOrientationOperator),
   _rows(rows),
   _columns(columns),
   _currentOrthogonalIndex(-1),
   _itemsInStructure(0),
   _absoluteStructureIndex(0),
   _logicalStructureIndex(0),
   _computedRows(-1),
   _computedColumns(-1),
   _maxItemCellSize(0.0),
   _itemsInFirstStructure(0),
   _gridOccupation(0),
   _firstInStructure(false),
   _completeBeginingMargin(0),
   _templateSpanAnalyzer(templateSpanAnalyzer),
   _pagesInfo(pagesInfo),
   _maxOrthogonalItemsCount(0),
   _currentStructureSize(0.0F),
   _visibleSize(0.0F),
   _unwrappedItemsSize(0.0F),
   _currentMinCellSize(0.0F),
   _emptyCellsPolicy(0),
   _firstItem(0)
{
   if (_rows <= 0)
   {
      _rows = 1;
   }

   if (_columns <= 0)
   {
      _columns = 1;
   }
}


OrthoGridItemAdder::SharedPointer OrthoGridItemAdder::Create(const ITemplateSpanAnalyzer::SharedPointer& templateSpanAnalyzer, ISizeContainer& sizesContainer, Candera::Node2D* groupNode, OrthoGridPageInfoCollection& pagesInfo, const OrientationOperator& directOrientationOperator, const OrientationOperator& orthogonalOrientationOperator, Candera::UInt32 rows, Candera::UInt32 columns)
{
   return OrthoGridItemAdder::SharedPointer(CANDERA_NEW(OrthoGridItemAdder)(templateSpanAnalyzer, sizesContainer, groupNode, pagesInfo, directOrientationOperator, orthogonalOrientationOperator, rows, columns));
}


OrthoGridItemAdder::~OrthoGridItemAdder()
{
   _gridOccupation = 0;
   _firstItem = 0;
   SetEmptyCellsPolicy(OrthoGridEmptyCellsPolicy::SharedPointer(0));
}


void OrthoGridItemAdder::OnBeforeBeginAdding()
{
   const Int32 structStartIndex(static_cast<Int32>(GetOriginalPosition()));
   _page0 = _pagesInfo.GetPageByStructIndex(structStartIndex);
   _pagesInfo.MarkCurrentPage(structStartIndex);

   const Int32 nextPageStructStart(_page0.GetEndStructIndex() + 1);
   _page1 = _pagesInfo.GetPageByStructIndex(nextPageStructStart);
}


void OrthoGridItemAdder::OnBeginAdding()
{
   _firstInStructure = true;
   _logicalStructureIndex = static_cast<Int32>(GetAdjustedPosition() - GetOriginalPosition());
   _absoluteStructureIndex = 0;
   _maxOrthogonalItemsCount = 0;

   Vector2 start(0.0F, 0.0F);
   _directOrientationOperator.SetVectorComponent(start, GetAdjustedPosition());
   _currentOccupation.Clear();

   _visibleSize = _directOrientationOperator.GetVectorComponent(GetVisibleArea());
   _unwrappedItemsSize = 0.0F;
   _currentStructureSize = 0.0F;

   _sizes.Clear();
   _sizes.Add(SizesCollection());

   _currentMinCellSize = Math::MaxFloat();
   _currentOrthogonalIndex = 0;
   _firstItem = 0;
}


void OrthoGridItemAdder::OnBeginReverseAdding()
{
   if (0 != GetGroupNode())
   {
      _currentOrthogonalIndex = GetOrthogonalItemsCount() - 1;
      _absoluteStructureIndex = -1;
   }
}


bool OrthoGridItemAdder::AddItem(Candera::Node2D& templateNode, Candera::Node2D& childNode)
{
   const Float itemSize(InsertItem(childNode));

   if (0 == _firstItem)
   {
      _firstItem = &childNode;
   }

   PreAdd(childNode, itemSize);
   Add(childNode, itemSize);
   PostAdd(itemSize);

   return _unwrappedItemsSize < (2 * _maxOrthogonalItemsCount * _visibleSize);
}


void OrthoGridItemAdder::OnFinishAdding()
{
   if (GetNumberOfAddedItems() > 0)
   {
      ProcessStructureEnd();
      CalculateCells();

      if (0 != _completeBeginingMargin)
      {
         _numberOfCompleteVisibleItems -= 1;
      }
   }
}


void OrthoGridItemAdder::OnFinishReverseAdding()
{
   if (_reverse)
   {
      Node2D* groupNode = GetGroupNode();
      if (0 != groupNode)
      {
         FeatStd::UInt8 maxShiftValue = GetMaxShiftValue();
         Node2D* childNode = groupNode->GetFirstChild();
         while (0 != childNode)
         {
            ShiftGridIndex(*childNode, maxShiftValue);
            childNode = childNode->GetNextSibling();
         }
      }
   }
}


Candera::Int16 OrthoGridItemAdder::GetBeginingMargin() const
{
   return _completeBeginingMargin;
}


void OrthoGridItemAdder::ShiftGridIndex(Candera::Node2D& node, FeatStd::UInt8 val) const
{
   val += _directOrientationOperator.GetRowOrColumn(node);
   _directOrientationOperator.SetRowOrColumn(node, val);
}


FeatStd::UInt8 OrthoGridItemAdder::GetShiftValue(const Candera::Node2D& node) const
{
   return _directOrientationOperator.GetRowOrColumn(node);
}


void OrthoGridItemAdder::Clear()
{
   _itemsInStructure = 0;
   _computedRows = -1;
   _computedColumns = -1;
   _maxItemCellSize = 0.0;
   _itemsInFirstStructure = 0;
   _completeBeginingMargin = 0;

   Base::Clear();
}


Candera::Int32 OrthoGridItemAdder::GetNumberOfItemsInFirstStructure() const
{
   return _itemsInFirstStructure;
}


Candera::Int32 OrthoGridItemAdder::GetNumberOfStructures() const
{
   return Math::Maximum(_directOrientationOperator.GetValue(_computedColumns, _computedRows), (FeatStd::Int32) 0);
}


Candera::Int32 OrthoGridItemAdder::GetCurrentIndex() const
{
   return _absoluteStructureIndex;
}


bool OrthoGridItemAdder::IsFirstInStructure() const
{
   return _firstInStructure;
}


Candera::UInt32 OrthoGridItemAdder::GetOrthogonalItems() const
{
   return GetOrthogonalItemsCount();
}


void OrthoGridItemAdder::SetGridOccupation(GridOccupation* val)
{
   _gridOccupation = val;
}


Candera::UInt32 OrthoGridItemAdder::GetScrollingIncrement() const
{
   return _directOrientationOperator.GetValue(_rows, _columns);
}


Candera::Float OrthoGridItemAdder::GetPositionCorrection(const FeatStd::Float position) const
{
   Int32 correction(0);

   const Int32 col(static_cast<Int32>(position));
   if (!_page0.IsEmpty())
   {
      correction = _page0.GetStartStructIndex() - col;
   }

   return static_cast<Float>(correction);
}


void OrthoGridItemAdder::PreAdd(Candera::Node2D& childNode, const Float itemSize)
{
//   if ((_absoluteStructureIndex == 0) && (_firstInStructure))
   {
      _maxOrthogonalItemsCount = GetOrthogonalItemsCount();
   }

   _unwrappedItemsSize += itemSize;

   _firstInStructure = (_itemsInStructure == 0);

   if ((!_firstInStructure) && (_currentStructureSize + itemSize) > _visibleSize)
   {
      EndStructure();
   }
}


void OrthoGridItemAdder::Add(Candera::Node2D& childNode, const Float itemSize)
{
   if (itemSize < _currentMinCellSize)
   {
      _currentMinCellSize = itemSize;
   }

   _sizes[_currentOrthogonalIndex].Add(itemSize);
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OrthoGridItemAdder::Add size = %f", itemSize));
}


void OrthoGridItemAdder::PostAdd(const Float itemSize)
{
   ++_itemsInStructure;
   _currentStructureSize += itemSize;

   if (_reverse)
   {
      // do nothing; not supported
   }
   else
   {
      if (_currentStructureSize > _visibleSize)
      {
         EndStructure();
      }
   }
}


void OrthoGridItemAdder::EndStructure()
{
   ProcessStructureEnd();

   ++_absoluteStructureIndex;
   ++_logicalStructureIndex;
   _currentStructureSize = 0.0F;
}


void OrthoGridItemAdder::ProcessStructureEnd()
{
   Float currentSize(GetCurrentSize());
   ResetSize();
   IncreaseSize(Math::Maximum(currentSize, _currentStructureSize));
   const Float visibleSize(_directOrientationOperator.GetVectorComponent(GetVisibleArea()));
   currentSize = GetCurrentSize();

   if (currentSize >= visibleSize)
   {
      _endingMargin = Int16(visibleSize - currentSize);

      const Int16 margin = 0 - (_completeBeginingMargin + _endingMargin);

      if ((margin > _maxItemCellSize) || ((_completeBeginingMargin == 0) != (_endingMargin == 0)))
      {
         _numberOfCompleteVisibleItems -= _itemsInStructure;
      }
   }

   _maxItemCellSize = 0;
   _itemsInStructure = 0;

   ++_currentOrthogonalIndex;
   _sizes.Add(SizesCollection());
}


void OrthoGridItemAdder::CalculateCells()
{
   PrepareCalculateCells();

   const Int32 numberOfCells(static_cast<Int32>(Math::Floor(_visibleSize / _currentMinCellSize)));

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OrthoGridItemAdder::CalculateCells visibleSIze = %f, minCellSize = %f, numberOfCells = %d", _visibleSize, _currentMinCellSize, numberOfCells));

   const Float deltaPosition(GetAdjustedPosition() - GetOriginalPosition());
   if (0 != _firstItem)
   {
      Node2D* item(_firstItem);
      bool newPage0(_page0.IsEmpty());
      bool newPage1(_page1.IsEmpty());

      Int32 itemIndex(PreparePages(numberOfCells));

      const Int32 startPosition(static_cast<Int32>(GetAdjustedPosition()));
      _numberOfCompleteVisibleItems = 0;

      Int32 pageIndex(0);

      OrthoCellResolver cellResolver(_emptyCellsPolicy, _sizes);
      OrthoItemArranger itemArranger(_directOrientationOperator, _orthogonalOrientationOperator, _numberOfCompleteVisibleItems,
                                     _pagesInfo, _page0, _page1, item, _maxOrthogonalItemsCount, itemIndex, _structureOffset, newPage0, newPage1);
      for (SizeType structIndex(0); itemArranger.HasNextItem() && (structIndex < _sizes.Size()); ++structIndex)
      {
         if (_maxOrthogonalItemsCount > 0)
         {
            pageIndex = structIndex / _maxOrthogonalItemsCount;
         }
         Float newStructureSize(0.0F);

         Courier::Vector<Int8> spans(cellResolver.CalculateSpans(_visibleSize, _currentMinCellSize, numberOfCells, structIndex));

         itemArranger.Layout(spans, structIndex, pageIndex, numberOfCells, startPosition, deltaPosition, _sizes.Size());
         CollectSize(_currentMinCellSize);
      }

      itemArranger.FinishLayout(pageIndex);

      Int8 maxRelPos(itemArranger.GetMaxRelPos());

      EndCalculateCells(maxRelPos);
   }
}


void OrthoGridItemAdder::EndCalculateCells(Int8& maxRelPos)
{
   Int8 cols(_directOrientationOperator.GetValue(maxRelPos, static_cast<Int8>(_maxOrthogonalItemsCount)));
   Int8 rows(_orthogonalOrientationOperator.GetValue(maxRelPos, static_cast<Int8>(_maxOrthogonalItemsCount)));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OrthoGridItemAdder::CalculateCells rows = %d, cols = %d", rows, cols));
   _computedColumns = cols;
   _computedRows = rows;
   FlexListLayouter* layouter = dynamic_cast<FlexListLayouter*>(GetGroupNodeLayouter());
   if (0 != layouter)
   {
      GridLayouter* gridLayouter = dynamic_cast<GridLayouter*>(layouter->GetOriginalLayouter());
      if (0 != gridLayouter)
      {
         gridLayouter->SetLayout(UInt8(rows), UInt8(cols));
         for (UInt8 i = 0; i < cols; ++i)
         {
            _directOrientationOperator.SetStructureSize(gridLayouter, i, _currentMinCellSize);
         }
      }
   }
}


Int32 OrthoGridItemAdder::PreparePages(Int32 numberOfCells)
{
   Int32 itemIndex(0);
   const Int32 startPos(static_cast<Int32>(GetAdjustedPosition()));
   const Int32 endPos(startPos + numberOfCells - 1);

   const OrthoGridPageInfo& previousPage(_pagesInfo.GetPageByStructIndex(startPos - 1));

   if (!previousPage.IsEmpty())
   {
      itemIndex = previousPage.GetEndItemIndex() + 1;
   } // else itemIndex == 0

   if (_page0.IsEmpty())
   {
      _page0.SetStructureIndices(startPos, endPos);
      _page0.SetStartItemIndex(itemIndex);
   }

   if (_page1.IsEmpty())
   {
      _page1.SetStructureIndices(endPos + 1, endPos + numberOfCells);
   }

   return itemIndex;
}


void OrthoGridItemAdder::PrepareCalculateCells()
{
   const Int32 rawNumberOfCells(static_cast<Int32>(Math::Floor(_visibleSize / _currentMinCellSize)));
   _currentMinCellSize = _visibleSize / static_cast<Float>(rawNumberOfCells);

   _pagesInfo.AdjustMinCellSize(_currentMinCellSize);

   const Float deltaPos(GetAdjustedPosition() - GetOriginalPosition());

   _completeBeginingMargin -= _beginingMargin;
   UpdateBeginingMargin(GetSizes());
   _completeBeginingMargin += _beginingMargin;
   _completeBeginingMargin += static_cast<Int16>(deltaPos * _currentMinCellSize);

   _beginingSize = _currentMinCellSize;
   _endingSize = _currentMinCellSize;
}


void OrthoGridItemAdder::AdjustSpans(const Int32 deltaSpan, Courier::Vector<Int8>& spans, const Courier::Vector<SpanInfo>& orderedSpans)
{
   if (!_emptyCellsPolicy.PointsToNull())
   {
      _emptyCellsPolicy->AdjustSpans(deltaSpan, spans, orderedSpans);
   }
}


Candera::Float OrthoGridItemAdder::GetItemOrthogonalSize(Candera::Node2D& childNode)
{
   return _orthogonalOrientationOperator.GetVectorComponent(_sizesContainer.MeasureClientSize(&childNode));
}


Candera::UInt32 OrthoGridItemAdder::GetItemsCount() const
{
   return _directOrientationOperator.GetValue(_columns, _rows);
}


Candera::UInt32 OrthoGridItemAdder::GetOrthogonalItemsCount() const
{
   return _orthogonalOrientationOperator.GetValue(_columns, _rows);
}


FeatStd::UInt8 OrthoGridItemAdder::GetMaxShiftValue()
{
   Node2D* groupNode = GetGroupNode();
   FeatStd::UInt8 maxShiftVal = 0;
   if (0 != groupNode)
   {
      Node2D* childNode = groupNode->GetFirstChild();
      if (0 != childNode)
      {
         FeatStd::Int8 val = GetShiftValue(*childNode);
         maxShiftVal = Math::Maximum(maxShiftVal, static_cast<UInt8>(MathPlatform::Absolute(static_cast<FeatStd::Int32>(val))));
         childNode = childNode->GetNextSibling();
      }
   }

   return maxShiftVal;
}
