/* ***************************************************************************************
* FILE:          OrthoCellResolver.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  OrthoCellResolver is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"

#include "SpanInfo.h"
#include "FeatStd/Util/BinarySearch.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/OrthoCellResolver.cpp.trc.h"
#endif

#include "OrthoCellResolver.h"

using namespace Candera;

OrthoCellResolver::OrthoCellResolver(OrthoGridEmptyCellsPolicy::SharedPointer emptyCellsPolicy, SizesBidimensionalCollections& sizes) :
   _emptyCellsPolicy(emptyCellsPolicy),
   _sizes(sizes),
   _totalSpan(),
   _newStructureSize(),
   _index()
{
}


Courier::Vector<Int8> OrthoCellResolver::CalculateSpans(FeatStd::Float visibleSize, FeatStd::Float currentMinCellSize, FeatStd::Int32 numberOfCells, UInt32 structIndex)
{
   SizesCollection& structureSizes(_sizes[structIndex]);
   FitItems(structureSizes, visibleSize, currentMinCellSize, numberOfCells);
   AdjustSpans(numberOfCells);
   DistributeToNextStructure(structureSizes.Size(), structIndex, visibleSize);

   return _spans;
}


void OrthoCellResolver::AdjustSpans(FeatStd::Int32 numberOfCells)
{
   if (!_emptyCellsPolicy.PointsToNull())
   {
      const Int32 deltaSpan(numberOfCells - _totalSpan);
      _emptyCellsPolicy->AdjustSpans(deltaSpan, _spans, _orderedSpans);
   }
}


void OrthoCellResolver::FitItems(const SizesCollection& structureSizes, Float visibleSize, Float currentMinCellSize, Int32 numberOfCells)
{
   _newStructureSize = 0.0F;

   _spans.Clear();
   _orderedSpans.Clear();

   // iterate and see how many items fit; calculate the remaining span
   _spans.Reserve(structureSizes.Size()); // reserve the max capacity from the begining to avoid memory relocation
   _totalSpan = 0;

   for (_index = 0; (_newStructureSize < visibleSize) && (_index < structureSizes.Size()); ++_index)
   {
      const Float size(structureSizes[_index]);
      const Float rawSpan(Math::Ceil(size / currentMinCellSize));
      const Float span(rawSpan > numberOfCells ? numberOfCells : rawSpan);

      _newStructureSize += span * currentMinCellSize;
      if ((_newStructureSize <= visibleSize) || (_index == 0))
      {
         _totalSpan += static_cast<Int32>(span);

         _spans.Add(static_cast<Int8>(span));
         SizeType index(_spans.Size() - 1);
         SpanInfo addedSpan(_spans[index], index);

         SizeType insertAt;
         FeatStd::Internal::BinarySearch<FeatStd::Internal::StdComparer<SpanInfo, SpanInfo> >(addedSpan, _orderedSpans, _orderedSpans.Size(), &insertAt);
         _orderedSpans.Insert(insertAt, addedSpan);
      }
   }
}


void OrthoCellResolver::DistributeToNextStructure(UInt32 structureSizesCount, UInt32 structIndex, Float visibleSize)
{
   const bool remainingItems(_index < structureSizesCount);
   const bool oversized = _newStructureSize > visibleSize;
   if (remainingItems || oversized)
   {
      if ((structIndex + 1) >= _sizes.Size())
      {
         _sizes.Add(SizesCollection());
      }

      SizesCollection& structureSizes(_sizes[structIndex]);

      const SizeType targetCopyIndex(oversized ? _index - 1 : _index);
      SizesCollection& nextStructureSizes(_sizes[structIndex + 1]);
      for (SizeType j(structureSizes.Size() - 1); j >= targetCopyIndex; --j)
      {
         Float size(structureSizes[j]);
         nextStructureSizes.Insert(0, size);
      }
   }
}
