/* ***************************************************************************************
* FILE:          OrientationOperator.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  OrientationOperator is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
#include <Candera/EngineBase/Layout/Layouter.h>
#else
#include <Candera/Engine2D/Layout/Layouter.h>
#endif

#include "OrientationOperator.h"

#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
#include <Candera/EngineBase/Layout/StackLayouter.h>
#else
#include <Candera/Engine2D/Layout/StackLayouter.h>
#endif

using namespace Candera;


static Candera::Float GetSizeDefault(const Candera::Vector2& /*v*/)
{
   return 0.0F;
}


static void SetSizeDefault(Candera::Vector2& /*v*/, FeatStd::Float /*val*/)
{
}


static Candera::Float GetHeight(const Candera::Vector2& v)
{
   return v.GetY();
}


static void SetX(Candera::Vector2& v, FeatStd::Float val)
{
   v.SetX(val);
}


static Candera::Float GetWidth(const Candera::Vector2& v)
{
   return v.GetX();
}


static void SetY(Candera::Vector2& v, FeatStd::Float val)
{
   v.SetY(val);
}


static void SetMarginDefault(Node2D& /*node*/, Int16 /*marginValue*/, Int16 /*marginValueOrthogonal*/, bool /*isFirstItem*/, bool /*isLastItem*/)
{
}


static void SetHorizontalMargin(Node2D& node, Int16 marginValue, Int16 marginValueOrthogonal, bool isFirstItem, bool isLastItem)
{
   if (isFirstItem)
   {
      Margin margin = Layouter::GetMargin(node);
      Margin newMargin(marginValue, marginValueOrthogonal, margin.GetRight(), margin.GetBottom());
      if (!(margin == newMargin))
      {
         Layouter::SetMargin(node, newMargin);
      }
   }
   else if (isLastItem)
   {
      Margin margin = Layouter::GetMargin(node);
      Margin newMargin(margin.GetLeft(), margin.GetTop(), marginValue, marginValueOrthogonal);
      if (!(margin == newMargin))
      {
         Layouter::SetMargin(node, newMargin);
      }
   }
}


static void SetVerticalMargin(Node2D& node, Int16 marginValue, Int16 marginValueOrthogonal, bool isFirstItem, bool isLastItem)
{
   if (isFirstItem)
   {
      Margin margin = Layouter::GetMargin(node);
      Margin newMargin(marginValueOrthogonal, marginValue, margin.GetRight(), margin.GetBottom());
      if (!(margin == newMargin))
      {
         Layouter::SetMargin(node, newMargin);
      }
   }
   else if (isLastItem)
   {
      Margin margin = Layouter::GetMargin(node);
      Margin newMargin(margin.GetLeft(), margin.GetTop(), marginValueOrthogonal, marginValue);
      if (!(margin == newMargin))
      {
         Layouter::SetMargin(node, newMargin);
      }
   }
}


static void SetDefaultAlignment(Node2D&, Candera::ListAlignment::Enum)
{
}


static void SetVerticalAlignment(Node2D& node, Candera::ListAlignment::Enum alignment)
{
   Candera::VerticalAlignment realAlignment = Candera::_VerticalAlignment_LAST_ITEM;

   switch (alignment)
   {
      case Candera::ListAlignment::Begin:
         realAlignment = Candera::VTop;
         break;
      case Candera::ListAlignment::Center:
         realAlignment = Candera::VCenter;
         break;
      case Candera::ListAlignment::End:
         realAlignment = Candera::VBottom;
         break;
   }

   if (realAlignment != _VerticalAlignment_LAST_ITEM)
   {
      Candera::Layouter::SetVerticalAlignment(node, realAlignment);
   }
}


static void SetHorizontalAlignment(Node2D& node, Candera::ListAlignment::Enum alignment)
{
   Candera::HorizontalAlignment realAlignment = Candera::_HorizontalAlignment_LAST_ITEM;

   switch (alignment)
   {
      case Candera::ListAlignment::Begin:
         realAlignment = Candera::HLeft;
         break;
      case Candera::ListAlignment::Center:
         realAlignment = Candera::HCenter;
         break;
      case Candera::ListAlignment::End:
         realAlignment = Candera::HRight;
         break;
   }

   if (realAlignment != _HorizontalAlignment_LAST_ITEM)
   {
      Candera::Layouter::SetHorizontalAlignment(node, realAlignment);
   }
}


static FeatStd::UInt8 GetRowOrColumnDefault(const Candera::Node2D& /*node*/)
{
   return 0;
}


static FeatStd::UInt8 GetRowOrColumnHorizontal(const Candera::Node2D& node)
{
   return GridLayouter::GetColumn(node);
}


static FeatStd::UInt8 GetRowOrColumnVertical(const Candera::Node2D& node)
{
   return GridLayouter::GetRow(node);
}


static void SetRowOrColumnDefault(Candera::Node2D& /*node*/, FeatStd::UInt8 /*val*/)
{
}


static void SetRowOrColumnHorizontal(Candera::Node2D& node, FeatStd::UInt8 val)
{
   GridLayouter::SetColumn(node, val);
}


static void SetRowOrColumnVertical(Candera::Node2D& node, FeatStd::UInt8 val)
{
   GridLayouter::SetRow(node, val);
}


static void SetRowOrColumnSpanDefault(Candera::Node2D& /*node*/, FeatStd::UInt8 /*val*/)
{
}


static void SetRowOrColumnSpanHorizontal(Candera::Node2D& node, FeatStd::UInt8 val)
{
   GridLayouter::SetColumnSpan(node, val);
}


static void SetRowOrColumnSpanVertical(Candera::Node2D& node, FeatStd::UInt8 val)
{
   GridLayouter::SetRowSpan(node, val);
}


static void SetStructureSizeDefault(Candera::GridLayouter* /*layouter*/, FeatStd::Int8 /*index*/, FeatStd::Float /*val*/)
{
}


static void SetStructureSizeHorizontal(Candera::GridLayouter* layouter, FeatStd::Int8 index, FeatStd::Float val)
{
   if (layouter != 0)
   {
      layouter->SetColumnWidth(index, val);
   }
}


static void SetStructureSizeVertical(Candera::GridLayouter* layouter, FeatStd::Int8 index, FeatStd::Float val)
{
   if (layouter != 0)
   {
      layouter->SetRowHeight(index, val);
   }
}


OrientationOperator::OrientationOperator(Candera::ListScrollingOrientation orientation, bool margins) :
   _orientation(orientation),
   _marginSetter(SetMarginDefault),
   _itemSizeGetter(GetSizeDefault),
   _sizeSetter(SetSizeDefault),
   _alignmentSetter(SetDefaultAlignment),
   _rowOrColumnGetter(GetRowOrColumnDefault),
   _rowOrColumnSetter(SetRowOrColumnDefault),
   _rowOrColumnSpanSetter(SetRowOrColumnSpanDefault),
   _structureSIzeSetter(SetStructureSizeDefault)
{
   switch (orientation)
   {
      case Horizontal:
         _itemSizeGetter = GetWidth;
         _marginSetter = (margins) ? SetHorizontalMargin : SetMarginDefault;
         _sizeSetter = SetX;
         _alignmentSetter = SetHorizontalAlignment;
         _rowOrColumnGetter = GetRowOrColumnHorizontal;
         _rowOrColumnSetter = SetRowOrColumnHorizontal;
         _rowOrColumnSpanSetter = SetRowOrColumnSpanHorizontal;
         _structureSIzeSetter = SetStructureSizeHorizontal;
         break;

      case Vertical:
         _itemSizeGetter = GetHeight;
         _marginSetter = (margins) ? SetVerticalMargin : SetMarginDefault;
         _sizeSetter = SetY;
         _alignmentSetter = SetVerticalAlignment;
         _rowOrColumnGetter = GetRowOrColumnVertical;
         _rowOrColumnSetter = SetRowOrColumnVertical;
         _rowOrColumnSpanSetter = SetRowOrColumnSpanVertical;
         _structureSIzeSetter = SetStructureSizeVertical;
         break;

      default:
         _itemSizeGetter = GetSizeDefault;
         _marginSetter = SetMarginDefault;
         _sizeSetter = SetSizeDefault;
         _alignmentSetter = SetDefaultAlignment;
         _rowOrColumnGetter = GetRowOrColumnDefault;
         _rowOrColumnSetter = SetRowOrColumnDefault;
         _rowOrColumnSpanSetter = SetRowOrColumnSpanDefault;
         _structureSIzeSetter = SetStructureSizeDefault;
         break;
   }
}


void OrientationOperator::SetMargin(Candera::Node2D& node, Candera::Int16 marginValue, Candera::Int16 marginValueOrthogonal, bool isFirstItem, bool isLastItem) const
{
   _marginSetter(node, marginValue, marginValueOrthogonal, isFirstItem, isLastItem);
}


FeatStd::Float OrientationOperator::GetVectorComponent(const Candera::Vector2& v) const
{
   return _itemSizeGetter(v);
}


void OrientationOperator::SetVectorComponent(Candera::Vector2& v, FeatStd::Float val) const
{
   _sizeSetter(v, val);
}


void OrientationOperator::SetAlignment(Candera::Node2D& node, Candera::ListAlignment::Enum alignment)
{
   _alignmentSetter(node, alignment);
}


FeatStd::UInt8 OrientationOperator::GetRowOrColumn(const Candera::Node2D& node) const
{
   return _rowOrColumnGetter(node);
}


void OrientationOperator::SetRowOrColumn(Candera::Node2D& node, FeatStd::UInt8 val) const
{
   _rowOrColumnSetter(node, val);
}


void OrientationOperator::SetRowOrColumnSpan(Candera::Node2D& node, FeatStd::UInt8 val) const
{
   _rowOrColumnSpanSetter(node, val);
}


void OrientationOperator::SetStructureSize(Candera::GridLayouter* layouter, FeatStd::Int8 index, FeatStd::Float val) const
{
   _structureSIzeSetter(layouter, index, val);
}
