/* ***************************************************************************************
* FILE:          ItemAdder.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ItemAdder is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef List_Item_Adder_H
#define List_Item_Adder_H


#include <map>
#include <Candera/Engine2D/Core/Node2D.h>
#include "Widgets/2D/Common/FlexScrollable.h"
#include "Widgets/2D/List/Content/ContentProvider.h"
#include <Widgets/2D/List/Content/Adders/OrientationOperator.h>
#include <Widgets/2D/List/Content/ISizeContainer.h>
#include <Widgets/2D/List/Content/ITemplateSpanAnalyzer.h>
#include <Widgets/2D/List/Swiping/Normalizer.h>
#include "Courier/Util/Vector.h"

/**
* @brief The class ItemAdder is an abstract class for adding items to a list
*/
class ItemAdder
{
   public:
      FEATSTD_TYPEDEF_SHARED_POINTER(ItemAdder);

      CANDERA_RTTI_DECLARATION(ItemAdder);

      virtual ~ItemAdder();

      /**
       * Method that needs to be called before adding any element.
       * @return the adjusted position according to the previous content
       */
      FeatStd::Float BeginAdding(const Candera::Vector2& visibleArea, Candera::Float originalPosition);

      /**
      * Method that needs to be called before reverse adding any element.
      */
      void BeginReverseAdding();

      /**
       * Method that needs to be called when direct adding was finished
       */
      void FinishDirectAdding();

      /**
      * Method that needs to be called when reverse adding was finished
      */
      void FinishReverseAdding();

      /**
      * Method that needs to be called when all elements were added
      */
      void FinishAdding();

      /**
      * @return the node item size (width or height depending on the scrolling orientation)
      */
      Candera::Float GetItemSize(Candera::Node2D& node);

      /**
      * @return the current number of complete visible items from the list
      */
      Candera::UInt32 GetNumberOfCompleteVisibleItems() const;

      /**
       * @return the number of items in the list(completely and partially visible)
       */
      const Candera::Int32 GetNumberOfAddedItems() const;

      /**
       * @return the size in pixels of all the elements that are added, without their margins
       */
      Candera::Float GetSizeOfAddedElements() const;

      /**
       * Adds the node received as parameter to the list (the margin may be set in order to make an item partial visible)
       * @return true if items can be added until the visible area is filled; false if the visible area is fully occupied
       *
       */
      virtual bool AddItem(Candera::Node2D& templateNode, Candera::Node2D& childNode) = 0;

      /**
      * Resets the fields before the items addition is started, a new update cycle is started
      */
      virtual void Clear();

      /**
      * @return the scrolling increment (how many items to scroll at a time)
      */
      virtual Candera::UInt32 GetScrollingIncrement() const = 0;

      virtual Candera::Int32 GetNumberOfItemsInFirstStructure() const = 0;

      virtual Candera::Int32 GetNumberOfStructures() const = 0;

      /**
      * Computes the size after the childNode would be added without adding it.
      * @return The current size hint after the child node would be added.
      *
      */
      Candera::Float GetPostAddSizeHint(Candera::Node2D& childNode, const Candera::Vector2& visibleAreaInPixels);
      Candera::Int16 GetFirstItemMargin() const;

      Candera::Int16 GetLastItemMargin() const;

      /**
       * Sets whether items aree added at the end(@param reverse == false) or at the begining(@param reverse == true)
       */
      void SetReverseMode(bool reverse);

      Candera::Float GetBeginingSize() const;
      Candera::Float GetEndingSize() const;

      virtual Candera::Int16 GetBeginingMargin() const;
      virtual Candera::Int16 GetEndingMargin() const;

      Candera::Float GetCurrentSize() const
      {
         return _currentSize;
      }

      void SetNormalizer(Normalizer* val)
      {
         _normalizer = val;
      }
      const Courier::Vector<FeatStd::Float>& GetSizes() const
      {
         return _sizes;
      }

      FeatStd::Int32 GetZeroSizedItems() const
      {
         return _zeroSizedItems;
      }

      void SetStructureOffset(FeatStd::Int32 val)
      {
         _structureOffset = val;
      }
   protected:

      ItemAdder(ISizeContainer& sizesContainer, Candera::Node2D* groupNode, const OrientationOperator& directOrientationOperator, const OrientationOperator& orthogonalOrientationOperator);

      FeatStd::Float InsertItem(Candera::Node2D& childNode);
      Candera::Layouter* GetGroupNodeLayouter() const;

      Candera::Float GetOriginalPosition() const;
      Candera::Float GetAdjustedPosition() const;
      Candera::Vector2 GetVisibleArea() const;
      Candera::Int32 GetAddedItems() const;
      Candera::Node2D* GetGroupNode() const;

      Candera::UInt32 _numberOfCompleteVisibleItems;
      Candera::Int16 _beginingMargin;
      Candera::Int16 _endingMargin;
      const OrientationOperator& _directOrientationOperator;
      const OrientationOperator& _orthogonalOrientationOperator;
      ISizeContainer& _sizesContainer;
      Candera::Float _beginingSize;
      Candera::Float _endingSize;
      Candera::Int32 _addedItems;
      Candera::Int32 _zeroSizedItems;
      FeatStd::Int32 _structureOffset;
      bool _reverse;

      void UpdateBeginingMargin(const Courier::Vector<FeatStd::Float>& sizes);
      void CollectSize(FeatStd::Float size);
      void ResetSize();
      void IncreaseSize(FeatStd::Float val);

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(ItemAdder);
      FEATSTD_SHARED_POINTER_DECLARATION();

      Candera::Float _originalPosition;
      Candera::Float _adjustedPosition;
      Candera::Vector2 _visibleArea;
      Candera::Node2D* _groupNode;
      Normalizer* _normalizer;
      Courier::Vector<FeatStd::Float> _sizes;
      Candera::Float _currentSize;    ///< Size of the visible elements.
      Candera::Float _actualSize;    ///< complete size of the added structures

      /**
       * @return the correction needed to be applied to the original position
       */
      virtual Candera::Float GetPositionCorrection(const FeatStd::Float /*position*/) const
      {
         return 0;
      }

      /**
      * Method called before the adding begins. It can be overridden by subclasses to ad specific behavior
      */
      virtual void OnBeforeBeginAdding() { }

      /**
       * Method called when the adding begins. It can be overridden by subclasses to ad specific behavior
       */
      virtual void OnBeginAdding() { }

      /**
      * Method called when the reverse adding begins. It can be overridden by subclasses to ad specific behavior
      */
      virtual void OnBeginReverseAdding() { }

      /**
       * Method called when the adding finished. It can be overridden by subclasses to ad specific behavior
       */
      virtual void OnFinishAdding() { }

      /**
      * Method called when direct adding was finished. It can be overridden by subclasses to ad specific behavior
      */
      virtual void OnFinishDirectAdding() { }

      /**
       * Method called when reverse adding was finished. It can be overridden by subclasses to ad specific behavior
       */
      virtual void OnFinishReverseAdding() { }

      Candera::Int16 ComputeStartMargin(const Courier::Vector<FeatStd::Float>& sizes, Candera::Float position) const;
};


#endif
