/* ***************************************************************************************
* FILE:          GridPositionsCalculator.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  GridPositionsCalculator is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "GridPositionsCalculator.h"

using namespace Candera;


GridPositionsCalculator::GridPositionsCalculator(const GridItemAdder::SharedPointer& adder, const ITemplateSpanAnalyzer::SharedPointer& templateSpanAnalyzer, const PositionsCache& positionsCache, const OrientationOperator& orthogonalOrientationOperator) :
   _adder(adder),
   _templateSpanAnalyzer(templateSpanAnalyzer),
   _positionsCache(positionsCache),
   _orthogonalOrientationOperator(orthogonalOrientationOperator)
{
}


GridPositionsCalculator::~GridPositionsCalculator()
{
}


class StructureIndexListener : public GridItemAdderListener
{
   public:
      StructureIndexListener(const GridItemAdder& adder) :
         _adder(adder),
         _structureIndex(0)
      { }

      virtual void OnItemAdded(FeatStd::Int32 /*column*/, FeatStd::Int32 /*row*/, FeatStd::Int32 /*directSpan*/, FeatStd::Int32 /*orthogonalSpan*/) override
      {
         _structureIndex = _adder.GetCurrentIndex();
      }

      FeatStd::Int32 GetStructureIndex() const
      {
         return _structureIndex;
      }

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(StructureIndexListener);

      const GridItemAdder& _adder;
      FeatStd::Int32 _structureIndex;
};


FeatStd::Float GridPositionsCalculator::CalculateStartingPosition(const ITemplateRetriever& templateRetriever, const ContentProvider& contentProvider, const Candera::UInt32 startIndex) const
{
   Float startingPos(0.0F);

   if (!_adder.PointsToNull())
   {
      StructureIndexListener structureIndexRetriever(*_adder);
      _adder->SetListener(&structureIndexRetriever);

      Candera::Node2D& invalidTemplate(*templateRetriever.GetTemplate(tSharedPtrListDataItem(0)));

      const PositionsCache::CacheData& cachedData(AdjustStartingPositionsByStartIndex(invalidTemplate, contentProvider, startIndex));
      UInt32 i(cachedData.GetFirstIndex());
      Int32 initialStructureIndex(cachedData.GetStructureIndex());
      bool invalidItems = false;

      _adder->BeginAdding(Vector2(Math::MaxFloat(), Math::MaxFloat()), 0.0F);

      for (; i <= startIndex; ++i)
      {
         Int32 dataIndex = Int32(i - contentProvider.GetStartIndex());
         bool validDataIndex = ((dataIndex >= 0) && (dataIndex < Int32(contentProvider.GetWindowElementSize())));

         invalidItems = invalidItems || (!validDataIndex);

         Node2D* item = &invalidTemplate;

         if (validDataIndex)
         {
            const tSharedPtrListDataItem& dataItem(contentProvider.GetItem(dataIndex));
            item = templateRetriever.GetTemplate(dataItem);
         }

         if (item != 0)
         {
            _adder->AddItem(*item, *item);
         }
      }

      Int32 adderStructureIndex(structureIndexRetriever.GetStructureIndex());

      startingPos = Float(initialStructureIndex + adderStructureIndex);
   }
   return startingPos;
}


FeatStd::Int32 GridPositionsCalculator::CalculateStartIndex(const ITemplateRetriever& templateRetriever, const ContentProvider& contentProvider, const Candera::Float position) const
{
   Int32 startIndex(0);
   Candera::Node2D* invalidTemplate(templateRetriever.GetTemplate(tSharedPtrListDataItem(0)));

   if ((!_adder.PointsToNull()) && (0 != invalidTemplate))
   {
      StructureIndexListener structureIndexRetriever(*_adder);
      _adder->SetListener(&structureIndexRetriever);

      const PositionsCache::CacheData& cachedData(AdjustStartingPositionsByStructureIndex(*invalidTemplate, contentProvider, UInt32(position)));

      Int32 i(cachedData.GetFirstIndex());
      const Int32 initialStructureIndex(cachedData.GetStructureIndex());
      bool invalidItems = false;

      const Candera::Float correctedPosition(_adder->BeginAdding(Vector2(Math::MaxFloat(), Math::MaxFloat()), position));
      FEATSTD_UNUSED(correctedPosition);

      Int32 structureIndex(initialStructureIndex + _adder->GetCurrentIndex());
      startIndex = i;
      bool found(false);

      for (; (!found) && (structureIndex <= static_cast<Int32>(position)); ++i)
      {
         Int32 dataIndex = i - static_cast<Int32>(contentProvider.GetStartIndex());
         bool validDataIndex = ((dataIndex >= 0) && (dataIndex < static_cast<Int32>(contentProvider.GetWindowElementSize())));

         invalidItems = invalidItems || (!validDataIndex);

         Node2D* item = invalidTemplate;

         if (validDataIndex)
         {
            const tSharedPtrListDataItem& dataItem = contentProvider.GetItem(dataIndex);
            item = templateRetriever.GetTemplate(dataItem);
         }

         _adder->AddItem(*item, *item);
         structureIndex = initialStructureIndex + structureIndexRetriever.GetStructureIndex();

         if (_adder->IsFirstInStructure())
         {
            found = (structureIndex == static_cast<Int32>(position));
            startIndex = i;
         }
      }
   }

   return startIndex;
}


PositionsCache::CacheData GridPositionsCalculator::AdjustStartingPositionsByStartIndex(Candera::Node2D& invalidTemplate, const ContentProvider& contentProvider, const Candera::UInt32 startIndex) const
{
   PositionsCache::CacheData cachedData = _positionsCache.FindByStartIndex(startIndex);
   if ((!_adder.PointsToNull()) && (!_templateSpanAnalyzer.PointsToNull()))
   {
      const FeatStd::Int32 firstIndex = cachedData.GetFirstIndex();
      const Int32 lastAvailableIndex = contentProvider.GetStartIndex() + contentProvider.GetWindowElementSize();

      if ((firstIndex < Int32(contentProvider.GetStartIndex())) || (firstIndex > lastAvailableIndex) ||
            (Int32(startIndex) < Int32(contentProvider.GetStartIndex())) || (Int32(startIndex) > lastAvailableIndex))
      {
         const Int32 span(static_cast<Int32>(_orthogonalOrientationOperator.GetVectorComponent(_templateSpanAnalyzer->GetSpan(&invalidTemplate))));
         const Int32 maxOrthogonalItemsCount(static_cast<Int32>(_adder->GetOrthogonalItems()));

         if ((0 < span) && (span <= maxOrthogonalItemsCount))
         {
            const Int32 itemsPerStructure = maxOrthogonalItemsCount / span;

            const Int32 numberOfStructures = (Int32(startIndex) - firstIndex) / itemsPerStructure;
            const Int32 numberOfItems = numberOfStructures * itemsPerStructure;

            cachedData = PositionsCache::CacheData(cachedData.GetStructureIndex() + numberOfStructures, cachedData.GetFirstIndex() + numberOfItems);
         }
      }
   }
   return cachedData;
}


PositionsCache::CacheData GridPositionsCalculator::AdjustStartingPositionsByStructureIndex(Candera::Node2D& invalidTemplate, const ContentProvider& contentProvider, const Candera::UInt32 structureIndex) const
{
   PositionsCache::CacheData cachedData = _positionsCache.FindByStructureIndex(structureIndex);

   if ((!_adder.PointsToNull()) && (!_templateSpanAnalyzer.PointsToNull()))
   {
      const FeatStd::Int32 firstIndex = cachedData.GetFirstIndex();
      const Int32 lastAvailableIndex = contentProvider.GetStartIndex() + contentProvider.GetWindowElementSize();

      const Int32 span(static_cast<Int32>(_orthogonalOrientationOperator.GetVectorComponent(_templateSpanAnalyzer->GetSpan(&invalidTemplate))));
      const Int32 maxOrthogonalItemsCount(static_cast<Int32>(_adder->GetOrthogonalItems()));

      if ((0 < span) && (span <= maxOrthogonalItemsCount))
      {
         const Int32 itemsPerStructure = maxOrthogonalItemsCount / span;

         if (Int32(firstIndex) < Int32(contentProvider.GetStartIndex()))
         {
            const Int32 numberOfItems = Int32(contentProvider.GetStartIndex()) - Int32(firstIndex);
            const Int32 numberOfStructures = numberOfItems / itemsPerStructure;

            cachedData = PositionsCache::CacheData(cachedData.GetStructureIndex() + numberOfStructures, cachedData.GetFirstIndex() + numberOfStructures * itemsPerStructure);
         }
         else if (Int32(firstIndex) > lastAvailableIndex)
         {
            const Int32 numberOfStructures = structureIndex - cachedData.GetStructureIndex();
            const Int32 numberOfItems = numberOfStructures * itemsPerStructure;

            cachedData = PositionsCache::CacheData(structureIndex, cachedData.GetFirstIndex() + numberOfItems);
         }
         else
         {
            // do nothing
         }
      }
   }
   return cachedData;
}
