/* ***************************************************************************************
* FILE:          GridItemAdder.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  GridItemAdder is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#if !defined(GridItemAdder_H)
#define GridItemAdder_H

#include "ItemAdder.h"
#include <Courier/Util/Vector.h>
#include <Widgets/2D/List/Content/GridOccupation.h>
#include "PositionsCache.h"


class GridItemAdderListener
{
   public:
      virtual ~GridItemAdderListener() { }

      virtual void OnItemAdded(FeatStd::Int32 column, FeatStd::Int32 row, FeatStd::Int32 directSpan, FeatStd::Int32 orthogonalSpan) = 0;
};


/**
* @brief The class GridItemAdder is an abstract class, derived from the ItemAdder which contains the common functionality used by a list with Grid Layouter
*/
class GridItemAdder : public ItemAdder
{
      FEATSTD_TYPEDEF_BASE(ItemAdder);
   public:
      FEATSTD_TYPEDEF_SHARED_POINTER(GridItemAdder);

      CANDERA_RTTI_DECLARATION(GridItemAdder);

      static GridItemAdder::SharedPointer Create(const ITemplateSpanAnalyzer::SharedPointer& templateSpanAnalyzer, ISizeContainer& sizesContainer, Candera::Node2D* groupNode, const OrientationOperator& directOrientationOperator, const OrientationOperator& orthogonalOrientationOperator, Candera::UInt32 rows, Candera::UInt32 columns);

      virtual ~GridItemAdder();

      virtual bool AddItem(Candera::Node2D& templateNode, Candera::Node2D& childNode);

      virtual void Clear();

      virtual Candera::Int32 GetNumberOfItemsInFirstStructure() const;

      virtual Candera::Int32 GetNumberOfStructures() const;

      virtual Candera::Int16 GetBeginingMargin() const;

      Candera::Int32 GetCurrentIndex() const;

      /**
       * @return true if the last added item is the first in its structure; false otherwise
       */
      bool IsFirstInStructure() const;

      /**
       * @return true if the last added item is the last in its structure; false otherwise
       */
      bool IsLastInStructure() const;

      Candera::UInt32 GetOrthogonalItems() const;

      void SetListener(GridItemAdderListener* listener);
      void SetGridOccupation(const GridOccupation* val);

      virtual Candera::UInt32 GetScrollingIncrement() const;

   protected:
      GridItemAdder(const ITemplateSpanAnalyzer::SharedPointer& templateSpanAnalyzer, ISizeContainer& sizesContainer, Candera::Node2D* groupNode, const OrientationOperator& directOrientationOperator, const OrientationOperator& orthogonalOrientationOperator, Candera::UInt32 rows, Candera::UInt32 columns);

      Candera::Int32 GetRows() const;
      Candera::Int32 GetColumns() const;
      Candera::Int32 GetCurrentOrthogonalIndex() const;
      Candera::Int32 GetAbsoluteStructureIndex() const;

      Candera::Int32 GetMaxValue(Candera::Int32 val1, Candera::Int32 val2) const;
   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(GridItemAdder);

      GridItemAdderListener* _listener;
      Candera::Int32 _computedRows;
      Candera::Int32 _computedColumns;
      Candera::Int32 _rows;
      Candera::Int32 _columns;
      Candera::Int32 _currentOrthogonalIndex;
      FeatStd::Int16 _completeBeginingMargin;
      Candera::Int32 _itemsInStructure;
      Candera::Int32 _absoluteStructureIndex; // index starting with 0
      Candera::Int32 _logicalStructureIndex;  // index relative to the first visible structure; invisible structures have negative indices

      Candera::Float _maxItemCellSize;
      Candera::Float _maxRemainingItemSize;
      Candera::Int32 _itemsInFirstStructure;
      bool _firstInStructure;
      bool _lastInStructure;
      const GridOccupation* _gridOccupation;
      GridOccupation _currentOccupation;
      ITemplateSpanAnalyzer::SharedPointer _templateSpanAnalyzer;

      void AdjustNumberOfCompleteVisibleItems();

      Candera::Float GetItemOrthogonalSize(Candera::Node2D& childNode);

      Candera::UInt32 GetItemsCount() const;
      Candera::UInt32 GetOrthogonalItemsCount() const;

      Candera::Int32 GetCurrentRowIndex() const;
      Candera::Int32 GetCurrentColumnIndex() const;

      void SetDirectSpan(Candera::Node2D& childNode, FeatStd::UInt32 span) const;
      void SetOrthogonalSpan(Candera::Node2D& childNode, FeatStd::UInt32 span) const;

      void EndStructure();

      void PreAdd(const FeatStd::Int32 directSpan, const FeatStd::Int32 orthogonalSpan, const FeatStd::Int32 maxOrthogonalItemsCount, const FeatStd::Float itemSize);
      void Add(Candera::Node2D& childNode, FeatStd::Int32 directSpan, FeatStd::Int32 orthogonalSpan);
      void OccupyForAddedItem(const FeatStd::Int32 xSpan, const FeatStd::Int32 ySpan);
      void PostAdd(const FeatStd::Int32 directSpan, const FeatStd::Int32 orthogonalSpan, const FeatStd::Int32 maxOrthogonalItemsCount, const FeatStd::Float itemSize);

      void CalculateMaxStructureSize(const FeatStd::Float itemSize, const FeatStd::Int32 directSpan);
      void AdvanceUntilFree(const FeatStd::Int32 directSpan, const FeatStd::Int32 maxOrthogonalItemsCount, const FeatStd::Float itemSize);

      virtual Candera::Float GetPositionCorrection(const FeatStd::Float position) const;

      virtual void OnBeginAdding();
      virtual void OnBeginReverseAdding();
      virtual void OnFinishAdding();
      virtual void OnFinishReverseAdding();

      void NotifyOnItemAdded(FeatStd::Int32 column, FeatStd::Int32 row, FeatStd::Int32 directSpan, FeatStd::Int32 orthogonalSpan) const;

      void ShiftGridIndex(Candera::Node2D& node, FeatStd::UInt8 val) const;
      FeatStd::UInt8 GetShiftValue(const Candera::Node2D& node) const;
      FeatStd::UInt8 GetMaxShiftValue();
      FeatStd::Float GetItemsInPartiallyVisibleStructures() const;
};


#endif
