/* ***************************************************************************************
* FILE:          GridItemAdder.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  GridItemAdder is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "Widgets/2D/List/Layouting/FlexListLayouter.h"


#include "widget2D_std_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/GridItemAdder.cpp.trc.h"
#endif

#include "GridItemAdder.h"

using namespace Candera;

CANDERA_RTTI_DEFINITION(GridItemAdder);


// GridItemAdder
GridItemAdder::GridItemAdder(const ITemplateSpanAnalyzer::SharedPointer& templateSpanAnalyzer, ISizeContainer& sizesContainer, Candera::Node2D* groupNode, const OrientationOperator& directOrientationOperator, const OrientationOperator& orthogonalOrientationOperator, Candera::UInt32 rows, Candera::UInt32 columns) :
   Base(sizesContainer, groupNode, directOrientationOperator, orthogonalOrientationOperator),
   _listener(0),
   _rows(rows),
   _columns(columns),
   _currentOrthogonalIndex(-1),
   _itemsInStructure(0),
   _absoluteStructureIndex(0),
   _logicalStructureIndex(0),
   _computedRows(0),
   _computedColumns(0),
   _maxItemCellSize(0.0),
   _itemsInFirstStructure(0),
   _gridOccupation(0),
   _firstInStructure(false),
   _lastInStructure(false),
   _completeBeginingMargin(0),
   _templateSpanAnalyzer(templateSpanAnalyzer),
   _maxRemainingItemSize(0.0F)
{
   if (_rows <= 0)
   {
      _rows = 1;
   }

   if (_columns <= 0)
   {
      _columns = 1;
   }
}


Candera::Int32 GridItemAdder::GetRows() const
{
   return _rows;
}


Candera::Int32 GridItemAdder::GetColumns() const
{
   return _columns;
}


Candera::Int32 GridItemAdder::GetCurrentOrthogonalIndex() const
{
   return _currentOrthogonalIndex;
}


Candera::Int32 GridItemAdder::GetAbsoluteStructureIndex() const
{
   return _absoluteStructureIndex;
}


GridItemAdder::SharedPointer GridItemAdder::Create(const ITemplateSpanAnalyzer::SharedPointer& templateSpanAnalyzer, ISizeContainer& sizesContainer, Candera::Node2D* groupNode, const OrientationOperator& directOrientationOperator, const OrientationOperator& orthogonalOrientationOperator, Candera::UInt32 rows, Candera::UInt32 columns)
{
   return GridItemAdder::SharedPointer(CANDERA_NEW(GridItemAdder)(templateSpanAnalyzer, sizesContainer, groupNode, directOrientationOperator, orthogonalOrientationOperator, rows, columns));
}


GridItemAdder::~GridItemAdder()
{
   _listener = 0;
   _gridOccupation = 0;
}


void GridItemAdder::OnBeginAdding()
{
   _firstInStructure = true;
   _lastInStructure = false;
   _logicalStructureIndex = static_cast<Int32>(GetAdjustedPosition() - GetOriginalPosition());

   Vector2 start(0.0F, 0.0F);
   _directOrientationOperator.SetVectorComponent(start, GetAdjustedPosition());
   _currentOccupation.Clear();
}


void GridItemAdder::OnBeginReverseAdding()
{
   if (0 != GetGroupNode())
   {
      _currentOrthogonalIndex = GetOrthogonalItemsCount() - 1;
      _absoluteStructureIndex = -1;
   }
}


bool GridItemAdder::AddItem(Candera::Node2D& templateNode, Candera::Node2D& childNode)
{
   bool fits(false);

   if (!_templateSpanAnalyzer.PointsToNull())
   {
      const Float itemSize(InsertItem(childNode));
      const Int32 maxOrthogonalItemsCount(static_cast<Int32>(GetOrthogonalItemsCount()));

      const Candera::Vector2 completeSpan(_templateSpanAnalyzer->GetSpan(&templateNode));
      const Int32 xSpan(static_cast<Int32>(completeSpan.GetX()));
      const Int32 ySpan(static_cast<Int32>(completeSpan.GetY()));
      const Int32 directSpan(_directOrientationOperator.GetValue(xSpan, ySpan));
      const Int32 orthogonalSpan(_orthogonalOrientationOperator.GetValue(xSpan, ySpan));

      PreAdd(directSpan, orthogonalSpan, maxOrthogonalItemsCount, itemSize);
      Add(childNode, directSpan, orthogonalSpan);
      if (!_reverse)
      {
         OccupyForAddedItem(xSpan, ySpan);
      }
      PostAdd(directSpan, orthogonalSpan, maxOrthogonalItemsCount, itemSize);

      const Float visibleSize(_directOrientationOperator.GetVectorComponent(GetVisibleArea()));
      fits = GetCurrentSize() < visibleSize;
   }

   return fits;
}


void GridItemAdder::OccupyForAddedItem(const Int32 xSpan, const Int32 ySpan)
{
   const Candera::Int32 currentColumnIndex(GetCurrentColumnIndex());
   const Candera::Int32 currentRowIndex(GetCurrentRowIndex());

   _currentOccupation.Occupy(currentColumnIndex, currentRowIndex, xSpan, ySpan);
   NotifyOnItemAdded(currentColumnIndex, currentRowIndex, xSpan, ySpan);
}


void GridItemAdder::OnFinishAdding()
{
   EndStructure();

   AdjustNumberOfCompleteVisibleItems();
}


void GridItemAdder::OnFinishReverseAdding()
{
   if (_reverse)
   {
      Node2D* groupNode = GetGroupNode();
      if (0 != groupNode)
      {
         FeatStd::UInt8 maxShiftValue = GetMaxShiftValue();
         Node2D* childNode = groupNode->GetFirstChild();
         while (0 != childNode)
         {
            ShiftGridIndex(*childNode, maxShiftValue);
            childNode = childNode->GetNextSibling();
         }
      }
   }
}


void GridItemAdder::NotifyOnItemAdded(FeatStd::Int32 column, FeatStd::Int32 row, FeatStd::Int32 directSpan, FeatStd::Int32 orthogonalSpan) const
{
   if (0 != _listener)
   {
      _listener->OnItemAdded(column, row, directSpan, orthogonalSpan);
   }
}


void GridItemAdder::ShiftGridIndex(Candera::Node2D& node, FeatStd::UInt8 val) const
{
   val += _directOrientationOperator.GetRowOrColumn(node);
   _directOrientationOperator.SetRowOrColumn(node, val);
}


FeatStd::UInt8 GridItemAdder::GetShiftValue(const Candera::Node2D& node) const
{
   return _directOrientationOperator.GetRowOrColumn(node);
}


void GridItemAdder::Clear()
{
   _currentOrthogonalIndex = 0;
   _itemsInStructure = 0;
   _absoluteStructureIndex = 0;
   _computedRows = -1;
   _computedColumns = -1;
   _maxItemCellSize = 0.0;
   _itemsInFirstStructure = 0;
   _completeBeginingMargin = 0;
   _maxRemainingItemSize = 0.0;

   Base::Clear();
}


Candera::Int32 GridItemAdder::GetNumberOfItemsInFirstStructure() const
{
   return _itemsInFirstStructure;
}


Candera::Int32 GridItemAdder::GetNumberOfStructures() const
{
   return Math::Maximum(_directOrientationOperator.GetValue(_computedColumns, _computedRows), (FeatStd::Int32) 0);
}


Candera::Int16 GridItemAdder::GetBeginingMargin() const
{
   return _completeBeginingMargin;
}


Candera::Int32 GridItemAdder::GetCurrentIndex() const
{
   return _absoluteStructureIndex;
}


bool GridItemAdder::IsFirstInStructure() const
{
   return _firstInStructure;
}


bool GridItemAdder::IsLastInStructure() const
{
   return _lastInStructure;
}


Candera::UInt32 GridItemAdder::GetOrthogonalItems() const
{
   return GetOrthogonalItemsCount();
}


void GridItemAdder::SetListener(GridItemAdderListener* listener)
{
   _listener = listener;
}


void GridItemAdder::SetGridOccupation(const GridOccupation* val)
{
   _gridOccupation = val;
}


Candera::UInt32 GridItemAdder::GetScrollingIncrement() const
{
   return _directOrientationOperator.GetValue(GetRows(), GetColumns());
}


Candera::Int32 GridItemAdder::GetMaxValue(Candera::Int32 val1, Candera::Int32 val2) const
{
   return (val1 > val2) ? val1 : val2;
}


void GridItemAdder::PreAdd(const Int32 directSpan, const Int32 orthogonalSpan, const Int32 maxOrthogonalItemsCount, const Float itemSize)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GridItemAdder::PreAdd _currentOrthogonalIndex=%d, directSpan=%d, orthogonalSpan=%d, maxOrthogonalItemsCount=%d, itemSize=%f",
                       _currentOrthogonalIndex, directSpan, orthogonalSpan, maxOrthogonalItemsCount, itemSize));
   _lastInStructure = false;

   if ((_currentOrthogonalIndex != 0) && ((_currentOrthogonalIndex + orthogonalSpan) > Int32(maxOrthogonalItemsCount)))
   {
      EndStructure();
   }

   CalculateMaxStructureSize(itemSize, directSpan);
   if (!_reverse)
   {
      AdvanceUntilFree(directSpan, maxOrthogonalItemsCount, itemSize);
   }
   CalculateMaxStructureSize(itemSize, directSpan);

   if (itemSize > _maxRemainingItemSize)
   {
      _maxRemainingItemSize = itemSize;
   }
}


void GridItemAdder::CalculateMaxStructureSize(const Float itemSize, const Int32 directSpan)
{
   Float size(itemSize);
   if (directSpan > 1)
   {
      size = _directOrientationOperator.GetVectorComponent(_templateSpanAnalyzer->GetSpanUnitSize());
   }

   if (_maxItemCellSize < size)
   {
      _maxItemCellSize = size;
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GridItemAdder::CalculateMaxStructureSize _maxItemCellSize=%f", _maxItemCellSize));
}


void GridItemAdder::AdvanceUntilFree(const FeatStd::Int32 directSpan, const FeatStd::Int32 maxOrthogonalItemsCount, const FeatStd::Float itemSize)
{
   Int32 rowIndex(GetCurrentRowIndex());
   Int32 columnIndex(GetCurrentColumnIndex());

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GridItemAdder::AdvanceUntilFree from row=%d, column=%d", rowIndex, columnIndex));

   GridOccupation::Coordinates previousItemStart(-1, -1);

   while (_currentOccupation.IsOccupied(columnIndex, rowIndex))
   {
      ++_currentOrthogonalIndex;

      GridOccupation::Coordinates currentItemStart(_currentOccupation.GetItemStart(columnIndex, rowIndex));
      if (previousItemStart != currentItemStart)
      {
         ++_itemsInStructure;
         previousItemStart = currentItemStart;
      }
      if (_currentOrthogonalIndex >= maxOrthogonalItemsCount)
      {
         CalculateMaxStructureSize(itemSize, directSpan);
         EndStructure();
      }

      rowIndex = GetCurrentRowIndex();
      columnIndex = GetCurrentColumnIndex();
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GridItemAdder::AdvanceUntilFree to row=%d, column=%d", rowIndex, columnIndex));
}


Candera::Float GridItemAdder::GetPositionCorrection(const FeatStd::Float position) const
{
   Int32 correction(0);

   if (0 != _gridOccupation)
   {
      const GridOccupation::Coordinates startAbs(_gridOccupation->GetAbsoluteStart());
      const Int32 col(static_cast<Int32>(position));
      const Int32 directIndex(_directOrientationOperator.GetValue(startAbs._x, startAbs._y));
      const Int32 directIndexRel(col - directIndex);
      const Candera::Int32 orthogonalItemsCount(static_cast<Int32>(GetOrthogonalItemsCount()));

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GridItemAdder::GetPositionCorrection to directIndex=%d, col=%d, orthogonalItemsCount=%d", directIndex, col, orthogonalItemsCount));

      Int32 minStart(directIndexRel);
      bool ok(true);

      do
      {
         ok = true;
         for (Int32 orthogonalIndexRel(0); orthogonalIndexRel < orthogonalItemsCount; ++orthogonalIndexRel)
         {
            const Int32 colIndex(_directOrientationOperator.GetValue(minStart, orthogonalIndexRel));
            const Int32 rowIndex(_orthogonalOrientationOperator.GetValue(minStart, orthogonalIndexRel));

            const GridOccupation::Coordinates itemStartRel(_gridOccupation->GetItemStart(colIndex, rowIndex));
            const Int32 startIndexRel(_directOrientationOperator.GetValue(itemStartRel._x, itemStartRel._y));

            if ((startIndexRel >= 0) && (startIndexRel < minStart))
            {
               minStart = startIndexRel;
               ok = false;
            }
         }
      }
      while (!ok);

      correction = minStart - directIndexRel;

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GridItemAdder::GetPositionCorrection to correction=%d", correction));
   }
   return static_cast<Float>(correction);
}


void GridItemAdder::Add(Candera::Node2D& childNode, FeatStd::Int32 directSpan, FeatStd::Int32 orthogonalSpan)
{
   Vector2 structureOffsetCorrection(0, 0);
   _directOrientationOperator.SetVectorComponent(structureOffsetCorrection, _structureOffset);

   const Int32 rowIndex(GetCurrentRowIndex() + structureOffsetCorrection.GetY());
   const Int32 columnIndex(GetCurrentColumnIndex() + structureOffsetCorrection.GetX());

   const Int32 rowSpan(_directOrientationOperator.GetValue(orthogonalSpan, directSpan));
   const Int32 columnSpan(_directOrientationOperator.GetValue(directSpan, orthogonalSpan));
   GridLayouter::SetRow(childNode, UInt8(rowIndex));
   GridLayouter::SetColumn(childNode, UInt8(columnIndex));

   SetDirectSpan(childNode, directSpan);
   SetOrthogonalSpan(childNode, orthogonalSpan);

   Vector2 correction(0, 0);
   if (_reverse)
   {
      _directOrientationOperator.SetVectorComponent(correction, -1);
   }

   _computedRows = GetMaxValue(_computedRows, MathPlatform::Absolute(rowIndex) + rowSpan + correction.GetY());
   _computedColumns = GetMaxValue(_computedColumns, MathPlatform::Absolute(columnIndex) + columnSpan + correction.GetX());

   _firstInStructure = (_currentOrthogonalIndex == 0);
}


void GridItemAdder::PostAdd(const Int32 directSpan, const Int32 orthogonalSpan, const Int32 maxOrthogonalItemsCount, const Float itemSize)
{
   ++_itemsInStructure;

   if (_reverse)
   {
      _currentOrthogonalIndex--;
      if (_currentOrthogonalIndex < 0)
      {
         EndStructure();
         _currentOrthogonalIndex = maxOrthogonalItemsCount - 1;
         _absoluteStructureIndex--;
      }
   }
   else
   {
      _currentOrthogonalIndex += orthogonalSpan;

      if (_currentOrthogonalIndex >= maxOrthogonalItemsCount)
      {
         EndStructure();
      }

      if (directSpan > 1)
      {
         AdvanceUntilFree(directSpan, maxOrthogonalItemsCount, itemSize);
      }
   }
}


void GridItemAdder::EndStructure()
{
   CollectSize(_maxItemCellSize);
   _lastInStructure = true;
   Int16 structureMargin(0);

   bool invisibleStructure(_logicalStructureIndex < 0);
   if (invisibleStructure)
   {
      structureMargin = -static_cast<Int16>(_maxItemCellSize);
      _completeBeginingMargin += structureMargin;
   }
   else
   {
      bool firstVisibleStructure(_logicalStructureIndex == 0);
      if (firstVisibleStructure)
      {
         _itemsInFirstStructure = _itemsInStructure;
         _beginingSize = _maxItemCellSize;
         structureMargin = _beginingMargin;
      }

      _completeBeginingMargin -= _beginingMargin;
      UpdateBeginingMargin(GetSizes());
      _completeBeginingMargin += _beginingMargin;
   }

   _endingSize = _maxItemCellSize;

   _maxItemCellSize += structureMargin;

   IncreaseSize(Math::Minimum(_maxItemCellSize, _maxRemainingItemSize));

   _maxRemainingItemSize -= _maxItemCellSize;
   if (_maxRemainingItemSize < 0.0)
   {
      _maxRemainingItemSize = 0.0;
   }

   ++_absoluteStructureIndex;
   ++_logicalStructureIndex;
   _currentOrthogonalIndex = 0;
   _itemsInStructure = 0;
   _maxItemCellSize = 0;
}


void GridItemAdder::AdjustNumberOfCompleteVisibleItems()
{
   if (0 != _completeBeginingMargin)
   {
      _numberOfCompleteVisibleItems -= _itemsInFirstStructure;
   }

   const Float structureSize(_directOrientationOperator.GetVectorComponent(_templateSpanAnalyzer->GetSpanUnitSize()));
   const Float visibleSize(_directOrientationOperator.GetVectorComponent(GetVisibleArea()));
   const Float currentSize(GetCurrentSize());

   if (currentSize >= visibleSize)
   {
      _endingMargin = Int16(visibleSize - currentSize);

      const Int16 margin = 0 - (_completeBeginingMargin + _endingMargin);

      if ((margin > structureSize) || ((_completeBeginingMargin == 0) != (_endingMargin == 0)))
      {
         _numberOfCompleteVisibleItems -= GetItemsInPartiallyVisibleStructures();
      }
   }
}


Candera::Float GridItemAdder::GetItemOrthogonalSize(Candera::Node2D& childNode)
{
   return _orthogonalOrientationOperator.GetVectorComponent(_sizesContainer.MeasureClientSize(&childNode));
}


Candera::UInt32 GridItemAdder::GetItemsCount() const
{
   return _directOrientationOperator.GetValue(GetColumns(), GetRows());
}


Candera::UInt32 GridItemAdder::GetOrthogonalItemsCount() const
{
   return _orthogonalOrientationOperator.GetValue(GetColumns(), GetRows());
}


Candera::Int32 GridItemAdder::GetCurrentRowIndex() const
{
   return _directOrientationOperator.GetValue(GetCurrentOrthogonalIndex(), GetAbsoluteStructureIndex());
}


Candera::Int32 GridItemAdder::GetCurrentColumnIndex() const
{
   return _directOrientationOperator.GetValue(GetAbsoluteStructureIndex(), GetCurrentOrthogonalIndex());
}


void GridItemAdder::SetDirectSpan(Candera::Node2D& childNode, FeatStd::UInt32 span) const
{
   _directOrientationOperator.SetRowOrColumnSpan(childNode, span);
}


void GridItemAdder::SetOrthogonalSpan(Candera::Node2D& childNode, FeatStd::UInt32 span) const
{
   _orthogonalOrientationOperator.SetRowOrColumnSpan(childNode, span);
}


FeatStd::UInt8 GridItemAdder::GetMaxShiftValue()
{
   Node2D* groupNode = GetGroupNode();
   FeatStd::UInt8 maxShiftVal = 0;
   if (0 != groupNode)
   {
      Node2D* childNode = groupNode->GetFirstChild();
      if (0 != childNode)
      {
         FeatStd::Int8 val = GetShiftValue(*childNode);
         maxShiftVal = Math::Maximum<FeatStd::UInt8>(maxShiftVal, static_cast<UInt8>(MathPlatform::Absolute(static_cast<FeatStd::Int32>(val))));
         childNode = childNode->GetNextSibling();
      }
   }

   return maxShiftVal;
}


FeatStd::Float GridItemAdder::GetItemsInPartiallyVisibleStructures() const
{
   Float items(0.0F);
   Float totalSize(GetCurrentSize());

   const Float visibleSize(_directOrientationOperator.GetVectorComponent(GetVisibleArea()));
   const Courier::Vector<FeatStd::Float>& sizes(GetSizes());
   const Int32 maxOrthogonalItemsCount(static_cast<Int32>(GetOrthogonalItemsCount()));

   Courier::Vector<FeatStd::Float>::ConstIterator sizesIterator(sizes.ConstEnd());
   --sizesIterator;
   Int32 structIndex(_absoluteStructureIndex);

   GridOccupation checkerOccupation;

   for (; (totalSize > visibleSize) && (sizesIterator != sizes.ConstBegin()); --sizesIterator)
   {
      --structIndex;
      const Float currentSize(*sizesIterator);

      for (Int32 orthogonalIndex(maxOrthogonalItemsCount - 1); orthogonalIndex >= 0; --orthogonalIndex)
      {
         const Int32 columnIndex(_orthogonalOrientationOperator.GetValue(orthogonalIndex, structIndex));
         const Int32 rowIndex(_directOrientationOperator.GetValue(orthogonalIndex, structIndex));

         if ((!checkerOccupation.IsOccupied(columnIndex, rowIndex)) && _currentOccupation.IsOccupied(columnIndex, rowIndex))
         {
            const GridOccupation::Coordinates itemStart(_currentOccupation.GetItemStart(columnIndex, rowIndex));
            const Int32 width(columnIndex - itemStart._x + 1);
            const Int32 height(rowIndex - itemStart._y + 1);

            checkerOccupation.Occupy(itemStart._x, itemStart._y, width, height);
            ++items;
         }
      }

      totalSize -= currentSize;
   }

   return items;
}
