/* ***************************************************************************************
* FILE:          GridAddingStrategy.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  GridAddingStrategy is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "Widgets/2D/List/Content/GridTemplateSpanAnalyzer.h"

#include "GridAddingStrategy.h"
#include "GridPositionsCalculator.h"
#include "Candera/System/Mathematics/Vector2.h"

using namespace FeatStd;
using namespace Candera;

GridAddingStrategy::SharedPointer GridAddingStrategy::Create(ISizeContainer& sizesContainer, const OrientationOperator& directOrientationOperator, const OrientationOperator& orthogonalOrientationOperator, const FeatStd::UInt8 rows, const FeatStd::UInt8 columns)
{
   return GridAddingStrategy::SharedPointer(FEATSTD_NEW(GridAddingStrategy)(sizesContainer, directOrientationOperator, orthogonalOrientationOperator, rows, columns));
}


ItemAdder::SharedPointer GridAddingStrategy::CreateItemAdderImpl(Candera::Node2D* groupNode)
{
   return CreateGridItemAdder(groupNode);
}


void GridAddingStrategy::ProcessDataCHangedImpl(FeatStd::Int32 dataSize)
{
   _positionsCache.Clear();
   _positionsCache.UpdateCachStep(dataSize);
}


void GridAddingStrategy::AnalyzeTemplatesImpl(const ITemplateRetriever& templateRetriever)
{
   if (!_templateSpanAnalyzer.PointsToNull())
   {
      _templateSpanAnalyzer->Analyze(templateRetriever);
   }
}


FeatStd::Int32 GridAddingStrategy::CalculateStartIndexImpl(const ITemplateRetriever& templateRetriever, const ContentProvider& contentProvider, const Candera::Float position)
{
   const Int32 size(contentProvider.GetVirtualListSize());
   FeatStd::Int32 startIndex(GetPositionsCalculator().CalculateStartIndex(templateRetriever, contentProvider, position));

   _positionsCache.Cache(PositionsCache::CacheData(static_cast<Int32>(position), size == 0 ? startIndex : startIndex % size));

   return startIndex;
}


GridAddingStrategy::GridAddingStrategy(ISizeContainer& sizesContainer, const OrientationOperator& directOrientationOperator, const OrientationOperator& orthogonalOrientationOperator, const FeatStd::UInt8 rows, const FeatStd::UInt8 columns) :
   Base(sizesContainer, directOrientationOperator, orthogonalOrientationOperator),
   _templateSpanAnalyzer(GridTemplateSpanAnalyzer::Create(sizesContainer)),
   _rows(rows),
   _columns(columns),
   _positionsCalculator(CreateGridItemAdder(0), _templateSpanAnalyzer, _positionsCache, orthogonalOrientationOperator),
   _infoCollector(CreateItemAdder(0))
{
}


class GridL : public GridItemAdderListener
{
   public:
      GridL(GridOccupation& gridOccupation) :
         _gridOccupation(gridOccupation)
      { }

      virtual void OnItemAdded(FeatStd::Int32 column, FeatStd::Int32 row, FeatStd::Int32 directSpan, FeatStd::Int32 orthogonalSpan) override
      {
         _gridOccupation.Occupy(column, row, directSpan, orthogonalSpan);
      }

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(GridL);

      GridOccupation& _gridOccupation;
};


const PositionsCalculator& GridAddingStrategy::GetPositionsCalculator() const
{
   return _positionsCalculator;
}


void GridAddingStrategy::CacheImpl(const ContentProvider& contentProvider, const ITemplateRetriever& templateRetriever)
{
   if (contentProvider.HasItems() && (!_templateSpanAnalyzer.PointsToNull()))
   {
      tSharedPtrListDataItem item0(contentProvider.GetItem(0));
      const GridItemAdder::SharedPointer& adder(CreateGridItemAdder(0));

      if ((!item0.PointsToNull()) && (!adder.PointsToNull()))
      {
         _gridOccupation.SetAbsoluteStart(0, 0);
         _gridOccupation.Clear();

         GridL listener(_gridOccupation);
         adder->SetListener(&listener);
         adder->SetGridOccupation(&_gridOccupation);
         const Int32 startIndex(contentProvider.GetStartIndex());

         /* not used for now; currently it is supposed that the whole data is available for grid lists
         const Int32 orthogonalOffset(_orthogonalOrientationOperator.GetValue(item0->getHdlCol(), item0->getHdlRow()));
         const Int32 structureIndex(0);
         */

         adder->BeginAdding(Vector2(Math::MaxFloat(), Math::MaxFloat()), static_cast<Float>(startIndex));

         for (Int32 i(0); i < contentProvider.GetWindowElementSize(); ++i)
         {
            tSharedPtrListDataItem item(contentProvider.GetItem(i));
            Candera::Node2D* itemTemplate(templateRetriever.GetTemplate(item));

            if (!item.PointsToNull() && (0 != itemTemplate))
            {
               adder->AddItem(*itemTemplate, *itemTemplate);
            }
         }

         adder->FinishAdding();
      }
   }
}


GridItemAdder::SharedPointer GridAddingStrategy::CreateGridItemAdder(Candera::Node2D* groupNode)
{
   GridItemAdder::SharedPointer adder(GridItemAdder::Create(_templateSpanAnalyzer, _sizesContainer, groupNode, _directOrientationOperator, _orthogonalOrientationOperator, _rows, _columns));

   if (!adder.PointsToNull())
   {
      adder->SetGridOccupation(&_gridOccupation);
   }

   return adder;
}


void GridAddingStrategy::CollectPagesInfoImpl(const ContentProvider& contentProvider, const AddChecker& addChecker, const ITemplateRetriever& templateRetriever, bool fixedPages, bool firstUpdateAfterNewData, FeatStd::Int32 indexOfFirstCompletelyVisibleElement, FeatStd::Int32 numberOfCompleteVisibleItems, bool circular, const Candera::Vector2 visibleArea)
{
   _infoCollector.CollectPagesInfo(contentProvider, addChecker, templateRetriever, fixedPages, firstUpdateAfterNewData, indexOfFirstCompletelyVisibleElement, numberOfCompleteVisibleItems, circular, visibleArea);
}


const PagesInfo& GridAddingStrategy::GetPagesInfoImpl() const
{
   return _infoCollector.GetPagesInfo();
}
