/* ***************************************************************************************
* FILE:          FlexListLimiter.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FlexListLimiter is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Widgets/2D/BaseWidget2D.h"
#include "FlexListLimiter.h"


using namespace Candera;

FlexListLimiter::FlexListLimiter():
   _limiterState(LimiterState::Idle),
   _listener(NULL),
   _previousFocusIndex(-1),
   _previousStartIndex(0),
   _previousNumberOfItems(0),
   _currentFocusIndex(-1),
   _currentStartIndex(0),
   _currentNumberOfItems(0),
   _listSize(0),
   _isCircularScrolling(false)
{
}


FlexListLimiter::~FlexListLimiter()
{
   _listener = 0;
}


void FlexListLimiter::OnListChangeMessage(const ListChangeMsg& msg)
{
   const Int32 msgValue(msg.GetValue());
   switch (msg.GetListChangeType())
   {
      case ListChangeUp:
         SetLimiterState(LimiterState::SingleItemStepExceedUp);
         break;
      case ListChangeDown:
         SetLimiterState(LimiterState::SingleItemStepExceedDown);
         break;
      case ListChangePageUp:
         SetLimiterState(LimiterState::MultiItemStepExceedUp);
         break;
      case ListChangePageDown:
         SetLimiterState(LimiterState::MultiItemStepExceedDown);
         break;
      case ListChangeSet:
         if (msgValue < static_cast<Int32>(_previousStartIndex))
         {
            SetLimiterState(LimiterState::MultiItemLimitReachUp);
         }
         else
         {
            SetLimiterState(LimiterState::MultiItemLimitReachDown);
         }
         break;
      default:
         SetLimiterState(LimiterState::Idle);
         break;
   }
}


void FlexListLimiter::SetCurrentValues(Candera::Int32 focusIndex, Candera::UInt32 first, Candera::UInt32 nrOfItems,
                                       Candera::UInt32 listSize, bool isCircularScrollingActive)
{
   _currentFocusIndex = focusIndex;
   _currentStartIndex = first;
   _currentNumberOfItems = nrOfItems;
   _isCircularScrolling = isCircularScrollingActive;
   _listSize = listSize;
}


void FlexListLimiter::CommitValues()
{
   if ((!_isCircularScrolling) && (_listSize > 0))
   {
      bool fireOnLimitExceededUpEvent = false;
      bool fireOnLimitExceededDownEvent = false;
      bool fireOnSwipeLimitReachedUpEvent = false;
      bool fireOnSwipeLimitReachedDownEvent = false;

      switch (_limiterState)
      {
         case LimiterState::SingleItemStepExceedUp:
            fireOnLimitExceededUpEvent = CheckSingleItemStepExceededUp();
            break;
         case LimiterState::SingleItemStepExceedDown:
            fireOnLimitExceededDownEvent = CheckSingleItemStepExceededDown();
            break;
         case LimiterState::MultiItemStepExceedUp:
            fireOnLimitExceededUpEvent = CheckMultiItemStepExceededUp();
            break;
         case LimiterState::MultiItemStepExceedDown:
            fireOnLimitExceededDownEvent = CheckMultiItemStepExceededDown();
            break;
         case LimiterState::MultiItemLimitReachUp:
            fireOnSwipeLimitReachedUpEvent = CheckLimitReachedUp();
            break;
         case LimiterState::MultiItemLimitReachDown:
            fireOnSwipeLimitReachedDownEvent = CheckLimitReachedDown();
            break;
         case LimiterState::Idle:
         default:
            break;
      }

      if (_listener != 0)
      {
         if (fireOnLimitExceededUpEvent)
         {
            _listener->OnLimitExceededUp();
         }

         if (fireOnLimitExceededDownEvent)
         {
            _listener->OnLimitExceededDown();
         }

         if (fireOnSwipeLimitReachedUpEvent)
         {
            _listener->OnLimitReachedUp();
         }

         if (fireOnSwipeLimitReachedDownEvent)
         {
            _listener->OnLimitReachedDown();
         }
      }
   }

   _previousFocusIndex = _currentFocusIndex;
   _previousNumberOfItems = _currentNumberOfItems;
   _previousStartIndex = _currentStartIndex;
   //Reset limiter state in the end. Shall not be influenced by Updates without messages.
   _limiterState = LimiterState::Idle;
}


bool FlexListLimiter::CheckSingleItemStepExceededUp() const
{
   if (_previousFocusIndex == _currentFocusIndex)
   {
      if (_currentFocusIndex == 0)
      {
         return true;
      }
   }
   return false;
}


bool FlexListLimiter::CheckSingleItemStepExceededDown() const
{
   if (_currentFocusIndex == _previousFocusIndex)
   {
      if (_currentFocusIndex == static_cast<Candera::Int32>((_listSize - 1)))
      {
         return true;
      }
   }
   return false;
}


bool FlexListLimiter::CheckMultiItemStepExceededUp() const
{
   if (_previousStartIndex == _currentStartIndex)
   {
      if (_currentStartIndex == 0)
      {
         return true;
      }
   }
   return false;
}


bool FlexListLimiter::CheckMultiItemStepExceededDown() const
{
   Candera::UInt32 previousEndIndex = _previousStartIndex + _previousNumberOfItems;
   Candera::UInt32 currentEndIndex = _currentStartIndex + _currentNumberOfItems;
   if (currentEndIndex >= previousEndIndex)
   {
      if (previousEndIndex >= (_listSize - 1))
      {
         return true;
      }
   }
   return false;
}


bool FlexListLimiter::CheckLimitReachedUp() const
{
   if (_previousStartIndex > _currentStartIndex)
   {
      if (_currentStartIndex == 0)
      {
         return true;
      }
   }
   return false;
}


bool FlexListLimiter::CheckLimitReachedDown() const
{
   Candera::UInt32 previousEndIndex = _previousStartIndex + _previousNumberOfItems;
   Candera::UInt32 currentEndIndex = _currentStartIndex + _currentNumberOfItems;
   Candera::UInt32 endOfList = _listSize - 1;
   if (previousEndIndex < endOfList)
   {
      if (currentEndIndex >= endOfList)
      {
         return true;
      }
   }
   return false;
}
