/* ***************************************************************************************
* FILE:          ExpandTriggerManager.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ExpandTriggerManager is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"

#include "ExpandTriggerManager.h"
#include "Widgets/2D/ControlTemplate/ControlTemplate.h"
#include <Widgets/2D/List/Data/ListDataWrapper.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ExpandTriggerManager.cpp.trc.h"
#endif

using namespace Courier;


ExpandTriggerManager::ExpandTriggerManager(ControlTemplateInstanceContainer& itemsInstanceContainer, const ListDataWrapper* listData) :
   _itemsInstanceContainer(itemsInstanceContainer),
   _markedIndices(),
   _timeDispatcher(0),
   _listData(listData)
{
}


ExpandTriggerManager::~ExpandTriggerManager()
{
   _markedIndices.Clear();
   _timeDispatcher = Candera::Animation::AnimationTimeDispatcher::SharedPointer(0);
}


void ExpandTriggerManager::SetTimeDispatcher(const Candera::Animation::AnimationTimeDispatcher::SharedPointer& timeDispatcher)
{
   _timeDispatcher = timeDispatcher;
}


bool ExpandTriggerManager::MarkToStart(FeatStd::UInt32 relativeItemIndex, bool forward)
{
   Courier::Vector<ControlTemplateInstancePtr>& instances(_itemsInstanceContainer.GetInstances());
   bool marked(relativeItemIndex < instances.Size());

   if (marked)
   {
      IndexInfo indexInfo;
      indexInfo.index = relativeItemIndex;
      indexInfo.playbackType = (forward ? Forward : Backward);
      _markedIndices.Add(indexInfo);
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ExpandTriggerManager::MarkToStart markedIndex=%d, forward=%d, marked=%d", relativeItemIndex, forward, marked));

   return marked;
}


bool ExpandTriggerManager::UpdateAnimation()
{
   bool update(false);

   if (!_timeDispatcher.PointsToNull())
   {
      Courier::Vector<ControlTemplateInstancePtr>& instances(_itemsInstanceContainer.GetInstances());
      for (Int index(0); index < _markedIndices.Size(); ++index)
      {
         const IndexInfo info(_markedIndices[index]);
         ControlTemplateInstancePtr& controlTemplateInstance(instances[info.index]);
         if (!controlTemplateInstance.PointsToNull())
         {
            AnimationsContainer& expandAnimations = controlTemplateInstance->GetExpandAnimationsManager();
            expandAnimations.SetWorldTime(_timeDispatcher->GetWorldTimeMs());

            switch (info.playbackType)
            {
               case ReverseCurrent:
               {
                  update = expandAnimations.Start(CTIAnimationsManager::ForwardPlayback) ||
                           expandAnimations.Start(CTIAnimationsManager::ReversePlayback) ||
                           update;
                  break;
               }

               case Forward:
               {
                  update = expandAnimations.Start(CTIAnimationsManager::ForwardPlayback) || update;
                  break;
               }

               case Backward:
               {
                  update = expandAnimations.Start(CTIAnimationsManager::ReversePlayback) || update;
                  break;
               }

               default:
                  break;
            }
         }
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ExpandTriggerManager::UpdateAnimation markedIndex=%d, update=%d", info.index, update));
      }
   }
   _markedIndices.Clear();

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ExpandTriggerManager::UpdateAnimation update=%d", update));
   return update;
}


bool ExpandTriggerManager::UpdateItemState(ControlTemplateInstance* templateInstance)
{
   bool changed = false;
   if ((0 != templateInstance) && (0 != _listData) && (_listData->IsDataAvailable()))
   {
      FeatStd::UInt32 absIndex = templateInstance->GetIndex();
      FeatStd::UInt32 dataIndex = absIndex - _listData->GetStartIndex();
      if (dataIndex < _listData->GetWindowElementSize())
      {
         tSharedPtrListDataItem item((*_listData)[dataIndex]);
         if (!item.PointsToNull())
         {
            AnimationsContainer& expandAnimations = templateInstance->GetExpandAnimationsManager();
            if (!expandAnimations.AreRunning())
            {
               changed = (item->IsCollapsed()) ? expandAnimations.ToBegin() : expandAnimations.ToEnd();
            }
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ExpandTriggerManager::UpdateItemState itemCollapsed=%d, changed=%d", item->IsCollapsed(), changed));
            expandAnimations.DispatchWorldTime();
         }
      }
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ExpandTriggerManager::UpdateItemState changed=%d", changed));
   return changed;
}
