/* ***************************************************************************************
* FILE:          CustomAnimationsContainer.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  CustomAnimationsContainer is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/CustomAnimationsContainer.cpp.trc.h"
#endif

#include "CustomAnimationsContainer.h"

#include "Widgets/2D/ControlTemplate/DefaultControlTemplateMap.h"
#include "Widgets/2D/ControlTemplate/ControlTemplateInstance.h"
#include "Widgets/2D/ControlTemplate/AnimationCloner.h"
#include "Widgets/2D/Common/ListItemAnimationDynamicPropertyHost.h"
#include "Widgets/2D/ControlTemplate/AnimationModifier.h"
#include "Widgets/2D/List/generated/ListMessages.h"
#include <Widgets/2D/List/generated/ListEnums.h>

using namespace Candera;


CustomAnimationsContainer::CustomAnimationsContainer(Candera::ArrayProperty<Candera::Animation::AnimationPlayer::SharedPointer>& animations) :
   _ownerId(0),
   _timeDispatcher(Animation::AnimationTimeDispatcher::Create()),
   _allAnimations(animations),
   _animationsGroup(0),
   _itemsGroup(0)
{
}


CustomAnimationsContainer::~CustomAnimationsContainer()
{
   if (!_timeDispatcher.PointsToNull())
   {
      for (Int i(0); i < (Int) _timeDispatcher->GetPlayerCount(); ++i)
      {
         MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> player(_timeDispatcher->GetPlayer(i));
         _timeDispatcher->RemovePlayer(player); // mark the player to be removed
      }

      _timeDispatcher->DispatchTime(); // actually remove the players
   }
   _animations.Clear();

   _timeDispatcher = Animation::AnimationTimeDispatcher::SharedPointer(0);
   _animationsGroup = 0;
   _itemsGroup = 0;
}


void CustomAnimationsContainer::SetAnimationsGroup(Candera::Node2D* val)
{
   _animationsGroup = val;
}


void CustomAnimationsContainer::SetItemsGroup(Candera::Node2D* val)
{
   _itemsGroup = val;
}


void CustomAnimationsContainer::SetOwnerId(FeatStd::UInt32 val)
{
   _ownerId = val;
}


bool CustomAnimationsContainer::ProcessMessage(const Courier::Message& msg, bool& invalidate)
{
   bool consumed(false);

   if (msg.GetId() == ListCustomAnimationReqMsg::ID)
   {
      const ListCustomAnimationReqMsg* reqMsg(Courier::message_cast<const ListCustomAnimationReqMsg*>(&msg));
      if ((0 != reqMsg) && ((0 == reqMsg->GetListId()) || (_ownerId == reqMsg->GetListId())))
      {
         const CustomListAnimationAction& action(reqMsg->GetAction());

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "CustomAnimationsContaine processing ListCustomAnimationReqMsg(ownerId=%d,  action=%d, animationId=%s)", reqMsg->GetListId(), action, reqMsg->GetAnimationId().CStr()));

         switch (action)
         {
            case CLAPlay:
               Start(reqMsg->GetAnimationId());
               break;

            case CLAHoldStart:
            case CLAHoldEnd:
               Hold(reqMsg->GetAnimationId(), action);
               break;

            default:
               // do nothing
               break;
         }

         invalidate = true;
      }
   }

   return consumed;
}


void CustomAnimationsContainer::Start(Courier::ItemId animationId)
{
   CustomAnimationEntity* targetAnimation(GetAnimation(animationId));

   if (0 != targetAnimation)
   {
      targetAnimation->PrepareForPlay();
   }
}


void CustomAnimationsContainer::Hold(Courier::ItemId animationId, CustomListAnimationAction action)
{
   CustomAnimationEntity* targetAnimation(GetAnimation(animationId));

   if ((0 != targetAnimation) && ((action == CLAHoldStart) || (action == CLAHoldEnd)))
   {
      CustomAnimationEntity::HoldType holdType((action == CLAHoldStart) ? CustomAnimationEntity::Start : CustomAnimationEntity::End);
      targetAnimation->PrepareForHold(holdType);
   }
}


CustomAnimationEntity* CustomAnimationsContainer::GetAnimation(Courier::ItemId animationId)
{
   CustomAnimationEntity* targetAnimation(0);

   for (CustomAnimations::Iterator it(_animations.Begin()); (0 == targetAnimation) && (it != _animations.End()); ++it)
   {
      CustomAnimationEntity& animation(*it);
      if (animation.HasId(animationId))
      {
         targetAnimation = &animation;
      }
   }

   for (UInt i(0); (0 == targetAnimation) && (i < _allAnimations.GetCount()); ++i)
   {
      const Candera::Animation::AnimationPlayer::SharedPointer& animation(_allAnimations.Get(i));
      if ((!animation.PointsToNull()) && (animationId == Courier::ItemId(animation->GetName())))
      {
         ListItemAnimationType animationType(static_cast<ListItemAnimationType>(ListItemAnimationDynamicPropertyHost::GetListItemAnimationType(animation)));
         if (animationType == LIANone)
         {
            if (_animations.Add(CustomAnimationEntity(animation, _ownerId)))
            {
               targetAnimation = &_animations[_animations.Size() - 1];
               targetAnimation->SetAnimationsGroup(_animationsGroup);
               targetAnimation->SetItemsGroup(_itemsGroup);
            }
         }
      }
   }

   return targetAnimation;
}


void CustomAnimationsContainer::Update(Candera::Animation::WorldTimeType worldTime)
{
   if (!_timeDispatcher.PointsToNull())
   {
      _timeDispatcher->SetWorldTimeMs(worldTime);

      for (CustomAnimations::Iterator it(_animations.Begin()); it != _animations.End(); ++it)
      {
         (*it).Update(*this);
      }

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "CustomAnimationsContainer::Update worldTime=%d", worldTime));
      _timeDispatcher->DispatchTime();
   }
}


bool CustomAnimationsContainer::IsRunning() const
{
   bool running(false);

   for (CustomAnimations::ConstIterator it(_animations.ConstBegin()); (!running) && (it != _animations.ConstEnd()); ++it)
   {
      running = running || (*it).IsRunning();
   }

   return running;
}


void CustomAnimationsContainer::Register(const Candera::Animation::AnimationPlayerBase::SharedPointer& player)
{
   if (!_timeDispatcher.PointsToNull())
   {
      _timeDispatcher->AddPlayer(player);
   }
}


void CustomAnimationsContainer::UnRegister(const Candera::Animation::AnimationPlayerBase::SharedPointer& player)
{
   if (!_timeDispatcher.PointsToNull())
   {
      _timeDispatcher->RemovePlayer(player);
   }
}
