/* ***************************************************************************************
* FILE:          CustomAnimationEntity.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  CustomAnimationEntity is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/CustomAnimationEntity.cpp.trc.h"
#endif

#include "CustomAnimationEntity.h"

#include "Widgets/2D/ControlTemplate/DefaultControlTemplateMap.h"
#include "Widgets/2D/ControlTemplate/ControlTemplateInstance.h"
#include "Widgets/2D/ControlTemplate/AnimationCloner.h"
#include "Widgets/2D/ControlTemplate/CloneProperties.h"
#include "Widgets/2D/Common/ListItemAnimationDynamicPropertyHost.h"
#include "Widgets/2D/ControlTemplate/AnimationModifier.h"
#include "Widgets/2D/List/generated/ListMessages.h"
#include "hmibase/util/Ticker.h"

using namespace Candera;
using namespace Courier;


CustomAnimationEntity::CustomAnimationEntity(Player original, FeatStd::UInt32 ownerId) :
   _ownerId(ownerId),
   _status(Idle),
   _hold(None),
   _original(original),
   _clone(0),
   _animationsGroup(0),
   _itemsGroup(0)
{
   CloneOriginal();
}


CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1538, "Base class needs not be copied; additionally, the base class is uncopyable.")
CustomAnimationEntity::CustomAnimationEntity(const CustomAnimationEntity& other) :
   _status(Idle),
   _hold(None),
   _animationsGroup(0),
   _itemsGroup(0)
{
   *this = other;
}


CustomAnimationEntity& CustomAnimationEntity::operator=(const CustomAnimationEntity& other)
{
   if (this != &other)
   {
      _ownerId = other._ownerId;
      _status = other._status;
      _hold = other._hold;
      _original = other._original;
      _clone = other._clone;

      CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1555, "Intended direct pointer copying.")
      _animationsGroup = other._animationsGroup;
      CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1555, "Intended direct pointer copying.")
      _itemsGroup = other._itemsGroup;
   }

   return *this;
}


CustomAnimationEntity::~CustomAnimationEntity()
{
   unsigned long t(hmibase::utils::Ticker::getTickCountMsec());
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "CustomAnimationEntity::Destructor t=%ld, status=%d, animationId=%s", t, _status, _original->GetName()));

   if (!_clone.PointsToNull())
   {
      _clone->RemoveAnimationPlayerListener(this);
   }

   _original = Animation::AnimationPlayer::SharedPointer(0);
   _clone = Animation::AnimationPlayer::SharedPointer(0);
   _animationsGroup = 0;
   _itemsGroup = 0;
}


bool CustomAnimationEntity::HasId(Courier::ItemId id)
{
   return (!_original.PointsToNull()) && (id == ItemId(_original->GetName()));
}


void CustomAnimationEntity::PrepareForPlay()
{
   _status = MarkedForPlaback;

   if (!_original.PointsToNull())
   {
      unsigned long t(hmibase::utils::Ticker::getTickCountMsec());
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "CustomAnimationEntity::PrepareForPlay t=%ld, animationId=%s", t, _original->GetName()));
   }
}


void CustomAnimationEntity::PrepareForHold(HoldType val)
{
   _hold = val;
   PostResMsg(GetHoldAction(), true);

   if (!_original.PointsToNull())
   {
      unsigned long t(hmibase::utils::Ticker::getTickCountMsec());
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "CustomAnimationEntity::PrepareForHold t=%ld, animationId=%s", t, _original->GetName()));
   }
}


CustomListAnimationAction CustomAnimationEntity::GetHoldAction() const
{
   CustomListAnimationAction action(CLAHoldNone);
   if (_hold == End)
   {
      action = CLAHoldEnd;
   }
   else if (_hold == Start)
   {
      action = CLAHoldStart;
   }
   else
   {
      action = CLAHoldNone;
   }

   return action;
}


void CustomAnimationEntity::PostResMsg(CustomListAnimationAction action, bool success) const
{
   if (!_original.PointsToNull())
   {
      ListCustomAnimationResMsg* resMsg(COURIER_MESSAGE_NEW(ListCustomAnimationResMsg)(_ownerId, ItemId(_original->GetName()), action, success));
      if (0 != resMsg)
      {
         resMsg->Post();
      }
   }
}


void CustomAnimationEntity::Update(TimeDispatcherRegistrar& registrar)
{
   if (_status == Idle)
   {
      Hold();
   }
   else if (_status == MarkedForPlaback)
   {
      Play(registrar);
   }

   if (_status == Running)
   {
      AdjustWhileRunning(registrar);
   }
}


void CustomAnimationEntity::AdjustWhileRunning(TimeDispatcherRegistrar& registrar)
{
   if ((!_original.PointsToNull()) && (!_clone.PointsToNull()))
   {
      bool modified(AnimationModifier().Modify(_original->GetController(), _clone->GetController(), _animationsGroup, _itemsGroup));
      if (!modified)
      {
         Stop();
         registrar.UnRegister(_clone);
      }

      unsigned long t(hmibase::utils::Ticker::getTickCountMsec());
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "CustomAnimationEntity::AdjustWhileRunning t=%ld, animationId=%30s modified=%d", t, _original->GetName(), modified));
   }
}


void CustomAnimationEntity::CloneOriginal()
{
   if (!_original.PointsToNull())
   {
      _clone = Animation::AnimationPlayer::Create();
      if (!_clone.PointsToNull())
      {
         Animation::AnimationController::SharedPointer cloneController(AnimationCloner::DuplicateAnimationController(_original.GetPointerToSharedInstance()));
         if (!cloneController.PointsToNull())
         {
            _clone->SetController(cloneController);
            PropertiesClonners::CloneProperties(*_original, *_clone);

            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "CustomAnimationEntity::CloneOriginal succesfully cloned animationId=%s)", _original->GetName()));
         }
      }
   }
}


void CustomAnimationEntity::Play(TimeDispatcherRegistrar& registrar)
{
   bool success(false);
   _status = Idle;

   if ((!_original.PointsToNull()) && (!_clone.PointsToNull()))
   {
      bool modified(AnimationModifier().Modify(_original->GetController(), _clone->GetController(), _animationsGroup, _itemsGroup));
      if (modified)
      {
         registrar.Register(_clone);
         _clone->AddAnimationPlayerListener(this);

         if ((_clone->Start()))
         {
            _status = Running;
            success = true;
         }
         else
         {
            registrar.UnRegister(_clone);
            _clone->RemoveAnimationPlayerListener(this);
         }
      }
   }

   PostResMsg(CLAPlay, success);

   if (!_original.PointsToNull())
   {
      unsigned long t(hmibase::utils::Ticker::getTickCountMsec());
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "CustomAnimationEntity::Play t=%ld, status=%d, success=%d, animationId=%s", t, _status, success, _original->GetName()));
   }
}


bool CustomAnimationEntity::IsRunning() const
{
   return (!_clone.PointsToNull()) && (_clone->IsEnabled());
}


void CustomAnimationEntity::Hold()
{
   if ((_hold != None) && (!_original.PointsToNull()) && (!_clone.PointsToNull()))
   {
      bool modified(AnimationModifier().Modify(_original->GetController(), _clone->GetController(), _animationsGroup, _itemsGroup));
      if (modified)
      {
         const Animation::AnimationController::SharedPointer& controller(_clone->GetController());
         const Animation::SequenceTimeType sequenceDurationMs(_clone->GetSequenceDurationMs());
         if (!controller.PointsToNull())
         {
            controller->AnimateSequenceTime(_hold == End ? sequenceDurationMs : 0);
            unsigned long t(hmibase::utils::Ticker::getTickCountMsec());
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "CustomAnimationEntity::Hold t=%ld, status=%d, animationId=%s", t, _status, _original->GetName()));
         }
      }
   }
}


void CustomAnimationEntity::OnPastEnd(Candera::Animation::AnimationPlayerBase* animationPlayer, Int32 /*completedIterationsCount*/)
{
   if ((0 != animationPlayer) && (_clone.GetPointerToSharedInstance() == animationPlayer))
   {
      Stop();

      if (!_original.PointsToNull())
      {
         ListCustomAnimationIndMsg* indMsg(COURIER_MESSAGE_NEW(ListCustomAnimationIndMsg)(_ownerId, ItemId(_original->GetName())));

         if (0 != indMsg)
         {
            indMsg->Post();
         }
         unsigned long t(hmibase::utils::Ticker::getTickCountMsec());
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "CustomAnimationEntity::OnPastEnd t=%ld, animationId=%s", t, _original->GetName()));
      }
   }
}


void CustomAnimationEntity::Stop()
{
   if (_status == Running)
   {
      _status = Idle;
   }

   if (!_clone.PointsToNull())
   {
      _clone->RemoveAnimationPlayerListener(this);
   }

   if (!_original.PointsToNull())
   {
      unsigned long t(hmibase::utils::Ticker::getTickCountMsec());
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "CustomAnimationEntity::Stop t=%ld animationId=%s", t, _original->GetName()));
   }
}
