/* ***************************************************************************************
* FILE:          ImageSwitchWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ImageSwitchWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "CanderaPlatform/Device/Common/Effects/BitmapBrushBlend.h"
#include "ImageSwitchWidget2D.h"
#include "Widgets/2D/ControlTemplate/ControlTemplateBinding.h"
#include "Widgets/2D/ImageEffect/ImageEffectWidget2D.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_IMAGE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ImageSwitchWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(ImageSwitchWidget2D);


ImageSwitchWidget2D::ImageSwitchWidget2D(): Base(),
   _bitmaps(GetCount()),
   _invalid(true)
{
}


ImageSwitchWidget2D::~ImageSwitchWidget2D()
{
}


void ImageSwitchWidget2D::OnChanged(Candera::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   switch (propertyId)
   {
      case IndexPropertyId:
      case CountPropertyId:
         _invalid = true;
         break;

      default:
         break;
   }
}


bool ImageSwitchWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const ImageSwitchWidget2D* original = CLONEABLE_WIDGET_CAST<const ImageSwitchWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetCount(original->GetCount());

      //copy constructor
      _bitmaps = const_cast<const std::vector<BitmapType>&>(original->_bitmaps);

      if (ControlTemplateBinding::IsSelectedBindable(*this))
      {
         Candera::Int32 value = ControlTemplateBinding::GetSelectedValue(*this);
         FEATSTD_DEBUG_ASSERT(value >= 0);
         SetIndex(static_cast<IndexType>(value));
      }
      else
      {
         SetIndex(original->GetIndex());
      }

      cloned = true;
   }
   return cloned;
}


void ImageSwitchWidget2D::SetBitmap(IndexType index, const BitmapType& bitmap)
{
   if (index < MAX_BITMAP_COUNT)
   {
      //resize the vector if necessary
      if (index >= _bitmaps.size())
      {
         _bitmaps.resize((index + 1 < GetCount()) ? GetCount() : (index + 1));
      }
      _bitmaps[index] = bitmap;
   }
}


const ImageSwitchWidget2D::BitmapType& ImageSwitchWidget2D::GetBitmap(IndexType index) const
{
   //return an empty shared pointer if the index is greater or equal to the number of bitmaps
   if (index >= _bitmaps.size())
   {
      static BitmapType nullBitmap;
      return nullBitmap;
   }
   return _bitmaps.at(index);
}


bool ImageSwitchWidget2D::IsBitmapPropertyVisible(IndexType index) const
{
   return index < GetCount();
}


void ImageSwitchWidget2D::Update()
{
   if (_invalid && (GetNode() != NULL))
   {
      Candera::MemoryManagement::SharedPointer<Candera::Image2D> newImage = GetBitmap(GetIndex());
      ImageEffectWidget2D::SetImage(*this, newImage);

      _invalid = false;
   }
}
