/* ***************************************************************************************
* FILE:          GizmoController2D.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  GizmoController2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef Gizmo_CONTROLLER2D_H
#define Gizmo_CONTROLLER2D_H

#include <bitset>
#include <View/CGI/Widget/WidgetController.h>
#include <Widgets/2D/Gizmo/generated/GizmoWidget2DTypes.h>


namespace hmibase {
namespace widget {
namespace gizmo {


/*****************************************************************************/
class GizmoWidget2D;


/*****************************************************************************/
/* Base class for all gizmo controllers. */
class GizmoController2D : public hmibase::widget::WidgetController
{
      FEATSTD_TYPEDEF_BASE(hmibase::widget::WidgetController);

   public:
      CANDERA_RTTI_DECLARATION(GizmoController2D);

      virtual ~GizmoController2D() {}

      WIDGET_CONTROLLER_ON_GESTURE()
      virtual bool OnTapGesture(DelegateWidget&, const hmibase::input::gesture::GestureEvent&)
      {
         return false;
      }
      virtual bool OnDragGesture(DelegateWidget&, const hmibase::input::gesture::GestureEvent&)
      {
         return false;
      }
      virtual bool OnSwipeGesture(DelegateWidget&, const hmibase::input::gesture::GestureEvent&)
      {
         return false;
      }
      virtual bool OnPinchSpreadGesture(DelegateWidget&, const hmibase::input::gesture::GestureEvent&)
      {
         return false;
      }
      virtual bool OnRotateGesture(DelegateWidget&, const hmibase::input::gesture::GestureEvent&)
      {
         return false;
      }
};


/*****************************************************************************/
/* Controller data used by the default gizmo controller. */
class DefaultGizmoController2DData : public hmibase::widget::WidgetControllerData
{
      FEATSTD_TYPEDEF_BASE(hmibase::widget::WidgetControllerData);

   public:
      CANDERA_RTTI_DECLARATION(DefaultGizmoController2DData);

      DefaultGizmoController2DData() : StartRotation(0.0f), RotationFactor(0.0f), InternalRotation(0.0f), IsPressed(false), IsWarning(false) {}
      virtual ~DefaultGizmoController2DData() {}

      /*****************************************************************************/
      /* Minimum size of the node.*/
      Candera::Vector2 MinSize;
      /* Maximum size of the node.*/
      Candera::Vector2 MaxSize;
      /* Limits the movement of the node.*/
      Candera::Rectangle LimitArea;

      /*****************************************************************************/
      /* Pt1 received when the gesture starts. */
      Candera::Vector2 StartPt1;
      /* Gesture data received when the gesture starts. */
      hmibase::input::gesture::GestureEvent StartGestureData;

      /*****************************************************************************/
      /* Data obtained from the widget when the drag gesture starts. */
      Candera::Vector2 StartTopLeft;
      Candera::Vector2 StartBottomRight;
      FeatStd::Float StartRotation;

      /*****************************************************************************/
      /* Factor calculated when the drag gesture starts based on the touched gizmo nodes. It is used during drag move to calculate the new internal values.*/
      Candera::Vector2 TopLeftFactor;
      Candera::Vector2 BottomRightFactor;
      FeatStd::Float RotationFactor;

      /*****************************************************************************/
      /* Internal values calculated by the controller. */
      Candera::Vector2 InternalPosition;
      Candera::Vector2 InternalSize;
      FeatStd::Float InternalRotation;

      /*****************************************************************************/
      /* Indicates that the gizmo is touched. */
      bool IsPressed;

      /* Indicates a warning situation. Not set currently by the controller. */
      bool IsWarning;

      /*****************************************************************************/
      /* Due to a bug in the gesture framework when multiple gestures are enabled for a widget, the Start of a new gesture is received before End/Abort of the old gesture.
       * For this reason we need to keep track what gestures are active in order to consider the gizmo as pressed. */
      std::bitset<static_cast<size_t>(hmibase::input::gesture::GestureEvent::GT_None)> ActiveGestures;

      /* Current simplified edit modes. */
      enum enEditMode { enTranslate, enResize, enRotate, enEditModeCount };
      std::bitset<static_cast<size_t>(enEditModeCount)> EditMode;

      /* Edit modes associated to all nodes which were touched in the beginning of the gesture. */
      std::bitset<static_cast<size_t>(enGizmoEditMode::_ValueCount)> FullEditMode;
};


/*****************************************************************************/
/*
* The default gizmo controller provides support to move/resize nodes or to rotate them (rotation support is rudimentary).
* It also uses two colors (property Colors): 1st color is set on gizmo nodes when no node is touched, 2nd color is used when at least one node is touched.
* It handles just the drag gesture.
*/
class DefaultGizmoController2D : public GizmoController2D
{
      FEATSTD_TYPEDEF_BASE(GizmoController2D);

   public:
      CANDERA_RTTI_DECLARATION(DefaultGizmoController2D);

      DefaultGizmoController2D();
      virtual ~DefaultGizmoController2D() {}

      /* Singleton instance. */
      static DefaultGizmoController2D& getInstance()
      {
         static DefaultGizmoController2D instance;
         return instance;
      }

      bool getAutoUpdateWidget() const
      {
         return _autoUpdateWidget;
      }
      void setAutoUpdateWidget(bool value)
      {
         _autoUpdateWidget = value;
      }

      bool getUpdateOnlyConfiguredModes() const
      {
         return _updateOnlyConfiguredModes;
      }
      void setUpdateOnlyConfiguredModes(bool value)
      {
         _updateOnlyConfiguredModes = value;
      }

      FeatStd::Int getEditingRenderOrderRank() const
      {
         return _editingRenderOrderRank;
      }
      void setEditingRenderOrderRank(FeatStd::Int value)
      {
         _editingRenderOrderRank = value;
      }

      virtual WidgetControllerData* CreateData(DelegateWidget&) override
      {
         return CANDERA_NEW(DefaultGizmoController2DData);
      }

      /*****************************************************************************/
      /* Delegate methods */
      virtual void Update(DelegateWidget& widget) override;
      virtual bool OnGesture(DelegateWidget& widget, const hmibase::input::gesture::GestureEvent& gestureData) override;
      virtual bool OnDragGesture(DelegateWidget& widget, const hmibase::input::gesture::GestureEvent& gestureData) override;
      virtual bool OnPinchSpreadGesture(DelegateWidget& widget, const hmibase::input::gesture::GestureEvent& gestureData) override;
      virtual FeatStd::Optional<WidgetGestureConfig> GetDefaultGestureConfig(const DelegateWidget& widget) override;

      /*****************************************************************************/
      static FeatStd::EventSource& GetEventSource();

   protected:
      virtual void onGestureBeforeStart(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData);
      virtual void onGestureAfterStart(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData);
      virtual void onGestureBeforeEnd(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData);
      virtual void onGestureAfterEnd(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData);
      virtual void onGestureBeforeAbort(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData);
      virtual void onGestureAfterAbort(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData);

      virtual bool onDragGestureStart(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData);
      virtual bool onDragGestureMove(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData);
      virtual bool onDragGestureEnd(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData);
      virtual bool onDragGestureAbort(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData);

      virtual bool onPinchSpreadGestureStart(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData);
      virtual bool onPinchSpreadGestureMove(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData);
      virtual bool onPinchSpreadGestureEnd(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData);
      virtual bool onPinchSpreadGestureAbort(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData);

      virtual void onGestureMove(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const Candera::Vector2& delta);

      /* Checks if the gizmo node specified by index intersects the point. */
      virtual bool isNodeIntersecting(GizmoWidget2D& widget, size_t index, const Candera::Vector2& point);

      /* Checks which edit modes are enabled. */
      virtual void checkEditModes(GizmoWidget2D& widget, bool& isTranslateEnabled, bool& isResizeEnabled, bool& isRotateEnabled) const;

      /* Updates the node. */
      virtual void updateNode(GizmoWidget2D& widget, DefaultGizmoController2DData& data, Candera::Node2D& node);

      /* Updates the gizmo. */
      virtual void updateGizmo(GizmoWidget2D& widget, DefaultGizmoController2DData& data);

      /* Returns the color to be used for gizmo nodes. */
      virtual FeatStd::Optional<Candera::Color> getGizmoNodesColor(GizmoWidget2D& widget, DefaultGizmoController2DData& data);

      /* Updates the gizmo nodes with the specified color. */
      virtual void updateGizmoNodesColor(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const Candera::Color& color);

      /* Updates the widget based on internal values. */
      virtual void updateWidget(GizmoWidget2D& widget, DefaultGizmoController2DData& data);

      /* Posts gizmo update message. */
      virtual void postUpdateMessage(GizmoWidget2D& widget, DefaultGizmoController2DData& data, bool completed);

      /* Update widget automatically on gesture end. */
      bool _autoUpdateWidget;

      /* Update only modes which are configured in scene composer. */
      bool _updateOnlyConfiguredModes;

      /* Render order rank set on the node while editing. This variable is used only if it is different from 0. */
      FeatStd::Int _editingRenderOrderRank;
};


/*****************************************************************************/
class DefaultBoundsLimiter
{
   public:
      DefaultBoundsLimiter(const Candera::Rectangle& limitArea, const Candera::Vector2& minSize, const Candera::Vector2& maxSize);
      virtual ~DefaultBoundsLimiter() {}

      virtual void limit(Candera::Vector2& position, Candera::Vector2& size);

      template <typename TVal>
      static void limitSize(TVal& size, const TVal& minSize, const TVal& maxSize)
      {
         if ((minSize >= TVal()) && (size < minSize))
         {
            size = minSize;
         }
         if ((maxSize >= TVal()) && (size > maxSize))
         {
            size = maxSize;
         }
      }

      template <typename TVal>
      static void limitPosition(TVal& position, TVal& size, const TVal& lowerLimit, const TVal& upperLimit)
      {
         if ((lowerLimit < upperLimit) && (size <= upperLimit - lowerLimit))
         {
            //clamp left edge
            if (position < lowerLimit)
            {
               position = lowerLimit;
            }
            //clamp right edge
            if (position + size > upperLimit)
            {
               position = upperLimit - size;
            }
         }
      }

   protected:
      Candera::Rectangle _limitArea;
      Candera::Vector2 _minSize;
      Candera::Vector2 _maxSize;
};


/*****************************************************************************/
class EditUpdEvent : public FeatStd::Event
{
      FEATSTD_TYPEDEF_BASE(FeatStd::Event);

   public:
      FEATSTD_RTTI_DECLARATION();

      typedef std::bitset<static_cast<size_t>(DefaultGizmoController2DData::enEditModeCount)> EditModeType;

      enum enAction { Start, Move, End};

      EditUpdEvent(GizmoWidget2D* g,
                   const hmibase::input::gesture::GestureEvent& gestureData,
                   const EditModeType& editMode,
                   enAction action,
                   const Candera::Vector2& position,
                   const Candera::Vector2& size,
                   FeatStd::Float rotation)
         : _gizmo(g), _gestureData(gestureData), _editMode(editMode), _action(action), _position(position), _size(size), _rotation(rotation)
      {
      }

      ~EditUpdEvent()
      {
         _gizmo = NULL;
      }

      GizmoWidget2D* GetGizmo() const
      {
         return _gizmo;
      }

      const hmibase::input::gesture::GestureEvent& GetGestureData() const
      {
         return _gestureData;
      }

      const EditModeType& GetEditMode() const
      {
         return _editMode;
      }

      enAction GetAction() const
      {
         return _action;
      }

      const Candera::Vector2& GetPosition() const
      {
         return _position;
      }

      const Candera::Vector2& GetSize() const
      {
         return _size;
      }

      FeatStd::Float GetRotation() const
      {
         return _rotation;
      }

   private:
      EditUpdEvent(const EditUpdEvent&);
      EditUpdEvent& operator =(const EditUpdEvent&);

      GizmoWidget2D* _gizmo;
      hmibase::input::gesture::GestureEvent _gestureData;
      EditModeType _editMode;
      enAction _action;
      Candera::Vector2 _position;
      Candera::Vector2 _size;
      FeatStd::Float _rotation;
};


}
}


}

#endif
