/* ***************************************************************************************
* FILE:          GizmoController2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  GizmoController2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "GizmoController2D.h"
#include "GizmoWidget2D.h"
#include <Widgets/2D/EffectControl/ColorEffectWidget2D.h>


#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_GIZMO
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/GizmoController2D.cpp.trc.h"
#endif


namespace hmibase {
namespace widget {
namespace gizmo {


using ::Candera::Char;
CANDERA_RTTI_DEFINITION(GizmoController2D)
CANDERA_RTTI_DEFINITION(DefaultGizmoController2DData)
CANDERA_RTTI_DEFINITION(DefaultGizmoController2D)
FEATSTD_RTTI_DEFINITION(EditUpdEvent, FeatStd::Event)


/*****************************************************************************/
DefaultGizmoController2D::DefaultGizmoController2D() : _autoUpdateWidget(true), _updateOnlyConfiguredModes(false), _editingRenderOrderRank(0)
{
}


/*****************************************************************************/
FeatStd::EventSource& DefaultGizmoController2D::GetEventSource()
{
   FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::EventSource, s_eventSource);
   return s_eventSource;
}


/*****************************************************************************/
void DefaultGizmoController2D::Update(DelegateWidget& baseWidget)
{
   Candera::Node2D* node = baseWidget.GetNode2D();
   GizmoWidget2D* widget = Candera::Dynamic_Cast<GizmoWidget2D*>(&baseWidget);
   DefaultGizmoController2DData* data = Candera::Dynamic_Cast<DefaultGizmoController2DData*>(baseWidget.GetControllerData());

   if ((node != NULL) && (widget != NULL) && (data != NULL))
   {
      if (!data->IsPressed)
      {
         data->InternalPosition = widget->GetPosition();
         data->InternalSize = widget->GetSize();
         data->InternalRotation = widget->GetRotation();
      }

      updateNode(*widget, *data, *node);
      updateGizmo(*widget, *data);
   }
}


/*****************************************************************************/
void DefaultGizmoController2D::updateNode(GizmoWidget2D& widget, DefaultGizmoController2DData& data, Candera::Node2D& node)
{
   bool isTranslateEnabled = false;
   bool isResizeEnabled = false;
   bool isRotateEnabled = false;

   // legacy mode, only modes configured in scene composer are updated
   if (getUpdateOnlyConfiguredModes() && !widget.IsOwnedByPageEdit())
   {
      checkEditModes(widget, isTranslateEnabled, isResizeEnabled, isRotateEnabled);

      // rotation is not supported together with move/resize
      if (isRotateEnabled)
      {
         isTranslateEnabled = false;
         isResizeEnabled = false;
      }
   }
   else
   {
      isTranslateEnabled = true;
      isResizeEnabled = true;
      isRotateEnabled = true;
   }

   if (isRotateEnabled)
   {
      if (!Candera::Math::FloatAlmostEqual(node.GetRotation(), data.InternalRotation))
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "UpdateNode rot=%f %s", data.InternalRotation, HMIBASE_TO_STRING_VW(&widget)));

         node.SetRotation(data.InternalRotation);
      }
   }

   bool invalidateLayout = false;
   if (isTranslateEnabled || isResizeEnabled)
   {
      node.SetPosition(data.InternalPosition);

      // set margin to control the position, parent group must have an overlay layouter
      Candera::Margin margin(static_cast<FeatStd::Int16>(data.InternalPosition.GetX()), static_cast<FeatStd::Int16>(data.InternalPosition.GetY()), 0, 0);
      if (Candera::Layouter::GetMargin(node) != margin)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "UpdateNode margin=[%d,%d] %s",
                             margin.GetBottom(), margin.GetLeft(), HMIBASE_TO_STRING_VW(&widget)));

         Candera::Layouter::SetMargin(node, margin);
         invalidateLayout = true;
      }

      // set also horizontal and vertical alignments in case the user forgot to do it in scene composer
      if (Candera::Layouter::GetHorizontalAlignment(node) != Candera::HLeft)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "UpdateNode horizontalAlignment=HLeft %s", HMIBASE_TO_STRING_VW(&widget)));

         Candera::Layouter::SetHorizontalAlignment(node, Candera::HLeft);
         invalidateLayout = true;
      }
      if (Candera::Layouter::GetVerticalAlignment(node) != Candera::VTop)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "UpdateNode verticalAlignment=VTop %s", HMIBASE_TO_STRING_VW(&widget)));

         Candera::Layouter::SetVerticalAlignment(node, Candera::VTop);
         invalidateLayout = true;
      }
   }

   if (isResizeEnabled)
   {
      if (Candera::Layouter::GetMinimumSize(node) != widget.GetMinimumSize())
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "UpdateNode minSize=%12s %s",
                             HMIBASE_TO_STRING(widget.GetMinimumSize()),
                             HMIBASE_TO_STRING_VW(&widget)));

         Candera::Layouter::SetMinimumSize(node, widget.GetMinimumSize());
         invalidateLayout = true;
      }

      if (Candera::Layouter::GetMaximumSize(node) != widget.GetMaximumSize())
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "UpdateNode maxSize=%12s %s",
                             HMIBASE_TO_STRING(widget.GetMaximumSize()),
                             HMIBASE_TO_STRING_VW(&widget)));

         Candera::Layouter::SetMaximumSize(node, widget.GetMaximumSize());
         invalidateLayout = true;
      }

      const Candera::Vector2& size = data.InternalSize;
      if (Candera::Layouter::GetSize(node) != size)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "UpdateNode size=%12s %s",
                             HMIBASE_TO_STRING(size),
                             HMIBASE_TO_STRING_VW(&widget)));

         Candera::Layouter::SetSize(node, size);
         invalidateLayout = true;
      }
   }

   // invalidate layout if necessary
   if (invalidateLayout)
   {
      Candera::Layouter::InvalidateLayout(&node);
   }
}


/*****************************************************************************/
void DefaultGizmoController2D::updateGizmo(GizmoWidget2D& widget, DefaultGizmoController2DData& data)
{
   FeatStd::Optional<Candera::Color> color(getGizmoNodesColor(widget, data));

#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
   if (color.IsSet())
#else
   if (color == true)
#endif
   {
      updateGizmoNodesColor(widget, data, *color);
   }
}


/*****************************************************************************/
FeatStd::Optional<Candera::Color> DefaultGizmoController2D::getGizmoNodesColor(GizmoWidget2D& widget, DefaultGizmoController2DData& data)
{
   // color at index 2 is used for warning appearance (notice that warning is not set by this controller so it is require to extend this class to show warning)
   if (data.IsWarning && (widget.GetColors().GetCount() > 2))
   {
      return FeatStd::Optional<Candera::Color>(widget.GetColors().Get(2));
   }
   // color at index 1 is used for pressed appearance
   else if (data.IsPressed && (widget.GetColors().GetCount() > 1))
   {
      return FeatStd::Optional<Candera::Color>(widget.GetColors().Get(1));
   }
   // color specified as widget property
   else if (widget.GetColors().GetCount() > widget.GetColorIndex())
   {
      return FeatStd::Optional<Candera::Color>(widget.GetColors().Get(widget.GetColorIndex()));
   }
   else
   {
      return FeatStd::Optional<Candera::Color>();
   }
}


/*****************************************************************************/
void DefaultGizmoController2D::updateGizmoNodesColor(GizmoWidget2D& widget, DefaultGizmoController2DData&, const Candera::Color& color)
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "UpdateGizmoNodesColor color=%24s %s",
                       HMIBASE_TO_STRING(color),
                       HMIBASE_TO_STRING_VW(&widget)));

   //set color on all gizmo nodes
   for (FeatStd::UInt i = 0; i < widget.GetNodes().GetCount(); ++i)
   {
      Candera::Node2D* node = widget.GetNodes().Get(i);
      if (node != NULL)
      {
         hmibase::widget::utils::EffectUtils::setColor(node, color);
      }
   }
}


/*****************************************************************************/
void DefaultGizmoController2D::updateWidget(GizmoWidget2D& widget, DefaultGizmoController2DData& data)
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "UpdateWidget position=%12s size=%12s rotation=%.3f %s",
                       HMIBASE_TO_STRING(data.InternalPosition),
                       HMIBASE_TO_STRING(data.InternalSize),
                       data.RotationFactor,
                       HMIBASE_TO_STRING_VW(&widget)));

   widget.SetPosition(data.InternalPosition);
   widget.SetSize(data.InternalSize);
   widget.SetRotation(data.InternalRotation);
}


/*****************************************************************************/
bool DefaultGizmoController2D::OnGesture(DelegateWidget& delegateWidget, const hmibase::input::gesture::GestureEvent& gestureData)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnGesture gesture=%100s %s",
                       HMIBASE_TO_STRING(gestureData),
                       HMIBASE_TO_STRING_VW(&delegateWidget)));

   bool consumed = false;
   GizmoWidget2D* widget = Candera::Dynamic_Cast<GizmoWidget2D*>(&delegateWidget);
   DefaultGizmoController2DData* data = Candera::Dynamic_Cast<DefaultGizmoController2DData*>(delegateWidget.GetControllerData());
   if ((widget != NULL) && (data != NULL))
   {
      switch (gestureData._gestureState)
      {
         case hmibase::input::gesture::GestureEvent::ET_START:
            onGestureBeforeStart(*widget, *data, gestureData);
            consumed = Base::OnGesture(delegateWidget, gestureData);
            onGestureAfterStart(*widget, *data, gestureData);
            break;
         case hmibase::input::gesture::GestureEvent::ET_END:
            onGestureBeforeEnd(*widget, *data, gestureData);
            consumed = Base::OnGesture(delegateWidget, gestureData);
            onGestureAfterEnd(*widget, *data, gestureData);
            break;
         case hmibase::input::gesture::GestureEvent::ET_ABORT:
            onGestureBeforeAbort(*widget, *data, gestureData);
            consumed = Base::OnGesture(delegateWidget, gestureData);
            onGestureAfterAbort(*widget, *data, gestureData);
            break;
         default:
            consumed = Base::OnGesture(delegateWidget, gestureData);
            break;
      }
   }

   return consumed;
}


/*****************************************************************************/
void DefaultGizmoController2D::onGestureBeforeStart(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData)
{
   if (widget.GetNode() == NULL)
   {
      return;
   }

   Candera::Vector2 pt1(static_cast<FeatStd::Float>(gestureData._pt1.x), static_cast<FeatStd::Float>(gestureData._pt1.y));

   data.MinSize = widget.GetMinimumSize();
   data.MaxSize = widget.GetMaximumSize();
   data.LimitArea = widget.GetLimitArea();

   data.IsPressed = true;
   data.StartPt1 = pt1;
   data.StartGestureData = gestureData;

   Candera::Rectangle boundingRectangle;
   widget.GetNode()->GetEffectiveBoundingRectangle(boundingRectangle);

   // when drag gesture starts remember the position, size and rotation from the node
   data.StartTopLeft = widget.GetNode()->GetPosition();
   data.StartBottomRight = widget.GetNode()->GetPosition() + boundingRectangle.GetSize();
   data.StartRotation = widget.GetNode()->GetRotation();

   data.TopLeftFactor.SetZero();
   data.BottomRightFactor.SetZero();
   data.RotationFactor = 0.0f;

   data.ActiveGestures.set(static_cast<size_t>(gestureData._gestureType));
}


/*****************************************************************************/
void DefaultGizmoController2D::onGestureAfterStart(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData)
{
   EditUpdEvent editUpdEvent(&widget, gestureData, data.EditMode, EditUpdEvent::Start, widget.GetPosition(), widget.GetSize(), widget.GetRotation());
   GetEventSource().DispatchEvent(editUpdEvent);

   if ((widget.GetNode() != NULL) && (getEditingRenderOrderRank() != 0))
   {
      widget.GetNode()->SetRenderOrderRank(getEditingRenderOrderRank());
   }
}


/*****************************************************************************/
void DefaultGizmoController2D::onGestureBeforeEnd(GizmoWidget2D& widget, DefaultGizmoController2DData& /*data*/, const hmibase::input::gesture::GestureEvent& /*gestureData*/)
{
   if ((widget.GetNode() != NULL) && (getEditingRenderOrderRank() != 0))
   {
      widget.GetNode()->SetRenderOrderRank(0);
   }
}


/*****************************************************************************/
void DefaultGizmoController2D::onGestureAfterEnd(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData)
{
   data.ActiveGestures.reset(static_cast<size_t>(gestureData._gestureType));

   data.IsPressed = data.ActiveGestures.any();
   if (!data.IsPressed)
   {
      data.StartPt1.SetZero();
      data.StartGestureData = hmibase::input::gesture::GestureEvent();
      data.EditMode.reset();
      data.FullEditMode.reset();

      if (_autoUpdateWidget)
      {
         updateWidget(widget, data);
      }

      EditUpdEvent editUpdEvent(&widget, gestureData, data.EditMode, EditUpdEvent::End, data.InternalPosition, data.InternalSize, data.InternalRotation);
      GetEventSource().DispatchEvent(editUpdEvent);
   }
}


/*****************************************************************************/
void DefaultGizmoController2D::onGestureBeforeAbort(GizmoWidget2D& /*widget*/, DefaultGizmoController2DData& /*data*/, const hmibase::input::gesture::GestureEvent& /*gestureData*/)
{
}


/*****************************************************************************/
void DefaultGizmoController2D::onGestureAfterAbort(GizmoWidget2D& /*widget*/, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData)
{
   data.ActiveGestures.reset(static_cast<size_t>(gestureData._gestureType));

   data.IsPressed = data.ActiveGestures.any();
   if (!data.IsPressed)
   {
      data.StartPt1.SetZero();
      data.StartGestureData = hmibase::input::gesture::GestureEvent();
      data.EditMode.reset();
      data.FullEditMode.reset();
   }
}


/*****************************************************************************/
void DefaultGizmoController2D::onGestureMove(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const Candera::Vector2& delta)
{
   Candera::Node2D* node = widget.GetNode();
   if (node != NULL)
   {
      Candera::Vector2 topLeftDelta = delta * data.TopLeftFactor;
      Candera::Vector2 bottomRightDelta = delta * data.BottomRightFactor;
      FeatStd::Float rotationDelta = delta.GetY() * data.RotationFactor;

      //if any transformation property changed, invalidate the widget and post the update message
      if ((topLeftDelta != Candera::Vector2()) || (bottomRightDelta != Candera::Vector2()) || (rotationDelta != 0.0f))
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnGestureMove minSize=%12s maxSize=%12s limitArea=%s",
                             HMIBASE_TO_STRING(data.MinSize),
                             HMIBASE_TO_STRING(data.MaxSize),
                             HMIBASE_TO_STRING(data.LimitArea)));

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnGestureMove topLeftDelta=%12s bottomRightDelta=%12s rotDelta=%f",
                             HMIBASE_TO_STRING(topLeftDelta),
                             HMIBASE_TO_STRING(bottomRightDelta),
                             rotationDelta));

         Candera::Rectangle limitArea(data.LimitArea);

         //adjust limit area to prevent invalid movement during resize
         if (data.TopLeftFactor.GetX() == 0.0f)
         {
            limitArea.SetLeft(data.StartTopLeft.GetX());
         }
         if ((data.BottomRightFactor.GetX() == 0.0f) && (data.MaxSize.GetX() > 0.0f))
         {
            limitArea.SetLeft(data.StartBottomRight.GetX() - data.MaxSize.GetX());
            limitArea.SetWidth(data.MaxSize.GetX());
         }

         if (data.TopLeftFactor.GetY() == 0.0f)
         {
            limitArea.SetTop(data.StartTopLeft.GetY());
         }
         if ((data.BottomRightFactor.GetY() == 0.0f) && (data.MaxSize.GetY() > 0.0f))
         {
            limitArea.SetTop(data.StartBottomRight.GetY() - data.MaxSize.GetY());
            limitArea.SetHeight(data.MaxSize.GetY());
         }

         //calculate new position and size
         Candera::Vector2 position(data.StartTopLeft + topLeftDelta);
         Candera::Vector2 size(data.StartBottomRight + bottomRightDelta - position);

         //adjust new position and size considering limit area, min size and max size
         DefaultBoundsLimiter(limitArea, data.MinSize, data.MaxSize).limit(position, size);

         //store new position
         if (data.TopLeftFactor.GetX() != 0.0f)
         {
            data.InternalPosition.SetX(position.GetX());
         }
         if (data.TopLeftFactor.GetY() != 0.0f)
         {
            data.InternalPosition.SetY(position.GetY());
         }

         //store new size
         if ((data.TopLeftFactor.GetX() != 0.0f) || (data.BottomRightFactor.GetX() != 0.0f))
         {
            data.InternalSize.SetX(size.GetX());
         }
         if ((data.TopLeftFactor.GetY() != 0.0f) || (data.BottomRightFactor.GetY() != 0.0f))
         {
            data.InternalSize.SetY(size.GetY());
         }

         //store new rotation
         data.InternalRotation = static_cast<FeatStd::Float>(static_cast<FeatStd::Int>((data.StartRotation + rotationDelta) * 1000.0f) % 360000) / 1000.0f;

         widget.Invalidate();

         postUpdateMessage(widget, data, false);
      }
   }
}


/*****************************************************************************/
bool DefaultGizmoController2D::onDragGestureStart(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData)
{
   Candera::Vector2 pt1(static_cast<FeatStd::Float>(gestureData._pt1.x), static_cast<FeatStd::Float>(gestureData._pt1.y));

   data.EditMode.reset();
   data.FullEditMode.reset();

   // check which gizmo was interacted with
   // multiple gizmo nodes can be touched
   for (FeatStd::UInt i = 0; i < widget.GetGizmoEditModes().GetCount(); ++i)
   {
      FeatStd::UInt8 configEditMode = widget.GetGizmoEditModes().Get(i);
      if (isNodeIntersecting(widget, static_cast<size_t>(i), pt1))
      {
         data.FullEditMode.set(static_cast<size_t>(configEditMode));
         switch (configEditMode)
         {
            case enGizmoEditMode::DragTranslateX:
               data.TopLeftFactor.SetX(1.0f);
               data.BottomRightFactor.SetX(1.0f);
               data.EditMode.set(static_cast<size_t>(DefaultGizmoController2DData::enTranslate));
               break;

            case enGizmoEditMode::_DragTranslateXNegativ:
               data.TopLeftFactor.SetX(-1.0f);
               data.BottomRightFactor.SetX(-1.0f);
               data.EditMode.set(static_cast<size_t>(DefaultGizmoController2DData::enTranslate));
               break;

            case enGizmoEditMode::DragTranslateY:
               data.TopLeftFactor.SetY(1.0f);
               data.BottomRightFactor.SetY(1.0f);
               data.EditMode.set(static_cast<size_t>(DefaultGizmoController2DData::enTranslate));
               break;

            case enGizmoEditMode::_DragTranslateYNegativ:
               data.TopLeftFactor.SetY(-1.0f);
               data.BottomRightFactor.SetY(-1.0f);
               data.EditMode.set(static_cast<size_t>(DefaultGizmoController2DData::enTranslate));
               break;

            case enGizmoEditMode::DragResizeRight:
               data.BottomRightFactor.SetX(1.0f);
               data.EditMode.set(static_cast<size_t>(DefaultGizmoController2DData::enResize));
               break;

            case enGizmoEditMode::DragResizeLeft:
               data.TopLeftFactor.SetX(1.0f);
               data.EditMode.set(static_cast<size_t>(DefaultGizmoController2DData::enResize));
               break;

            case enGizmoEditMode::DragResizeBottom:
               data.BottomRightFactor.SetY(1.0f);
               data.EditMode.set(static_cast<size_t>(DefaultGizmoController2DData::enResize));
               break;

            case enGizmoEditMode::DragResizeTop:
               data.TopLeftFactor.SetY(1.0f);
               data.EditMode.set(static_cast<size_t>(DefaultGizmoController2DData::enResize));
               break;

            case enGizmoEditMode::DragRotateClockwise:
               data.RotationFactor = 1.0f;
               data.EditMode.set(static_cast<size_t>(DefaultGizmoController2DData::enRotate));
               break;

            case enGizmoEditMode::DragRotateAntiClockwise:
               data.RotationFactor = -1.0f;
               data.EditMode.set(static_cast<size_t>(DefaultGizmoController2DData::enRotate));
               break;

            default:
               break;
         }
      }
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnDragGestureStart startTopLeft=%12s topLeftFactor=%12s startBottomRight=%12s bottomRightFactor=%12s rotFactor=%.3f",
                       HMIBASE_TO_STRING(data.StartTopLeft),
                       HMIBASE_TO_STRING(data.TopLeftFactor),
                       HMIBASE_TO_STRING(data.StartBottomRight),
                       HMIBASE_TO_STRING(data.BottomRightFactor),
                       data.RotationFactor));

   widget.Invalidate();

   return data.EditMode.any();
}


/*****************************************************************************/
bool DefaultGizmoController2D::onDragGestureMove(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData)
{
   Candera::Vector2 pt1(static_cast<FeatStd::Float>(gestureData._pt1.x), static_cast<FeatStd::Float>(gestureData._pt1.y));
   Candera::Vector2 startPt1(static_cast<FeatStd::Float>(data.StartGestureData._pt1.x), static_cast<FeatStd::Float>(data.StartGestureData._pt1.y));

   // on drag move if the widget is pressed, calculate new values for position, size and rotation depending on delta value
   bool consumed = false;
   if (data.IsPressed)
   {
      Candera::Vector2 delta = pt1 - startPt1;
      onGestureMove(widget, data, delta);

      EditUpdEvent editUpdEvent(&widget, gestureData, data.EditMode, EditUpdEvent::Move, data.InternalPosition, data.InternalSize, data.InternalRotation);
      GetEventSource().DispatchEvent(editUpdEvent);

      consumed = true;
   }
   return consumed;
}


/*****************************************************************************/
bool DefaultGizmoController2D::onDragGestureEnd(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& /*gestureData*/)
{
   widget.Invalidate();
   postUpdateMessage(widget, data, true);

   return true;
}


/*****************************************************************************/
bool DefaultGizmoController2D::onDragGestureAbort(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& /*gestureData*/)
{
   widget.Invalidate();
   postUpdateMessage(widget, data, true);

   return true;
}


/*****************************************************************************/
bool DefaultGizmoController2D::OnDragGesture(DelegateWidget& baseWidget, const hmibase::input::gesture::GestureEvent& gestureData)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnDragGesture gesture=%100s %s",
                       HMIBASE_TO_STRING(gestureData),
                       HMIBASE_TO_STRING_VW(&baseWidget)));

   bool consumed = false;
   GizmoWidget2D* widget = Candera::Dynamic_Cast<GizmoWidget2D*>(&baseWidget);
   DefaultGizmoController2DData* data = dynamic_cast<DefaultGizmoController2DData*>(baseWidget.GetControllerData());
   if ((widget != NULL) && (widget->GetNode() != NULL) && (data != NULL))
   {
      switch (gestureData._gestureState)
      {
         case hmibase::input::gesture::GestureEvent::ET_START:
            consumed = onDragGestureStart(*widget, *data, gestureData);
            break;
         case hmibase::input::gesture::GestureEvent::ET_MOVE:
            consumed = onDragGestureMove(*widget, *data, gestureData);
            break;
         case hmibase::input::gesture::GestureEvent::ET_END:
            consumed = onDragGestureEnd(*widget, *data, gestureData);
            break;
         case hmibase::input::gesture::GestureEvent::ET_ABORT:
            consumed = onDragGestureAbort(*widget, *data, gestureData);
            break;
         default:
            break;
      }
   }

   return consumed;
}


/*****************************************************************************/
bool DefaultGizmoController2D::onPinchSpreadGestureStart(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData)
{
   Candera::Vector2 pt1(static_cast<FeatStd::Float>(gestureData._pt1.x), static_cast<FeatStd::Float>(gestureData._pt1.y));
   Candera::Vector2 pt2(static_cast<FeatStd::Float>(gestureData._pt2.x), static_cast<FeatStd::Float>(gestureData._pt2.y));

   data.EditMode.reset();
   data.FullEditMode.reset();

   //check which gizmo was interacted with
   // multiple gizmo nodes can be touched
   for (FeatStd::UInt i = 0; i < widget.GetGizmoEditModes().GetCount(); ++i)
   {
      Candera::UInt8 configEditMode = widget.GetGizmoEditModes().Get(i);
      if (isNodeIntersecting(widget, static_cast<size_t>(i), pt1) && isNodeIntersecting(widget, static_cast<size_t>(i), pt2))
      {
         data.FullEditMode.set(static_cast<size_t>(configEditMode));
         switch (configEditMode)
         {
            case enGizmoEditMode::PinchSpreadX:
               data.TopLeftFactor.SetX(-0.5f);
               data.BottomRightFactor.SetX(0.5f);
               data.EditMode.set(static_cast<size_t>(DefaultGizmoController2DData::enTranslate));
               data.EditMode.set(static_cast<size_t>(DefaultGizmoController2DData::enResize));
               break;

            case enGizmoEditMode::PinchSpreadY:
               data.TopLeftFactor.SetY(-0.5f);
               data.BottomRightFactor.SetY(0.5f);
               data.EditMode.set(static_cast<size_t>(DefaultGizmoController2DData::enTranslate));
               data.EditMode.set(static_cast<size_t>(DefaultGizmoController2DData::enResize));
               break;

            default:
               break;
         }
      }
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnPinchSpreadGestureStart startTopLeft=%12s topLeftFactor=%12s startBottomRight=%12s bottomRightFactor=%12s rotFactor=%.3f",
                       HMIBASE_TO_STRING(data.StartTopLeft),
                       HMIBASE_TO_STRING(data.TopLeftFactor),
                       HMIBASE_TO_STRING(data.StartBottomRight),
                       HMIBASE_TO_STRING(data.BottomRightFactor),
                       data.RotationFactor));

   widget.Invalidate();

   return data.EditMode.any();
}


/*****************************************************************************/
bool DefaultGizmoController2D::onPinchSpreadGestureMove(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData)
{
   Candera::Vector2 pt1(static_cast<FeatStd::Float>(gestureData._pt1.x), static_cast<FeatStd::Float>(gestureData._pt1.y));
   Candera::Vector2 pt2(static_cast<FeatStd::Float>(gestureData._pt2.x), static_cast<FeatStd::Float>(gestureData._pt2.y));
   Candera::Vector2 startPt1(static_cast<FeatStd::Float>(data.StartGestureData._pt1.x), static_cast<FeatStd::Float>(data.StartGestureData._pt1.y));
   Candera::Vector2 startPt2(static_cast<FeatStd::Float>(data.StartGestureData._pt2.x), static_cast<FeatStd::Float>(data.StartGestureData._pt2.y));

   // on PinchSpread move if the widget is pressed, calculate new values for position, size and rotation depending on delta value
   bool consumed = false;
   if (data.IsPressed)
   {
      //it is spread if the distance between pt1 and pt2 at start of the gesture is lower than the it is currently
      bool isSpread = (startPt1 - startPt2).GetSquaredLength() < (pt1 - pt2).GetSquaredLength();

      Candera::Vector2 delta = (pt1 - startPt1) + (startPt2 - pt2);
      if (isSpread != (delta.GetX() > 0))
      {
         delta.SetX(-delta.GetX());
      }
      if (isSpread != (delta.GetY() > 0))
      {
         delta.SetY(-delta.GetY());
      }

      onGestureMove(widget, data, delta);
      consumed = true;
   }
   return consumed;
}


/*****************************************************************************/
bool DefaultGizmoController2D::onPinchSpreadGestureEnd(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& /*gestureData*/)
{
   widget.Invalidate();
   postUpdateMessage(widget, data, true);

   return true;
}


/*****************************************************************************/
bool DefaultGizmoController2D::onPinchSpreadGestureAbort(GizmoWidget2D& widget, DefaultGizmoController2DData& data, const hmibase::input::gesture::GestureEvent& /*gestureData*/)
{
   widget.Invalidate();
   postUpdateMessage(widget, data, true);

   return true;
}


/*****************************************************************************/
bool DefaultGizmoController2D::OnPinchSpreadGesture(DelegateWidget& baseWidget, const hmibase::input::gesture::GestureEvent& gestureData)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnPinchSpreadGesture gesture=%100s %s",
                       HMIBASE_TO_STRING(gestureData),
                       HMIBASE_TO_STRING_VW(&baseWidget)));

   bool consumed = false;
   GizmoWidget2D* widget = Candera::Dynamic_Cast<GizmoWidget2D*>(&baseWidget);
   DefaultGizmoController2DData* data = dynamic_cast<DefaultGizmoController2DData*>(baseWidget.GetControllerData());
   if ((widget != NULL) && (widget->GetNode() != NULL) && (data != NULL))
   {
      switch (gestureData._gestureState)
      {
         case hmibase::input::gesture::GestureEvent::ET_START:
            data->ActiveGestures.set(static_cast<size_t>(gestureData._gestureType));
            consumed = onPinchSpreadGestureStart(*widget, *data, gestureData);
            break;
         case hmibase::input::gesture::GestureEvent::ET_MOVE:
            consumed = onPinchSpreadGestureMove(*widget, *data, gestureData);
            break;
         case hmibase::input::gesture::GestureEvent::ET_END:
            data->ActiveGestures.reset(static_cast<size_t>(gestureData._gestureType));
            consumed = onPinchSpreadGestureEnd(*widget, *data, gestureData);
            break;
         case hmibase::input::gesture::GestureEvent::ET_ABORT:
            data->ActiveGestures.reset(static_cast<size_t>(gestureData._gestureType));
            consumed = onPinchSpreadGestureAbort(*widget, *data, gestureData);
            break;
         default:
            break;
      }
   }

   return consumed;
}


/*****************************************************************************/
void DefaultGizmoController2D::checkEditModes(GizmoWidget2D& widget, bool& isTranslateEnabled, bool& isResizeEnabled, bool& isRotateEnabled) const
{
   isTranslateEnabled = false;
   isResizeEnabled = false;
   isRotateEnabled = false;

   for (FeatStd::UInt i = 0; i < widget.GetGizmoEditModes().GetCount(); ++i)
   {
      switch (widget.GetGizmoEditModes().Get(i))
      {
         case enGizmoEditMode::DragTranslateX:
         case enGizmoEditMode::_DragTranslateXNegativ:
         case enGizmoEditMode::DragTranslateY:
         case enGizmoEditMode::_DragTranslateYNegativ:
            isTranslateEnabled = true;
            break;

         case enGizmoEditMode::DragResizeRight:
         case enGizmoEditMode::DragResizeLeft:
         case enGizmoEditMode::DragResizeBottom:
         case enGizmoEditMode::DragResizeTop:
            isResizeEnabled = true;
            break;

         case enGizmoEditMode::DragRotateClockwise:
         case enGizmoEditMode::DragRotateAntiClockwise:
            isRotateEnabled = true;
            break;

         case enGizmoEditMode::PinchSpreadX:
         case enGizmoEditMode::PinchSpreadY:
            isTranslateEnabled = true;
            isResizeEnabled = true;
            break;

         default:
            break;
      }
   }
}


/*****************************************************************************/
bool DefaultGizmoController2D::isNodeIntersecting(GizmoWidget2D& widget, size_t index, const Candera::Vector2& p)
{
   if (static_cast<FeatStd::UInt>(index) < widget.GetNodes().GetCount())
   {
      Candera::Node2D* node = widget.GetNodes().Get(static_cast<FeatStd::UInt>(index));
      if ((node != NULL) && (node->IsEffectiveRenderingEnabled()))
      {
         Candera::Rectangle rect;
         node->GetWorldAxisAlignedBoundingRectangle(rect);
         if (rect.Contains(p))
         {
            return true;
         }
      }
   }
   return false;
}


/*****************************************************************************/
void DefaultGizmoController2D::postUpdateMessage(GizmoWidget2D& widget, DefaultGizmoController2DData& data, bool completed)
{
   Courier::View* view = widget.GetParentView();
   if (view != NULL)
   {
      const FeatStd::Char* widgetName = widget.GetLegacyName();
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PostUpdateMessage pos=%12s size=%12s rot=%f completed=%u %s",
                          HMIBASE_TO_STRING(data.InternalPosition),
                          HMIBASE_TO_STRING(data.InternalSize),
                          data.InternalRotation,
                          completed,
                          HMIBASE_TO_STRING_VW(&widget)));

      GizmoUpdMsg* msg = COURIER_MESSAGE_NEW(GizmoUpdMsg)(view->GetId(), Courier::Identifier(widgetName), widget.GetUserData(), data.InternalPosition, data.InternalSize, data.InternalRotation, completed);
      if (msg != NULL)
      {
         msg->Post();
      }
   }
}


/*****************************************************************************/
FeatStd::Optional<WidgetGestureConfig> DefaultGizmoController2D::GetDefaultGestureConfig(const DelegateWidget& /*widget*/)
{
   //by default it supports Drag and PinchSpread gestures
   return FeatStd::Optional<WidgetGestureConfig>(WidgetGestureConfig(WidgetGestureConfig::Tap(false), WidgetGestureConfig::Drag(true), WidgetGestureConfig::Swipe(false), WidgetGestureConfig::PinchSpread(true)));
}


/*****************************************************************************/
DefaultBoundsLimiter::DefaultBoundsLimiter(const Candera::Rectangle& limitArea, const Candera::Vector2& minSize, const Candera::Vector2& maxSize)
   : _limitArea(limitArea), _minSize(minSize), _maxSize(maxSize)
{
}


/*****************************************************************************/
void DefaultBoundsLimiter::limit(Candera::Vector2& position, Candera::Vector2& size)
{
   //horizontal
   {
      FeatStd::Float x = position.GetX();
      FeatStd::Float width = size.GetX();

      limitSize(width, _minSize.GetX(), _maxSize.GetX());
      limitPosition(x, width, _limitArea.GetLeft(), _limitArea.GetLeft() + _limitArea.GetWidth());

      position.SetX(x);
      size.SetX(width);
   }

   //vertical
   {
      FeatStd::Float y = position.GetY();
      FeatStd::Float height = size.GetY();

      limitSize(height, _minSize.GetY(), _maxSize.GetY());
      limitPosition(y, height, _limitArea.GetTop(), _limitArea.GetTop() + _limitArea.GetHeight());

      position.SetY(y);
      size.SetY(height);
   }
}


}
}


}
