/* ***************************************************************************************
* FILE:          Focusable2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Focusable2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Focusable2D.h"
#include "FocusUtils2D.h"
#include "Focus/FCommon.h"
#include "Focus/FConfigInfo.h"
#include "Focus/FDataSet.h"
#include "Focus/FData.h"
#include "Focus/FManager.h"
#include "Focus/Default/FDefaultAnimationManager.h"
#include "Widgets/2D/WidgetFinder2D.h"
#include "Widgets/2D/Button/ButtonWidget2D.h"


#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_FOCUS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/Focusable2D.cpp.trc.h"
#endif


Focusable2DBase::Focusable2DBase() :
   _FocusableStatus(false),
   _FocusOrder(FocusOrderType()),
   _FocusParentNode(NULL),
   _FocusControllerSet(FocusControllerSetType(0))
{
}


Focusable2DBase::~Focusable2DBase()
{
   _FocusParentNode = NULL;
}


void Focusable2DBase::CloneFocusableFrom(const Focusable2DBase& original, ControlTemplateMap& controlTemplateMap)
{
   SetFocusableStatus(original.GetFocusableStatus());
   SetFocusOrder(original.GetFocusOrder());
   SetFocusParentNode(controlTemplateMap.ResolveNodeClone(original.GetFocusParentNode()));
   SetFocusControllerSet(original.GetFocusControllerSet());
}


void Focusable2DBase::UpdateFocusable(BaseWidget2D& widget)
{
   if (GetFocusableStatus() || widget.IsFocused())
   {
      bool isFocused = FocusUtils::isFocused(widget);
      if (isFocused != widget.IsFocused())
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Update new focus info isFocused=%u %s", isFocused, HMIBASE_TO_STRING_VW(&widget)));
         widget.SetFocused(isFocused);
      }
   }
}


static bool IsNodeAncestorOf(const Candera::Node2D* parent, Candera::Node2D* child)
{
   if (parent != NULL)
   {
      while (child != NULL)
      {
         child = child->GetParent();
         if (child == parent)
         {
            return true;
         }
      }
   }
   return false;
}


void Focusable2DBase::ConfigureFocusControllers(BaseWidget2D& widget, const Focus::FManager& focusManager, Focus::FSession& session, Focus::FWidgetConfig& handle, Focus::FControllerSetId controllerSetId)
{
   const Focus::FControllerList* controllerSet = focusManager.getControllerSet(static_cast<Focus::FControllerSetId>(controllerSetId));
   if ((controllerSet != NULL) && (controllerSet->count() > 0))
   {
      for (size_t i = 0; i < controllerSet->count(); ++i)
      {
         Focus::FController* controller = (*controllerSet)[i];
         if (controller != NULL)
         {
            controller->configureWidget(session, handle, widget);
         }
      }
   }
}


void Focusable2DBase::RegisterToFocusManager(BaseWidget2D& widget)
{
   Courier::View* view = widget.GetParentView();
   if ((view != NULL) && GetFocusableStatus() && (widget.GetNode() != NULL) && (widget.GetNode()->IsEffectiveRenderingEnabled()))
   {
      Focus::FManager& focusManager = Focus::FManager::getInstance();

      Focus::FSession* session = focusManager.getSession();
      Focus::FWidgetConfig* handle = focusManager.getHandle(view->GetId(), widget.GetLegacyName());
      if ((handle != NULL) && (session != NULL))
      {
         RegisterToFocusManagerImpl(focusManager, *session, *handle);

         if (_FocusControllerSet > 0)
         {
            ConfigureFocusControllers(widget, focusManager, *session, *handle, static_cast<unsigned short>(_FocusControllerSet));     // Using FocusControllerSetType (short) instead of FControllerSetId (unsigned int) is OK
         }                                                                                                                    // The cast to unsigned is OK -> >0 is checked before
      }
   }
}


void Focusable2DBase::DefaultRegisterToFocusManagerImpl(BaseWidget2D& widget, Focus::FManager& focusManager, Focus::FSession& session, Focus::FWidgetConfig& handle)
{
   (void)focusManager;

   Focus::FWidgetData* widgetData = handle.getOrCreateData<Focus::FWidgetData>();
   if (widgetData != NULL)
   {
      widgetData->Enabled = widget.IsEnabled();
      widgetData->Order = GetFocusOrder();
      widgetData->SequenceNr = static_cast<int>(session.generateSequenceNr());
      if (_FocusControllerSet > 0)
      {
         widgetData->ControllerSetId = static_cast<Focus::FControllerSetId>(static_cast<int>(_FocusControllerSet));
      }
   }

   const Focus::FListItemData* listItemInfo = widget.GetListItemInfo();
   if (listItemInfo != NULL)
   {
      handle.setData(*listItemInfo);
   }

   Candera::Node2D* node = widget.GetNode();
   if (node != NULL)
   {
      Focus::FRectangle* rectangle = handle.getOrCreateData<Focus::FRectangle>();
      if (rectangle != NULL)
      {
         node->GetEffectiveBoundingRectangle(*rectangle);
         Candera::Vector2 worldPosition(node->GetWorldPosition());
         rectangle->SetLeft(worldPosition.GetX());
         rectangle->SetTop(worldPosition.GetY());
      }
   }

   Candera::Node2D* groupNode = GetFocusParentNode();
   if (IsNodeAncestorOf(node, groupNode))
   {
      groupNode = NULL;
   }
   if ((groupNode == NULL) && (node != NULL))
   {
      groupNode = node->GetParent();
   }

   if (groupNode != NULL)
   {
      BaseWidget2D* focusGroupBase = WidgetFinder::FindAncestorWidget<BaseWidget2D>(WidgetFinder::GetSceneContext(&widget), groupNode, FocusUtils::isFocusGroup);   //lint !e740
      if (focusGroupBase != NULL)
      {
         Focus::FWidgetConfig* focusGroupHandle = FocusUtils::getFocusableHandle(*focusGroupBase);
         if ((&handle != focusGroupHandle) && (focusGroupHandle != NULL))
         {
            handle.setParent(focusGroupHandle);
         }
      }
   }
}
