/* ***************************************************************************************
* FILE:          FocusGroupWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FocusGroupWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "FocusGroupWidget2D.h"
#include "Focus/FCommon.h"
#include "Focus/FConfigInfo.h"
#include "Focus/FDataSet.h"
#include "Focus/FData.h"
#include "Focus/FManager.h"
#include "Focus/Default/FDefaultAnimationManager.h"
#include "Widgets/2D/WidgetFinder2D.h"
#include "Widgets/2D/Button/ButtonWidget2D.h"


#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_FOCUS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/FocusGroupWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(FocusGroupWidget2D)


FocusGroupBase::FocusGroupBase() : _focusMarkerNode(FocusMarkerNodeType())
{
   //these are the values shown in SceneComposer and should not be affected by runtime configuration in the application
   _groupData.WrapAround = false;
   _groupData.PreserveFocus = true;
}


FocusGroupBase::~FocusGroupBase()
{
   _focusMarkerNode = FocusMarkerNodeType();
}


void FocusGroupBase::registerGroupToFocusManager(Focus::FManager& /*focusManager*/, Focus::FSession& /*session*/, Focus::FWidgetConfig& handle)
{
   if (getGroupConfigured())
   {
      handle.setData(_groupData);

      if (getFocusMarkerNode() != NULL)
      {
         handle.setData(Focus::FMarkerNodeData(true));
      }
   }
   else
   {
      handle.setData(Focus::FGroupData());
   }
}


void FocusGroupBase::updateFocusGroup(BaseWidget2D& widget, Focusable2DBase& focusable)
{
   bool isFocused = false;

   //group is focusable or focused already
   if (focusable.GetFocusableStatus() || widget.IsFocused())
   {
      Focus::FManager& focusManager = Focus::FManager::getInstance();
      const Focus::FAppCurrentFocusInfo& currentFocusInfo = focusManager.getCurrentFocus();

      //current focus matches parent view
      Focus::FAppStateSharedPtr appState = focusManager.getCurrentAppState();
      if ((widget.GetParentView() != NULL) && !appState.PointsToNull()
            && (currentFocusInfo.ViewId == widget.GetParentView()->GetId())
            && (currentFocusInfo.ViewId != Focus::Constants::InvalidViewId)
            && (currentFocusInfo.WidgetId != Focus::Constants::InvalidId))
      {
         Focus::FViewState* viewState = appState->getChild(widget.GetParentView()->GetId());
         if (viewState != NULL)
         {
            Focus::FGroupState* groupState = viewState->getChild(widget.GetLegacyName());
            if (groupState != NULL)
            {
               isFocused = (groupState->CurrentFocusInfo.Id == currentFocusInfo.WidgetId);
            }
         }
      }
   }
   if (isFocused != widget.IsFocused())
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Update new focus info isFocused=%u %s", isFocused, HMIBASE_TO_STRING_VW(&widget)));
      widget.SetFocused(isFocused);
      widget.Invalidate();
   }
}


FocusGroupWidget2D::FocusGroupWidget2D() :
   BaseWidget2D(),
   FocusGroupBase()
{
   SetFocusableStatus(true);
}


FocusGroupWidget2D::~FocusGroupWidget2D()
{
}


bool FocusGroupWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const FocusGroupWidget2D* original = CLONEABLE_WIDGET_CAST<const FocusGroupWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      setGroupConfigured(original->getGroupConfigured());
      setGroupWrapAround(original->getGroupWrapAround());
      setGroupPreserveFocus(original->getGroupPreserveFocus());
      setGroupDefaultOrder(original->getGroupDefaultOrder());
      setGroupLayer(original->getGroupLayer());
      setFocusMarkerNode(controlTemplateMap.ResolveNodeClone(original->getFocusMarkerNode()));
      setGroupAnchorMode(original->getGroupAnchorMode());
      setGroupAnchorId(original->getGroupAnchorId());

      CloneFocusableFrom(*original, controlTemplateMap);

      cloned = true;
   }
   return cloned;
}


bool FocusGroupWidget2D::OnMessage(const Message& msg)
{
   if (Base::OnMessage(msg))
   {
      return true;
   }

   bool consumed = false;
   switch (msg.GetId())
   {
      case RegisterFocusableWidgetsReqMsg::ID:
         RegisterToFocusManager(*this);
         break;

      default:
         break;
   }

   return consumed;
}


void FocusGroupWidget2D::Update()
{
   Base::Update();
   FocusGroupBase::updateFocusGroup(*this, *this);
}


void FocusGroupWidget2D::RegisterToFocusManagerImpl(Focus::FManager& focusManager, Focus::FSession& session, Focus::FWidgetConfig& handle)
{
   DefaultRegisterToFocusManagerImpl(*this, focusManager, session, handle);
   FocusGroupBase::registerGroupToFocusManager(focusManager, session, handle);
}
