/* ***************************************************************************************
* FILE:          EffectControlWidget2D.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  EffectControlWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef EffectControl_WIDGET2D_H
#define EffectControl_WIDGET2D_H

#include <bitset>
#include <Widgets/2D/EffectControl/generated/EffectControlWidget2DBase.h>
#include <Widgets/2D/ControlTemplate/ControlTemplateCloneableWidget.h>

/* ***************************************************************************************
* Provides control over an effect allowing to change its properties by using data binding sources or to expose them outside composites.
* There is no internal storage in the widget for the property values, they are set/retrieved directly on/from the effect.
*************************************************************************************** */
class EffectControlWidget2D : public EffectControlWidget2DBase
   IMPLEMENTS_CLONEABLE_WIDGET
{
      enum PropertiesEnum
      {
         ////Blend
         //enColorBlendFactorSrc,
         //enColorBlendFactorDst,
         //enColorBlendOperation,
         //enAlphaBlendFactorSrc,
         //enAlphaBlendFactorDst,
         //enAlphaBlendOperation,

         ////Bitmap
         //enImage,
         //enFilter,
         //enMipMapFilter,

         ////SolidColor
         //enSolidFillColor,
         //enSolidSize,

         //Blur
         enBlurFilterSize,

         //Color
         enColor,

         //DropShadow
         enDropShadowEnabled,
         enDropShadowColor,
         enDropShadowLightAngle,
         enDropShadowDistance,
         enDropShadowScale,
         //enDropShadowBlurFilter,

         //Flip
         enFlipH,
         enFlipV,
         //enFlipUseNodeLayoutDirection,

         //RadialGradient
         enGradientSize,
         enGradientCenter,
         enGradientDirection,
         enGradientMagnitude,
         enGradientCenterColor,
         enGradientRadialColor,
         enGradientPositiveColor,
         enGradientNegativeColor,

         //Mask
         enMaskNode,
         enMaskImage,

         //Mirror
         enMirrorAxisFrom,
         enMirrorAxisTo,
         enMirrorAlpha,

         //Outline
         enOutlineColor,
         enOutlineWidth,

         //Shadow
         enShadowColor,
         enShadowScale,
         enShadowOffset,

         //Shear
         enShearAngleX,
         enShearAngleY,

         enPropertyCount
      };

      typedef std::bitset<static_cast<int>(EffectControlWidget2D::enPropertyCount)> PropertyConfig;

   public:

      CGI_WIDGET_RTTI_DECLARATION(EffectControlWidget2D, EffectControlWidget2DBase);

      EffectControlWidget2D();
      virtual ~EffectControlWidget2D();

      virtual bool CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap);
      virtual void Update() override;

#define DefEffectControlProperty(propType, propName) \
      typedef propType propName##Type; \
      void set##propName(const propName##Type& value); \
      propName##Type get##propName() const;\
      void set##propName##Enabled(bool value) { _config.set(static_cast<size_t>(EffectControlWidget2D::en##propName), value); } \
      bool is##propName##Enabled() const { return _config.test(static_cast<size_t>(EffectControlWidget2D::en##propName)); }\
      bool is##propName##EnabledVisible() const { return isEnabledPropertyVisible(EffectControlWidget2D::en##propName); }

      //Blur
      DefEffectControlProperty(Candera::UInt8, BlurFilterSize)

      //Color
      DefEffectControlProperty(Candera::Color, Color)

      //DropShadow
      DefEffectControlProperty(bool, DropShadowEnabled)
      DefEffectControlProperty(Candera::Color, DropShadowColor)
      DefEffectControlProperty(Candera::Int, DropShadowLightAngle)
      DefEffectControlProperty(Candera::UInt, DropShadowDistance)
      DefEffectControlProperty(Candera::UInt, DropShadowScale)

      //Flip
      DefEffectControlProperty(bool, FlipH)
      DefEffectControlProperty(bool, FlipV)

      //Gradient
      DefEffectControlProperty(Candera::Vector2, GradientSize)
      DefEffectControlProperty(Candera::Vector2, GradientCenter)
      DefEffectControlProperty(Candera::Vector2, GradientDirection)
      DefEffectControlProperty(Candera::Float, GradientMagnitude)
      DefEffectControlProperty(Candera::Color, GradientCenterColor)
      DefEffectControlProperty(Candera::Color, GradientRadialColor)
      DefEffectControlProperty(Candera::Color, GradientPositiveColor)
      DefEffectControlProperty(Candera::Color, GradientNegativeColor)

      //Mask
      DefEffectControlProperty(Candera::Node2D*, MaskNode)
      DefEffectControlProperty(Candera::MemoryManagement::SharedPointer<Candera::Image2D> , MaskImage)

      //Mirror
      DefEffectControlProperty(Candera::Vector2, MirrorAxisFrom)
      DefEffectControlProperty(Candera::Vector2, MirrorAxisTo)
      DefEffectControlProperty(Candera::Float, MirrorAlpha)

      //Outline
      DefEffectControlProperty(Candera::Color, OutlineColor)
      DefEffectControlProperty(Candera::UInt8, OutlineWidth)

      //Shadow
      DefEffectControlProperty(Candera::Color, ShadowColor)
      DefEffectControlProperty(Candera::Vector2, ShadowScale)
      DefEffectControlProperty(Candera::Vector2, ShadowOffset)

      //Shear
      DefEffectControlProperty(Candera::Float, ShearAngleX)
      DefEffectControlProperty(Candera::Float, ShearAngleY)

#undef DefEffectControlProperty

   private:
#define CdaEffectControlEnabledProperty(propName, propCategory) \
      CdaProperty(propName##Enabled, bool, is##propName##Enabled, set##propName##Enabled)\
         CdaDescription("Enables configuration of the property " #propName)\
         CdaCategory(propCategory)\
         CdaVisibilityTest(is##propName##EnabledVisible)\
      CdaPropertyEnd()

#define CdaEffectControlProperty(propName, propCategory) \
      CdaBindableProperty(propName, propName##Type, get##propName, set##propName)\
         CdaDescription("See Effect's property " #propName)\
         CdaCategory(propCategory)\
         CdaVisibilityTest(is##propName##Enabled)\
      CdaBindablePropertyEnd()

#define CdaEffectControlNonBindableProperty(propName, propCategory) \
      CdaProperty(propName, propName##Type, get##propName, set##propName)\
         CdaDescription("See Effect's property " #propName)\
         CdaCategory(propCategory)\
         CdaVisibilityTest(is##propName##Enabled)\
      CdaPropertyEnd()

      CdaWidget2DDef(EffectControlWidget2D, EffectControlWidget2DBase)
      CdaProperties()

      //Blur
      CdaEffectControlEnabledProperty(BlurFilterSize, "Blur")
      CdaEffectControlProperty(BlurFilterSize, "Blur")

      //Color
      CdaEffectControlEnabledProperty(Color, "Color")
      CdaEffectControlProperty(Color, "Color")

      //DropShadow
      CdaEffectControlEnabledProperty(DropShadowEnabled, "DropShadow")
      CdaEffectControlProperty(DropShadowEnabled, "DropShadow")
      CdaEffectControlEnabledProperty(DropShadowColor, "DropShadow")
      CdaEffectControlProperty(DropShadowColor, "DropShadow")
      CdaEffectControlEnabledProperty(DropShadowLightAngle, "DropShadow")
      CdaEffectControlProperty(DropShadowLightAngle, "DropShadow")
      CdaEffectControlEnabledProperty(DropShadowDistance, "DropShadow")
      CdaEffectControlProperty(DropShadowDistance, "DropShadow")
      CdaEffectControlEnabledProperty(DropShadowScale, "DropShadow")
      CdaEffectControlProperty(DropShadowScale, "DropShadow")

      //Flip
      CdaEffectControlEnabledProperty(FlipH, "Flip")
      CdaEffectControlProperty(FlipH, "Flip")
      CdaEffectControlEnabledProperty(FlipV, "Flip")
      CdaEffectControlProperty(FlipV, "Flip")

      //Gradient
      CdaEffectControlEnabledProperty(GradientSize, "Gradient")
      CdaEffectControlProperty(GradientSize, "Gradient")
      CdaEffectControlEnabledProperty(GradientCenter, "Gradient")
      CdaEffectControlProperty(GradientCenter, "Gradient")
      CdaEffectControlEnabledProperty(GradientDirection, "Gradient")
      CdaEffectControlProperty(GradientDirection, "Gradient")
      CdaEffectControlEnabledProperty(GradientMagnitude, "Gradient")
      CdaEffectControlProperty(GradientMagnitude, "Gradient")
      CdaEffectControlEnabledProperty(GradientCenterColor, "Gradient")
      CdaEffectControlProperty(GradientCenterColor, "Gradient")
      CdaEffectControlEnabledProperty(GradientRadialColor, "Gradient")
      CdaEffectControlProperty(GradientRadialColor, "Gradient")
      CdaEffectControlEnabledProperty(GradientPositiveColor, "Gradient")
      CdaEffectControlProperty(GradientPositiveColor, "Gradient")
      CdaEffectControlEnabledProperty(GradientNegativeColor, "Gradient")
      CdaEffectControlProperty(GradientNegativeColor, "Gradient")

      //Mask
      CdaEffectControlEnabledProperty(MaskNode, "Mask")
      CdaEffectControlNonBindableProperty(MaskNode, "Mask")
      CdaEffectControlEnabledProperty(MaskImage, "Mask")
      CdaEffectControlProperty(MaskImage, "Mask")

      //Mirror
      CdaEffectControlEnabledProperty(MirrorAxisFrom, "Mirror")
      CdaEffectControlProperty(MirrorAxisFrom, "Mirror")
      CdaEffectControlEnabledProperty(MirrorAxisTo, "Mirror")
      CdaEffectControlProperty(MirrorAxisTo, "Mirror")
      CdaEffectControlEnabledProperty(MirrorAlpha, "Mirror")
      CdaEffectControlProperty(MirrorAlpha, "Mirror")

      //Outline
      CdaEffectControlEnabledProperty(OutlineColor, "Outline")
      CdaEffectControlProperty(OutlineColor, "Outline")
      CdaEffectControlEnabledProperty(OutlineWidth, "Outline")
      CdaEffectControlProperty(OutlineWidth, "Outline")

      //Shadow
      CdaEffectControlEnabledProperty(ShadowColor, "Shadow")
      CdaEffectControlProperty(ShadowColor, "Shadow")
      CdaEffectControlEnabledProperty(ShadowScale, "Shadow")
      CdaEffectControlProperty(ShadowScale, "Shadow")
      CdaEffectControlEnabledProperty(ShadowOffset, "Shadow")
      CdaEffectControlProperty(ShadowOffset, "Shadow")

      //Shear
      CdaEffectControlEnabledProperty(ShearAngleX, "Shear")
      CdaEffectControlProperty(ShearAngleX, "Shear")
      CdaEffectControlEnabledProperty(ShearAngleY, "Shear")
      CdaEffectControlProperty(ShearAngleY, "Shear")

      CdaPropertiesEnd()
      CdaWidgetDefEnd()

#undef CdaEffectControlEnabledProperty
#undef CdaEffectControlProperty
#undef CdaEffectControlNonBindableProperty

      Candera::Effect2D* getCombinedEffect() const;
      bool isEnabledPropertyVisible(EffectControlWidget2D::PropertiesEnum propId) const;

      PropertyConfig _config;
      bool _invalid;
};


#endif
