/* ***************************************************************************************
* FILE:          EffectControlWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  EffectControlWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "EffectControlWidget2D.h"

#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlDropShadowBitmapBrushBlend.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlDropShadowTextBrushBlend.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlFlipEffect.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlLinearGradientBrush.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlRadialGradientBrush.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlOutlineEffect.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlOutlineBitmapBrushColorBlend.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlOutlineTextBrushBlend.h>
#include <CanderaPlatform/Device/Common/Effects/BlurEffect.h>
#include <CanderaPlatform/Device/Common/Effects/ColorEffect.h>
#include <CanderaPlatform/Device/Common/Effects/GlMaskEffect.h>
#include <CanderaPlatform/Device/Common/Effects/MirrorEffect.h>
#include <CanderaPlatform/Device/Common/Effects/ShadowEffect.h>
#include <CanderaPlatform/Device/Common/Effects/ShearEffect.h>

#include <Widgets/utils/EffectUtils.h>
typedef hmibase::widget::utils::EffectUtils EffectUtils;

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_NODECONTROL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/EffectControlWidget2D.cpp.trc.h"
#endif
#define IS_TRACE_ACTIVE etg_bIsTraceActive(APP_TRACECLASS_ID(), 5)


CGI_WIDGET_RTTI_DEFINITION(EffectControlWidget2D);


/*****************************************************************************/
static void TraceSetter(const EffectControlWidget2D& widget, const char* propName, const char* propValue)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Set %25s=%75s %s", propName, propValue, HMIBASE_TO_STRING_VW(&widget)));
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Set node=%s", HMIBASE_TO_STRING_SPN2D(widget.GetNode())));
}


/*****************************************************************************/
EffectControlWidget2D::EffectControlWidget2D() : Base(), _invalid(true)
{
}


/*****************************************************************************/
EffectControlWidget2D::~EffectControlWidget2D()
{
}


/*****************************************************************************/
void EffectControlWidget2D::Update()
{
   Base::Update();

   if (_invalid)
   {
      _invalid = false;
      Invalidate();
   }
}


/*****************************************************************************/
class DummyEffect : public Candera::CombinedEffect2D
{
      virtual void Render(Candera::SurfaceHandle, const Candera::Rectangle&, const Candera::Matrix3x2&, const Candera::Node2D&, Candera::ContextHandle2D, Candera::Rectangle&)
      {
      }

      virtual Candera::Effect2D::SharedPointer Clone() const
      {
         return Candera::Effect2D::SharedPointer();
      }
};


/*****************************************************************************/
template <typename TEffect>
class EffectControlWrapper : public TEffect
{
   public:
      static TEffect& getDefaultInstance()
      {
         static EffectControlWrapper _effect;
         return _effect;
      }
};


/*****************************************************************************/
Candera::Effect2D* EffectControlWidget2D::getCombinedEffect() const
{
   Candera::RenderNode* node = Candera::Dynamic_Cast<Candera::RenderNode*>(GetNode());

   if (node != NULL)
   {
      Candera::Effect2D* effect = node->GetEffect(0);
      if (effect != NULL)
      {
         return effect;
      }
   }

   return NULL;
}


/*****************************************************************************/
#define DefEffectControlCopyProp(propName)\
 if (is##propName##Enabled())\
   {\
      set##propName(original->get##propName());\
   }


/*****************************************************************************/
#define DefEffectControlSetPropertyImpl(effectType, effectProp, propName)\
   void EffectControlWidget2D::set##propName(const propName##Type& value)\
   {\
      if ((GetNode() != NULL) && is##propName##Enabled())\
      {\
         effectType* effect = EffectUtils::getEffect<effectType>(getCombinedEffect());\
         if (effect != NULL)\
         {\
            if (IS_TRACE_ACTIVE)\
            {\
               TraceSetter(*this, #propName, HMIBASE_TO_STRING(value));\
            }\
            effect->effectProp().Set(value);\
            _invalid = true;\
            triggerUpdate();\
         }\
      }\
   }

/*****************************************************************************/
#define DefEffectControlGetPropertyImpl(effectType, effectProp, propName)\
   EffectControlWidget2D::propName##Type EffectControlWidget2D::get##propName() const\
   {\
      Candera::Effect2D* combinedEffect = getCombinedEffect();\
      if (combinedEffect == NULL)\
      {\
         return propName##Type(EffectControlWrapper<effectType>::getDefaultInstance().effectProp().Get());\
      }\
      effectType* effect = EffectUtils::getEffect<effectType>(combinedEffect);\
      return (effect != NULL) ? propName##Type(effect->effectProp().Get()) : propName##Type();\
   }

/*****************************************************************************/
#define DefEffectControlPropertyImpl(effectType, effectProp, propName)\
   DefEffectControlSetPropertyImpl(effectType, effectProp, propName)\
   DefEffectControlGetPropertyImpl(effectType, effectProp, propName)

/*****************************************************************************/
#define DefEffectControlSetPropertyImpl2(effectType, effectType2, effectProp, propName)\
   void EffectControlWidget2D::set##propName(const propName##Type& value)\
   {\
      if ((GetNode() != NULL) && is##propName##Enabled())\
      {\
         effectType* effect = EffectUtils::getEffect<effectType>(getCombinedEffect());\
         if (effect != NULL)\
         {\
            if (IS_TRACE_ACTIVE)\
            {\
               TraceSetter(*this, #propName, HMIBASE_TO_STRING(value));\
            }\
            effect->effectProp().Set(value);\
            _invalid = true;\
            triggerUpdate();\
         }\
         else\
         {\
            effectType2* effect2 = EffectUtils::getEffect<effectType2>(getCombinedEffect());\
            if (effect2 != NULL)\
            {\
               if (IS_TRACE_ACTIVE)\
               {\
                  TraceSetter(*this, #propName, HMIBASE_TO_STRING(value));\
               }\
               effect2->effectProp().Set(value);\
               _invalid = true;\
               triggerUpdate();\
            }\
         }\
      }\
   }

/*****************************************************************************/
#define DefEffectControlGetPropertyImpl2(effectType, effectType2, effectProp, propName)\
   EffectControlWidget2D::propName##Type EffectControlWidget2D::get##propName() const\
   {\
      Candera::Effect2D* combinedEffect = getCombinedEffect();\
      if (combinedEffect == NULL)\
      {\
         return EffectControlWrapper<effectType>::getDefaultInstance().effectProp().Get();\
      }\
      effectType* effect = EffectUtils::getEffect<effectType>(combinedEffect);\
      if (effect != NULL) return propName##Type(effect->effectProp().Get());\
      effectType2* effect2 = EffectUtils::getEffect<effectType2>(combinedEffect);\
      return (effect2 != NULL) ? propName##Type(effect2->effectProp().Get()) : propName##Type();\
   }

/*****************************************************************************/
#define DefEffectControlPropertyImpl2(effectType, effectType2, effectProp, propName)\
   DefEffectControlSetPropertyImpl2(effectType, effectType2, effectProp, propName)\
   DefEffectControlGetPropertyImpl2(effectType, effectType2, effectProp, propName)


/*****************************************************************************/
DefEffectControlPropertyImpl(Candera::BlurEffect, FilterSize, BlurFilterSize)

DefEffectControlPropertyImpl(Candera::ColorEffect, Color, Color)

DefEffectControlPropertyImpl(Candera::GlDropShadowEffect, ShadowEnabled, DropShadowEnabled)
DefEffectControlPropertyImpl(Candera::GlDropShadowEffect, ShadowColor, DropShadowColor)
DefEffectControlPropertyImpl(Candera::GlDropShadowEffect, LightAngle, DropShadowLightAngle)
DefEffectControlPropertyImpl(Candera::GlDropShadowEffect, Distance, DropShadowDistance)
DefEffectControlPropertyImpl(Candera::GlDropShadowEffect, ShadowScale, DropShadowScale)

DefEffectControlPropertyImpl(Candera::GlFlipEffect, FlipHorizontal, FlipH)
DefEffectControlPropertyImpl(Candera::GlFlipEffect, FlipVertical, FlipV)

DefEffectControlPropertyImpl2(Candera::GlRadialGradientBrush, Candera::GlLinearGradientBrush, Size, GradientSize)
DefEffectControlPropertyImpl2(Candera::GlRadialGradientBrush, Candera::GlLinearGradientBrush, GradientCenter, GradientCenter)
DefEffectControlPropertyImpl2(Candera::GlRadialGradientBrush, Candera::GlLinearGradientBrush, GradientMagnitude, GradientMagnitude)

DefEffectControlPropertyImpl(Candera::GlRadialGradientBrush, CenterColor, GradientCenterColor)
DefEffectControlPropertyImpl(Candera::GlRadialGradientBrush, RadialColor, GradientRadialColor)
DefEffectControlPropertyImpl(Candera::GlLinearGradientBrush, GradientDirection, GradientDirection)
DefEffectControlPropertyImpl(Candera::GlLinearGradientBrush, PositiveColor, GradientPositiveColor)
DefEffectControlPropertyImpl(Candera::GlLinearGradientBrush, NegativeColor, GradientNegativeColor)

DefEffectControlPropertyImpl(Candera::GlMaskEffect, MaskNode, MaskNode)
DefEffectControlPropertyImpl(Candera::GlMaskEffect, Mask, MaskImage)

DefEffectControlPropertyImpl(Candera::MirrorEffect, MirrorAxisFrom, MirrorAxisFrom)
DefEffectControlPropertyImpl(Candera::MirrorEffect, MirrorAxisTo, MirrorAxisTo)
DefEffectControlPropertyImpl(Candera::MirrorEffect, Alpha, MirrorAlpha)

DefEffectControlPropertyImpl(Candera::GlOutlineEffect, OutlineColor, OutlineColor)
DefEffectControlPropertyImpl(Candera::GlOutlineEffect, OutlineWidth, OutlineWidth)

DefEffectControlPropertyImpl(Candera::ShadowEffect, ShadowColor, ShadowColor)
DefEffectControlPropertyImpl(Candera::ShadowEffect, ShadowScale, ShadowScale)
DefEffectControlPropertyImpl(Candera::ShadowEffect, ShadowPositionOffset, ShadowOffset)

DefEffectControlPropertyImpl(Candera::ShearEffect, ShearAngleX, ShearAngleX)
DefEffectControlPropertyImpl(Candera::ShearEffect, ShearAngleY, ShearAngleY)


/*****************************************************************************/
bool EffectControlWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   if (!Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      return false;
   }

   const EffectControlWidget2D* original = CLONEABLE_WIDGET_CAST<const EffectControlWidget2D*>(originalWidget);
   if (original == NULL)
   {
      return false;
   }

   _config = original->_config;

   DefEffectControlCopyProp(BlurFilterSize);

   DefEffectControlCopyProp(Color);

   DefEffectControlCopyProp(DropShadowEnabled);
   DefEffectControlCopyProp(DropShadowColor);
   DefEffectControlCopyProp(DropShadowLightAngle);
   DefEffectControlCopyProp(DropShadowDistance);
   DefEffectControlCopyProp(DropShadowScale);

   DefEffectControlCopyProp(FlipH);
   DefEffectControlCopyProp(FlipV);

   DefEffectControlCopyProp(GradientSize);
   DefEffectControlCopyProp(GradientCenter);
   DefEffectControlCopyProp(GradientDirection);
   DefEffectControlCopyProp(GradientMagnitude);
   DefEffectControlCopyProp(GradientCenterColor);
   DefEffectControlCopyProp(GradientRadialColor);
   DefEffectControlCopyProp(GradientPositiveColor);
   DefEffectControlCopyProp(GradientNegativeColor);

   DefEffectControlCopyProp(MaskNode);
   DefEffectControlCopyProp(MaskImage);

   DefEffectControlCopyProp(MirrorAxisFrom);
   DefEffectControlCopyProp(MirrorAxisTo);
   DefEffectControlCopyProp(MirrorAlpha);

   DefEffectControlCopyProp(OutlineColor);
   DefEffectControlCopyProp(OutlineWidth);

   DefEffectControlCopyProp(ShadowColor);
   DefEffectControlCopyProp(ShadowScale);
   DefEffectControlCopyProp(ShadowOffset);

   DefEffectControlCopyProp(ShearAngleX);
   DefEffectControlCopyProp(ShearAngleX);

   return true;
}


/*****************************************************************************/
bool EffectControlWidget2D::isEnabledPropertyVisible(EffectControlWidget2D::PropertiesEnum propId) const
{
   Candera::Effect2D* effect = getCombinedEffect();

   switch (propId)
   {
      case EffectControlWidget2D::enBlurFilterSize:
         return EffectUtils::getEffect<Candera::BlurEffect>(effect) != NULL;

      case EffectControlWidget2D::enColor:
         return EffectUtils::getEffect<Candera::ColorEffect>(effect) != NULL;

      case EffectControlWidget2D::enDropShadowEnabled:
      case EffectControlWidget2D::enDropShadowColor:
      case EffectControlWidget2D::enDropShadowLightAngle:
      case EffectControlWidget2D::enDropShadowDistance:
      case EffectControlWidget2D::enDropShadowScale:
         return EffectUtils::getEffect<Candera::GlDropShadowEffect>(effect) != NULL;

      case EffectControlWidget2D::enFlipH:
      case EffectControlWidget2D::enFlipV:
         return EffectUtils::getEffect<Candera::GlFlipEffect>(effect) != NULL;

      case EffectControlWidget2D::enGradientCenterColor:
      case EffectControlWidget2D::enGradientRadialColor:
         return EffectUtils::getEffect<Candera::GlRadialGradientBrush>(effect) != NULL;

      case EffectControlWidget2D::enGradientDirection:
      case EffectControlWidget2D::enGradientPositiveColor:
      case EffectControlWidget2D::enGradientNegativeColor:
         return EffectUtils::getEffect<Candera::GlLinearGradientBrush>(effect) != NULL;

      case EffectControlWidget2D::enGradientSize:
      case EffectControlWidget2D::enGradientCenter:
      case EffectControlWidget2D::enGradientMagnitude:
         return EffectUtils::getEffect<Candera::GlRadialGradientBrush>(effect) != NULL
                || EffectUtils::getEffect<Candera::GlLinearGradientBrush>(effect) != NULL;

      case EffectControlWidget2D::enMaskNode:
      case EffectControlWidget2D::enMaskImage:
         return EffectUtils::getEffect<Candera::GlMaskEffect>(effect) != NULL;

      case EffectControlWidget2D::enMirrorAxisFrom:
      case EffectControlWidget2D::enMirrorAxisTo:
      case EffectControlWidget2D::enMirrorAlpha:
         return EffectUtils::getEffect<Candera::MirrorEffect>(effect) != NULL;

      case EffectControlWidget2D::enOutlineColor:
      case EffectControlWidget2D::enOutlineWidth:
         return EffectUtils::getEffect<Candera::GlOutlineEffect>(effect) != NULL;

      case EffectControlWidget2D::enShadowColor:
      case EffectControlWidget2D::enShadowScale:
      case EffectControlWidget2D::enShadowOffset:
         return EffectUtils::getEffect<Candera::ShadowEffect>(effect) != NULL;

      case EffectControlWidget2D::enShearAngleX:
      case EffectControlWidget2D::enShearAngleY:
         return EffectUtils::getEffect<Candera::ShearEffect>(effect) != NULL;

      default:
         return false;
   }
}
