/* ***************************************************************************************
* FILE:          DropDownListWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DropDownListWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "DropDownListWidget2D.h"

#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
#include <Candera/EngineBase/Layout/OverlayLayouter.h>
#else
#include <Candera/Engine2D/Layout/OverlayLayouter.h>
#endif

#include "FeatStd/Util/StaticObject.h"
#include "Courier/Visualization/ViewScene2D.h"
#include "Widgets/2D/ControlTemplate/ControlTemplateBinding.h"
#include "Widgets/2D/ControlTemplate/ControlTemplate.h"
#include "Widgets/2D/DropDownList/generated/DropDownListMessages.h"

#if defined SESA_ARABIC_LAYOUT_FIX
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
#include "Candera/EngineBase/Layout/ArabicLayouterPatch.h"
#else
#include "Candera/Engine2D/Layout/ArabicLayouterPatch.h"
#endif
#endif

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/DropDownListWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(DropDownListWidget2D);

using namespace Candera;
using namespace Courier;

class FlexDropdownListLayouter : public Candera::Layouter
{
   public:
      static FlexDropdownListLayouter* GetInstance();

      static void SetUseListPosition(Candera::Node2D& node, bool value)
      {
         static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(UseListPosition), value));
      }
      static bool IsUseListPositionSet(Candera::Node2D& node)
      {
         return node.IsValueSet(CdaDynamicPropertyInstance(UseListPosition));
      }
      static void ClearUseListPosition(Candera::Node2D& node)
      {
         static_cast<void>(node.ClearValue(CdaDynamicPropertyInstance(UseListPosition)));
      }
      static bool GetUseListPosition(Candera::Node2D& node)
      {
         return node.GetValue(CdaDynamicPropertyInstance(UseListPosition));
      }

      static void SetListPosition(Candera::Node2D& node, Candera::Vector2 value)
      {
         static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(ListPosition), value));
      }
      static bool IsListPositionSet(Candera::Node2D& node)
      {
         return node.IsValueSet(CdaDynamicPropertyInstance(ListPosition));
      }
      static void ClearListPosition(Candera::Node2D& node)
      {
         static_cast<void>(node.ClearValue(CdaDynamicPropertyInstance(ListPosition)));
      }
      static Candera::Vector2 GetListPosition(Candera::Node2D& node)
      {
         return node.GetValue(CdaDynamicPropertyInstance(ListPosition));
      }

      static void SetListSize(Candera::Node2D& node, Candera::Vector2 value)
      {
         static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(ListSize), value));
      }
      static bool IsListSizeSet(Candera::Node2D& node)
      {
         return node.IsValueSet(CdaDynamicPropertyInstance(ListSize));
      }
      static void ClearListSize(Candera::Node2D& node)
      {
         static_cast<void>(node.ClearValue(CdaDynamicPropertyInstance(ListSize)));
      }
      static Candera::Vector2 GetListSize(Candera::Node2D& node)
      {
         return node.GetValue(CdaDynamicPropertyInstance(ListSize));
      }
      virtual void Dispose() { }

#if defined SESA_ARABIC_LAYOUT_FIX
      virtual void OnSetPosition(const AbstractNodePointer& node, const Vector2& position);
#endif

   protected:
      Candera::Vector2 OnMeasure(Candera::Node2D& node, const Candera::Vector2& clientArea);
      void OnArrange(Candera::Node2D& node, const Candera::Rectangle& clientArea);

#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
      virtual Candera::Vector2 OnMeasure(const Candera::AbstractNodePointer& node, const Candera::Vector2& clientArea) override
      {
         return OnMeasure(*(node.ToNode2D()), clientArea);
      }

      virtual void OnArrange(const Candera::AbstractNodePointer& node, const Candera::Rectangle& clientArea) override
      {
         OnArrange(*(node.ToNode2D()), clientArea);
      }

#endif

      virtual Candera::Layouter* Clone() const

      {
         return GetInstance();   // as this is a singleton it will just return the same instance
      }

   private:
      CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::FlexDropdownListLayouter::FlexDropdownListLayouter, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
      FlexDropdownListLayouter() { }

      static const Candera::Vector2& DefaultListSize()
      {
         static Candera::Vector2 g_defaultListSize(-1.0F, -1.0F);
         return g_defaultListSize;
      }

      static const Candera::Vector2& DefaultListPosition()
      {
         static Candera::Vector2 g_defaultListPosition(0.0F, 0.0F);
         return g_defaultListPosition;
      }

      static void SetArrangeHeight(Candera::Node2D& node, Candera::Float value)
      {
         static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(ArrangeHeight), value));
      }
      static bool IsArrangeHeightSet(Candera::Node2D& node)
      {
         return node.IsValueSet(CdaDynamicPropertyInstance(ArrangeHeight));
      }
      static Candera::Float GetArrangeHeight(Candera::Node2D& node)
      {
         return node.GetValue(CdaDynamicPropertyInstance(ArrangeHeight));
      }

      CdaDynamicProperties(FlexDropdownListLayouter, Candera::Layouter);
      CdaDynamicProperty(ArrangeHeight, Candera::Float);
      CdaDynamicPropertyDescription("")
      CdaDynamicPropertyDefaultValue(0.0F)
      CdaDynamicPropertyCategory("Layout")
      CdaDynamicPropertyEnd();
      CdaDynamicProperty(UseListPosition, bool);
      CdaDynamicPropertyDescription("")
      CdaDynamicPropertyDefaultValue(false)
      CdaDynamicPropertyCategory("Layout")
      CdaDynamicPropertyEnd();
      CdaDynamicProperty(ListPosition, Candera::Vector2);
      CdaDynamicPropertyDescription("")
      CdaDynamicPropertyDefaultValue(DefaultListPosition())
      CdaDynamicPropertyCategory("Layout")
      CdaDynamicPropertyEnd();
      CdaDynamicProperty(ListSize, Candera::Vector2);
      CdaDynamicPropertyDescription("")
      CdaDynamicPropertyDefaultValue(DefaultListSize())
      CdaDynamicPropertyCategory("Layout")
      CdaDynamicPropertyEnd();
      CdaDynamicPropertiesEnd();
};


Candera::Vector2 FlexDropdownListLayouter::OnMeasure(Candera::Node2D& node, const Candera::Vector2& clientArea)
{
   Candera::Float width = 0.0F;
   Candera::Float height = 0.0F;
   Candera::Float clientWidth = 0.0F;
   Candera::Float clientHeight = 0.0F;
   if (node.IsEffectiveRenderingEnabled())
   {
      Candera::Vector2 listSize = GetListSize(node);
      clientWidth = (listSize.GetX() >= 0) ? listSize.GetX() : clientArea.GetX();
      clientHeight = (listSize.GetY() >= 0) ? listSize.GetY() : clientArea.GetY();
   }
   SetArrangeHeight(node, clientHeight);
   Candera::Vector2 childArea(clientWidth, clientHeight);
   Candera::Node2D* child = node.GetFirstChild();
   while (child != 0)
   {
      Layouter* childLayouter = child->GetLayouter();
      if (childLayouter != 0)
      {
         childLayouter->Measure(*child, childArea);
         Candera::Vector2 preferredChildSize = GetClientSize(*child);
         if (width < preferredChildSize.GetX())
         {
            width = preferredChildSize.GetX();
         }
         if (height < preferredChildSize.GetY())
         {
            height = preferredChildSize.GetY();
         }
      }
      child = child->GetNextSibling();
   }
   return Candera::Vector2(0.0F, 0.0F);
}


#if defined SESA_ARABIC_LAYOUT_FIX
void FlexDropdownListLayouter::OnArrange(Candera::Node2D& node, const Candera::Rectangle& clientArea)
{
   if (Candera::ArabicLayouterPatch::IsSceneEnabled(node))
   {
      Candera::Float clientWidth = 0.0F;
      Candera::Float clientHeight = 0.0F;
      if (node.IsEffectiveRenderingEnabled())
      {
         Candera::Vector2 listSize = GetListSize(node);
         clientWidth = (listSize.GetX() >= 0) ? listSize.GetX() : clientArea.GetWidth();
         clientHeight = (listSize.GetY() >= 0) ? listSize.GetY() : GetArrangeHeight(node);
      }
      Candera::Node2D* child = node.GetFirstChild();
      while (child != 0)
      {
         Layouter* childLayouter = child->GetLayouter();
         if (childLayouter != 0)
         {
            Candera::Rectangle childArea(0.0F, 0.0F, clientWidth, clientHeight);
            childLayouter->Arrange(*child, childArea);
         }
         child = child->GetNextSibling();
      }
   }
   else
   {
      Candera::Float clientWidth = 0.0F;
      Candera::Float clientHeight = 0.0F;
      if (node.IsEffectiveRenderingEnabled())
      {
         Candera::Vector2 listSize = GetListSize(node);
         clientWidth = (listSize.GetX() >= 0) ? listSize.GetX() : clientArea.GetWidth();
         clientHeight = (listSize.GetY() >= 0) ? listSize.GetY() : GetArrangeHeight(node);
      }
      Candera::Node2D* child = node.GetFirstChild();
      while (child != 0)
      {
         Layouter* childLayouter = child->GetLayouter();
         if (childLayouter != 0)
         {
            Candera::Rectangle childArea(0.0F, 0.0F, clientWidth, clientHeight);
            AlignRectangle(childArea, childArea, *child);
            childLayouter->Arrange(*child, childArea);
         }
         child = child->GetNextSibling();
      }
      if (!GetUseListPosition(node))
      {
         node.SetPosition(0.0F, clientArea.GetPosition().GetY());
      }
   }
}


void FlexDropdownListLayouter::OnSetPosition(const AbstractNodePointer& node, const Candera::Vector2& position)
{
   Node2D* node2D = node.ToNode2D();
   if (node2D != 0)
   {
      if (!GetUseListPosition(*node2D))
      {
         node2D->SetPosition(0.0F, position.GetY());
      }
      else
      {
         if (node2D->GetParent() != 0)
         {
            Candera::Vector2 parentPosition = node2D->GetParent()->GetWorldPosition();
            node2D->SetPosition(GetListPosition(*node2D).GetX() - parentPosition.GetX(), GetListPosition(*node2D).GetY() - parentPosition.GetY());
         }
         else
         {
            ETG_TRACE_ERR(("node2D->GetParent() is NULL"));
         }
      }
   }
}


#else
void FlexDropdownListLayouter::OnArrange(Candera::Node2D& node, const Candera::Rectangle& clientArea)
{
   Candera::Float clientWidth = 0.0F;
   Candera::Float clientHeight = 0.0F;
   if (node.IsEffectiveRenderingEnabled())
   {
      Candera::Vector2 listSize = GetListSize(node);
      clientWidth = (listSize.GetX() >= 0) ? listSize.GetX() : clientArea.GetWidth();
      clientHeight = (listSize.GetY() >= 0) ? listSize.GetY() : GetArrangeHeight(node);
   }
   Candera::Node2D* child = node.GetFirstChild();
   while (child != 0)
   {
      Layouter* childLayouter = child->GetLayouter();
      if (childLayouter != 0)
      {
         Candera::Rectangle childArea(0.0F, 0.0F, clientWidth, clientHeight);
         AlignRectangle(childArea, childArea, *child);
         childLayouter->Arrange(*child, childArea);
      }
      child = child->GetNextSibling();
   }
   if (!GetUseListPosition(node))
   {
      node.SetPosition(0.0F, clientArea.GetPosition().GetY());
   }
}


#endif

FlexDropdownListLayouter* FlexDropdownListLayouter::GetInstance()
{
   FEATSTD_UNSYNCED_STATIC_OBJECT(FlexDropdownListLayouter, s_flexDropdownListLayouter);
   return &s_flexDropdownListLayouter;
}


DropDownListWidget2D::DropDownListWidget2D() :
   _open(false),
   _selected(0),
   _delayUpdateContent(false)
{
   SetListSize(Candera::Vector2(-1.0F, -1.0F));
}


DropDownListWidget2D::~DropDownListWidget2D()
{
}


void DropDownListWidget2D::Finalize()
{
   Base::Finalize();
}


void DropDownListWidget2D::OnChanged(Courier::UInt32 propertyId)
{
   Base::OnChanged(propertyId);
   _itemsInvalid = true;

   switch (propertyId)
   {
      case ListNodePropertyId:
      {
         Node2D* listNode(GetListNode());
         if ((0 != listNode) && (listNode->GetLayouter() != FlexDropdownListLayouter::GetInstance()))
         {
            listNode->SetLayouter(FlexDropdownListLayouter::GetInstance());
         }
         break;
      }
   }

   Invalidate();
}


bool DropDownListWidget2D::OnMessage(const Message& msg)
{
   if (Base::OnMessage(msg))
   {
      return true;
   }

   switch (msg.GetId())
   {
      case DropdownChangeMsg::ID:
      {
         bool result = false;
         bool open = _open;
         const DropdownChangeMsg* ddChangeMsg = message_cast<const DropdownChangeMsg*>(&msg);

         if ((GetListId() != 0u) && (ddChangeMsg->GetListId() == GetListId()))
         {
            result = true;
            switch (ddChangeMsg->GetDropdownChangeType())
            {
               case Candera::DropdownChangeOpen:
                  open = true;
                  break;
               case Candera::DropdownChangeClose:
                  open = false;
                  break;
               case Candera::DropdownChangeToggle:
                  open = !_open;
                  break;
               default:
                  break;
            }
         }
         //close this dropdown without consuming the message
         //this allows us to close all dropdowns with the same message
         else if ((ddChangeMsg->GetListId() == 0u) && (ddChangeMsg->GetDropdownChangeType() == Candera::DropdownChangeClose))
         {
            open = false;
         }

         if (_open != open)
         {
            _open = open;
            OnOpenChanged();
         }
         return result;
      }
      case TouchMsg::ID:
      {
         const TouchMsg* touchMsg = message_cast<const TouchMsg*>(&msg);
         if ((touchMsg->GetState() == TouchMsgState::Up) && (0 != GetParentView()) && GetParentView()->IsActive())
         {
            Courier::ViewScene2D* view2D = static_cast<Courier::ViewScene2D*>(GetParentView());
            const Candera::Vector2 point(static_cast<Courier::Float>(touchMsg->GetXPos()), static_cast<Courier::Float>(touchMsg->GetYPos()));
            const Courier::ViewScene2D::CameraPtrVector& cameras = view2D->GetCameraPtrVector();

            Candera::Node2D* itemsNode = GetItemsNode();
            if (_open && (0 != itemsNode))
            {
               Candera::Node2D* currentItemNode = itemsNode->GetFirstChild();
               Int32 i = Int32(GetStartIndex());
               while (0 != currentItemNode)
               {
                  for (Int cameraIndex = 0; cameraIndex < (Int) cameras.Size(); ++cameraIndex)
                  {
                     Candera::Camera2D* camera2D = cameras[cameraIndex];
                     if (camera2D->IsRenderingEnabled())
                     {
                        if (IsPickIntersectingNode(*camera2D, currentItemNode, point))
                        {
                           DropdownCurrentValueChangedMsg* msg = COURIER_MESSAGE_NEW(DropdownCurrentValueChangedMsg)(GetParentView()->GetId(), Identifier(GetLegacyName()), i);
                           if (0 != msg)
                           {
                              _selected = i;
                              if (ControlTemplateBinding::IsSelectedBindable(*this))
                              {
                                 ControlTemplateBinding::SetSelectedValue(*this, _selected);
                              }
                              static_cast<void>(msg->Post());
                           }
                           _open = !_open;
                           OnOpenChanged();
                           return true;
                        }
                     }
                  }
                  ++i;
                  currentItemNode = currentItemNode->GetNextSibling();
               }
            }

            Candera::Node2D* buttonNode = GetButtonNode();
            if (0 != buttonNode)
            {
               for (Int cameraIndex = 0; cameraIndex < (Int) cameras.Size(); ++cameraIndex)
               {
                  Candera::Camera2D* camera2D = cameras[cameraIndex];
                  if (camera2D->IsRenderingEnabled())
                  {
                     if (IsPickIntersectingNode(*camera2D, buttonNode, point))
                     {
                        _open = !_open;
                        OnOpenChanged();
                        return true;
                     }
                  }
               }
            }

            //if it is open any TouchMsgState::Down should close it
            //however the message should be consumed only if the list was touched (background)
            //if the list was not touched we do not consume the message so that other elements can react on it
            if (_open)
            {
               Candera::Node2D* listNode = GetListNode();
               if (0 != listNode)
               {
                  for (Int cameraIndex = 0; cameraIndex < (Int) cameras.Size(); ++cameraIndex)
                  {
                     Candera::Camera2D* camera2D = cameras[cameraIndex];
                     if (camera2D->IsRenderingEnabled())
                     {
                        if (IsPickIntersectingNode(*camera2D, listNode, point))
                        {
                           _open = false;
                           OnOpenChanged();
                           return true;
                        }
                     }
                  }
               }

               _open = false;
               OnOpenChanged();
               return false;
            }
         }

         return false;
      }

      default:
      {
         //do nothing
      }
   }
   return false;
}


UInt32 DropDownListWidget2D::GetEffectiveTouchPriority() const
{
   return (_open) ? GetTouchPriority() : 0 ;
}


void DropDownListWidget2D::InitWidget()
{
   Base::InitWidget();

   const ListNodeType& listNode = GetListNode();

   if (0 != listNode)
   {
      listNode->SetRenderingEnabled(false);
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "DropDownListWidget2D:ButtonNode= %p", GetButtonNode()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "DropDownListWidget2D:ContentNode= %p", GetContentNode()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "DropDownListWidget2D:ListNode= %p", GetListNode()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "DropDownListWidget2D::UseListPosition= %u", GetUseListPosition()));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "DropDownListWidget2D::ListSize= (%.2f, %.2f)", GetListSize().GetX(), GetListSize().GetY()));
}


void DropDownListWidget2D::PerformUpdate(bool& invalidate, bool& invalidateLayout)
{
   const ListNodeType& listNode = GetListNode();

   if ((0 != listNode) && (0 != GetParentView()))
   {
      if (listNode->IsRenderingEnabled() != _open)
      {
         listNode->SetRenderingEnabled(_open);
         invalidate = true;
         invalidateLayout = true;
         _delayUpdateContent = true;
      }

      Candera::Node2D* itemsGroup = GetContentNode();
      Candera::Node2D* templateGroup = GetTemplateGroup();
      if (_itemsInvalid)
      {
         if (!_delayUpdateContent)
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "DropDownListWidget2D:PerformUpdate Updating content for list id=%d", GetListId()));
            UpdateContent();
         }
         _delayUpdateContent = false;

         invalidate = invalidate || _itemsInvalid;
         invalidateLayout = invalidateLayout || _itemsInvalid;

         if ((0 != itemsGroup) && (0 != templateGroup) && HasItems())
         {
            if (GetUseListPosition())
            {
               FlexDropdownListLayouter::SetUseListPosition(*listNode, GetUseListPosition());
               FlexDropdownListLayouter::SetListPosition(*listNode, GetListPosition());
            }
            else
            {
               FlexDropdownListLayouter::ClearUseListPosition(*listNode);
               FlexDropdownListLayouter::ClearListPosition(*listNode);
            }
            if (FlexDropdownListLayouter::GetListSize(*listNode) != GetListSize())
            {
               FlexDropdownListLayouter::SetListSize(*listNode, GetListSize());
            }
         }
      }
   }

   ValidateDataProvider();
}


bool DropDownListWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const DropDownListWidget2D* original = CLONEABLE_WIDGET_CAST<const DropDownListWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetButtonNode(controlTemplateMap.ResolveNodeClone(original->GetButtonNode()));
      SetContentNode(controlTemplateMap.ResolveNodeClone(original->GetContentNode()));
      SetListNode(controlTemplateMap.ResolveNodeClone(original->GetListNode()));
      SetUseListPosition(original->GetUseListPosition());
      SetListPosition(original->GetListPosition());
      SetListSize(original->GetListSize());
      if (ControlTemplateBinding::IsSelectedBindable(*this))
      {
         _selected = ControlTemplateBinding::GetSelectedValue(*this);
      }

      cloned = true;
   }
   return cloned;
}


void DropDownListWidget2D::OnOpenChanged()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "DropDownListWidget2D:OnOpenChanged id=%d open=%u", GetListId(), _open));

   ControlTemplateInstance* controlTemplateInstance = 0;
   Node2D* node = GetNode();
   while ((0 != node) && (0 == controlTemplateInstance))
   {
      controlTemplateInstance = ControlTemplate::GetControlTemplateInstance(*node);
      node = node->GetParent();
   }
   if (0 != controlTemplateInstance)
   {
      ListWidget2D* list = CLONEABLE_WIDGET_CAST<ListWidget2D*>(controlTemplateInstance->GetOwner());
      if (0 != list)
      {
         list->ResetStatemachineToIdle();
      }
   }
   _itemsInvalid = true;
   Invalidate();//without this Invalidate(), the Update method is not called
}

