/* ***************************************************************************************
* FILE:          DDManager.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DDManager is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include <widget2D_std_if.h>
#include "DDManager.h"
#include "DDAdorner.h"
#include <View/CGI/CgiExtensions/AppViewHandler.h>
#include <Widgets/2D/Touchable2D.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_DRAG_DROP
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include <trcGenProj/Header/DDManager.cpp.trc.h>
#endif

static uint32_t getLocalTraceClass()
{
   return TR_CLASS_HMI_FW_DRAG_DROP;
}


namespace DragDrop {
DDManager& DDManager::getInstance()
{
   static DDManager instance;
   return instance;
}


DDManager::DDManager() :
   _viewHandler(NULL),
   _dragStartTimer(NULL),
   _dragStartTimerTimeout(0),
   _nextSessionId(0),
   _nextRequestId(0),
   _touched(false),
   _acceptedTouch(NULL),
   _wasTouchMoveIgnored(false),
   _ignoredTouchMoveInfo(0, 0, 0, 0, 0),
   _oldAdorner(NULL),
   _adornerManager(NULL)
{
}


DDManager::~DDManager()
{
   _viewHandler = NULL;
   _acceptedTouch = NULL;

   if (_dragStartTimer != NULL)
   {
      _dragStartTimer->stop();
      CANDERA_DELETE(_dragStartTimer);
      _dragStartTimer = NULL;
   }

   if (_oldAdorner != NULL)
   {
      CANDERA_DELETE(_oldAdorner);
      _oldAdorner = NULL;
   }
   _adornerManager = NULL;
}


void DDManager::setDragStartTimerTimeout(FeatStd::UInt32 timeout)
{
   _dragStartTimerTimeout = timeout;
}


void DDManager::setAdornerManager(hmibase::widget::adorner::AdornerManager* adornerManager)
{
   _adornerManager = adornerManager;
}


void DDManager::setViewHandler(AppViewHandler* viewHandler)
{
   _viewHandler = viewHandler;
}


AppViewHandler* DDManager::getViewHandler() const
{
   return _viewHandler;
}


void DDManager::setAcceptedTouch(Touchable2D* widget)
{
   ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::setAcceptedTouch() widget=%s", widget != NULL ? widget->GetLegacyName() : "<null>"));
   _acceptedTouch = widget;
}


Touchable2D* DDManager::getAcceptedTouch() const
{
   return _acceptedTouch;
}


FeatStd::UInt32 DDManager::getNextSessionId()
{
   return _nextSessionId++;
}


FeatStd::UInt32 DDManager::getNextRequestId()
{
   return _nextRequestId++;
}


void DDManager::startDragStartTimer()
{
   if (_dragStartTimerTimeout > 0)
   {
      if (_dragStartTimer == NULL)
      {
         _dragStartTimer = CANDERA_NEW(Util::Timer);
      }
      if (_dragStartTimer != NULL)
      {
         ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::startDragStartTimer() timeout=%u", _dragStartTimerTimeout));

         _pendingOperation.OpType = PendingStartTimer;
         _pendingOperation.ReqId = 0;

         _dragStartTimer->setName("DragDropStartTimer", 0);
         _dragStartTimer->setTimeout(0, _dragStartTimerTimeout);
         _dragStartTimer->start();
      }
   }
}


void DDManager::stopDragStartTimer()
{
   if (_dragStartTimer != NULL)
   {
      _dragStartTimer->stop();
   }
}


bool DDManager::onMessage(const Courier::Message& msg)
{
   bool consumed = false;

   switch (msg.GetId())
   {
      case TimerExpiredMsg::ID:
      {
         const TimerExpiredMsg* timerMsg = Courier::message_cast<const TimerExpiredMsg*>(&msg);
         if ((timerMsg != NULL) && (_dragStartTimer != NULL) && (_dragStartTimer == timerMsg->GetTimer()))
         {
            onDragStartTimerExpired();
            consumed = true;
         }
      }
      break;

      case DragDropCheckSourceResMsg::ID:
      {
         const DragDropCheckSourceResMsg* resMsg = Courier::message_cast<const DragDropCheckSourceResMsg*>(&msg);
         if (resMsg != NULL)
         {
            onCheckSourceRes(resMsg->GetReqId(), resMsg->GetSourceData(), resMsg->GetResult());
            consumed = true;
         }
      }
      break;

      case DragDropCheckDestinationResMsg::ID:
      {
         const DragDropCheckDestinationResMsg* resMsg = Courier::message_cast<const DragDropCheckDestinationResMsg*>(&msg);
         if (resMsg != NULL)
         {
            onCheckDestinationRes(resMsg->GetReqId(), resMsg->GetDestinationData(), resMsg->GetResult());
            consumed = true;
         }
      }
      break;

      case DragDropCompleteResMsg::ID:
      {
         const DragDropCompleteResMsg* resMsg = Courier::message_cast<const DragDropCompleteResMsg*>(&msg);
         if (resMsg != NULL)
         {
            onCompleteRes(resMsg->GetReqId(), resMsg->GetResult());
            consumed = true;
         }
      }
      break;

      case DragDropAbortReqMsg::ID:
      {
         ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onMessage() Abort received. Pending(op=%u, reqId=%u)",
                             _pendingOperation.OpType, _pendingOperation.ReqId));
         cancelOperation();
         consumed = true;
      }
      break;

      default:
         break;
   }

   return consumed;
}


bool DDManager::checkPendingOperation(PendingOperationType opType, FeatStd::UInt32 reqId)
{
   if ((_pendingOperation.OpType != opType) || (_pendingOperation.ReqId != reqId))
   {
      ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::checkPendingOperation() Expected(op=%u, reqId=%u) != Received(op=%u, reqId=%u)",
                          _pendingOperation.OpType, _pendingOperation.ReqId, opType, reqId));

      return false;
   }

   return true;
}


void DDManager::clearPendingOperation()
{
   _pendingOperation = PendingOperation();
}


void DDManager::sendStatusInfo(hmibase::DragDropStatusEnum status)
{
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(_info.SourceWidget);
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(_info.DestinationWidget);
   POST_MSG((COURIER_MESSAGE_NEW(DragDropStatusUpdMsg)(status, _info.SessionId,
             _info.SourceView,
             Courier::Identifier(_info.SourceWidget.GetCString()),
             _info.SourceData,
             _info.DestinationView,
             Courier::Identifier(_info.DestinationWidget.GetCString()),
             _info.DestinationData)));

   SECURE_FEATSTD_STRING_ACCESS_END();
   SECURE_FEATSTD_STRING_ACCESS_END();
}


void DDManager::onDragStartTimerExpired()
{
   ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onDragStartTimerExpired()"));

   if (checkPendingOperation(PendingStartTimer, 0))
   {
      ViewWidgetId viewWidgetId = _pendingOperation.ViewWidget;
      clearPendingOperation();

      sendCheckSourceReq(viewWidgetId);
   }
}


void DDManager::sendCheckSourceReq(const ViewWidgetId& viewWidgetId)
{
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(viewWidgetId.Widget);
   ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::sendCheckSourceReq() widget=%s", viewWidgetId.Widget.GetCString()));
   SECURE_FEATSTD_STRING_ACCESS_END();

   _pendingOperation.OpType = PendingCheckSource;
   _pendingOperation.ReqId = getNextRequestId();
   _pendingOperation.ViewWidget = viewWidgetId;

   SECURE_FEATSTD_STRING_ACCESS_BEGIN(viewWidgetId.Widget);
   POST_MSG((COURIER_MESSAGE_NEW(DragDropCheckSourceReqMsg)(_info.SessionId, _pendingOperation.ReqId,
             viewWidgetId.View,
             Courier::Identifier(viewWidgetId.Widget.GetCString()))));
   SECURE_FEATSTD_STRING_ACCESS_END();
}


void DDManager::onCheckSourceRes(FeatStd::UInt32 reqId, FeatStd::UInt32 sourceData, bool result)
{
   ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onCheckSourceRes() reqId=%u, sourceData=%u, result=%u",
                       reqId, sourceData, result));

   if (checkPendingOperation(PendingCheckSource, reqId))
   {
      //checked source is valid
      if (result)
      {
         _info.SourceView = _pendingOperation.ViewWidget.View;
         _info.SourceWidget = _pendingOperation.ViewWidget.Widget;
         _info.SourceData = sourceData;
         clearPendingOperation();

         showAdorner();

         sendStatusInfo(hmibase::DRAG_DROP_STARTED);

         //if we ignored a touch move while waiting for check source res now is the moment to check it
         if (_wasTouchMoveIgnored)
         {
            _wasTouchMoveIgnored = false;
            if (_touched)
            {
               ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onCheckSourceRes() checking ignored touch move..."));
               checkDestination(_ignoredTouchMoveInfo);
            }
         }
      }
      //invalid source
      else
      {
         endSession();
      }
   }
}


void DDManager::sendCheckDestinationReq(const ViewWidgetId& viewWidgetId)
{
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(viewWidgetId.Widget);
   ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::sendCheckDestinationReq() widget=%s", viewWidgetId.Widget.GetCString()));
   SECURE_FEATSTD_STRING_ACCESS_END();

   _pendingOperation.OpType = PendingCheckDestination;
   _pendingOperation.ReqId = getNextRequestId();
   _pendingOperation.ViewWidget = viewWidgetId;

   SECURE_FEATSTD_STRING_ACCESS_BEGIN(viewWidgetId.Widget);
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(_info.SourceWidget);
   POST_MSG((COURIER_MESSAGE_NEW(DragDropCheckDestinationReqMsg)(_info.SessionId, _pendingOperation.ReqId,
             _info.SourceView,
             Courier::Identifier(_info.SourceWidget.GetCString()),
             _info.SourceData,
             viewWidgetId.View,
             Courier::Identifier(viewWidgetId.Widget.GetCString()))));
   SECURE_FEATSTD_STRING_ACCESS_END();
   SECURE_FEATSTD_STRING_ACCESS_END();
}


void DDManager::onCheckDestinationRes(FeatStd::UInt32 reqId, FeatStd::UInt32 destinationData, bool result)
{
   ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onCheckDestinationRes() reqId=%u, destinationData=%u, result=%u",
                       reqId, destinationData, result));

   if (checkPendingOperation(PendingCheckDestination, reqId))
   {
      //checked destination is valid
      if (result)
      {
         _info.DestinationView = _pendingOperation.ViewWidget.View;
         _info.DestinationWidget = _pendingOperation.ViewWidget.Widget;
         _info.DestinationData = destinationData;
      }
      else
      {
         _info.DestinationView = Courier::ViewId();
         _info.DestinationWidget = "";
         _info.DestinationData = 0;
      }
      clearPendingOperation();

      sendStatusInfo(hmibase::DRAG_DROP_DESTINATION_CHANGED);
   }
}


void DDManager::sendCompleteReq()
{
   ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::sendCompleteReq()"));

   _pendingOperation.OpType = PendingComplete;
   _pendingOperation.ReqId = getNextRequestId();
   _pendingOperation.ViewWidget = ViewWidgetId();

   SECURE_FEATSTD_STRING_ACCESS_BEGIN(_info.SourceWidget);
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(_info.DestinationWidget);
   POST_MSG((COURIER_MESSAGE_NEW(DragDropCompleteReqMsg)(_info.SessionId, _pendingOperation.ReqId,
             _info.SourceView,
             Courier::Identifier(_info.SourceWidget.GetCString()),
             _info.SourceData,
             _info.DestinationView,
             Courier::Identifier(_info.DestinationWidget.GetCString()),
             _info.DestinationData)));
   SECURE_FEATSTD_STRING_ACCESS_END();
   SECURE_FEATSTD_STRING_ACCESS_END();
}


void DDManager::onCompleteRes(FeatStd::UInt32 reqId, bool result)
{
   ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onCompleteRes() reqId=%u, result=%u", reqId, result));

   clearPendingOperation();
   sendStatusInfo(hmibase::DRAG_DROP_COMPLETED);

   endSession();
}


bool DDManager::onTouchPress(Touchable2D& sourceWidget, const Courier::TouchInfo& touchInfo)
{
   ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onTouchPress() x=%u, y=%u, pointer=%u, widget=%s",
                       touchInfo.mX, touchInfo.mY, touchInfo.mPointerId, sourceWidget.GetLegacyName()));

   if ((_acceptedTouch != NULL) && (_acceptedTouch != &sourceWidget))
   {
      ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onTouchPress() source widget (%50s) does not match accepted touch widget (%s)",
                          sourceWidget.GetLegacyName(), _acceptedTouch->GetLegacyName()));
      return false;
   }

   if (touchInfo.mPointerId != 0)
   {
      ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onTouchPress() ignoring touch message because touch pointer is not 0..."));
      return false;
   }

   clearSession();

   Touchable2D* dragDropSourceWidget = NULL;
   if (sourceWidget.GetParentView() != NULL)
   {
      if (sourceWidget.GetDragDropSourceEnabled())
      {
         dragDropSourceWidget = &sourceWidget;
      }
      else
      {
         dragDropSourceWidget = findSourceWidget(touchInfo);
      }

      if ((dragDropSourceWidget != NULL) && (dragDropSourceWidget->GetParentView() != NULL))
      {
         _touched = true;
         initSession();

         createAdorner(*dragDropSourceWidget, touchInfo);

         ViewWidgetId viewWidgetId(dragDropSourceWidget->GetParentView()->GetId(), dragDropSourceWidget->GetLegacyName());
         if (_dragStartTimerTimeout > 0)
         {
            _pendingOperation.ViewWidget = viewWidgetId;

            startDragStartTimer();
         }
         else
         {
            sendCheckSourceReq(viewWidgetId);
         }
      }
      else
      {
         ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onTouchPress() no drag source widget found."));
      }
   }

   return (dragDropSourceWidget != NULL);
}


//lint -esym(1764, sourceWidget) TODO: Remove this line as soon as parameter sourceWidget is used in this function
bool DDManager::onTouchMove(Touchable2D& sourceWidget, const Courier::TouchInfo& touchInfo)
{
   ETG_TRACE_USR4_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onTouchMove() x=%u, y=%u, pointer=%u, widget=%s",
                       touchInfo.mX, touchInfo.mY, touchInfo.mPointerId, sourceWidget.GetLegacyName()));

   if ((_acceptedTouch != NULL) && (_acceptedTouch != &sourceWidget))
   {
      ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onTouchMove() source widget (%50s) does not match accepted touch widget (%s)",
                          sourceWidget.GetLegacyName(), _acceptedTouch->GetLegacyName()));
      return false;
   }

   if (touchInfo.mPointerId != 0)
   {
      ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onTouchMove() ignoring touch message because touch pointer is not 0..."));
      return false;
   }

   if (!_touched)
   {
      ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onTouchMove() ignoring touch message because touched flag is false..."));
      return false;
   }

   moveAdorner(touchInfo);

   if (_pendingOperation.OpType == PendingCheckSource)
   {
      ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onTouchMove() ignoring touch move because CheckSource response is pending..."));
      _wasTouchMoveIgnored = true;
      _ignoredTouchMoveInfo = touchInfo;
      return true;
   }

   if (_pendingOperation.OpType == PendingStartTimer)
   {
      ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onTouchMove() ignoring touch move because DragStartTimer is running..."));
      _wasTouchMoveIgnored = true;
      _ignoredTouchMoveInfo = touchInfo;
      return true;
   }

   _wasTouchMoveIgnored = false;
   checkDestination(touchInfo);
   return true;
}


void DDManager::checkDestination(const Courier::TouchInfo& touchInfo)
{
   Courier::ViewId viewId;
   FeatStd::String widgetId;

   BaseWidget2D* destinationWidget = findDestinationWidget(touchInfo);
   if ((destinationWidget != NULL) && (destinationWidget->GetParentView() != NULL))
   {
      viewId = destinationWidget->GetParentView()->GetId();
      widgetId = destinationWidget->GetLegacyName();
   }

   //different widget
   if ((viewId != _info.DestinationView) || (widgetId != _info.DestinationWidget))
   {
      _info.DestinationView = Courier::ViewId();
      _info.DestinationWidget = "";
      _info.DestinationData = 0;

      if ((_pendingOperation.OpType == PendingCheckDestination)
            && (viewId == _pendingOperation.ViewWidget.View)
            && (widgetId == _pendingOperation.ViewWidget.Widget))
      {
         //already checking this widget
      }
      else if ((viewId != _info.SourceView) || (widgetId != _info.SourceWidget))
      {
         sendCheckDestinationReq(ViewWidgetId(viewId, widgetId));
      }
      else
      {
         sendStatusInfo(hmibase::DRAG_DROP_DESTINATION_CHANGED);
      }
   }
}


bool DDManager::onTouchRelease(Touchable2D& sourceWidget, const Courier::TouchInfo& touchInfo)
{
   ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onTouchRelease() x=%u, y=%u, pointer=%u, widget=%s",
                       touchInfo.mX, touchInfo.mY, touchInfo.mPointerId, sourceWidget.GetLegacyName()));

   if ((_acceptedTouch != NULL) && (_acceptedTouch != &sourceWidget))
   {
      ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onTouchRelease() source widget (%50s) does not match accepted touch widget (%s)",
                          sourceWidget.GetLegacyName(), _acceptedTouch->GetLegacyName()));
      return false;
   }

   if (touchInfo.mPointerId != 0)
   {
      ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onTouchRelease() ignoring touch message because touch pointer is not 0..."));
      return false;
   }

   if (!_touched)
   {
      ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::onTouchRelease() ignoring touch message because touched flag is false..."));
      return false;
   }

   _touched = false;

   hideAdorner();

   //post complete request always to allow proper clean up (NCG3D-51143)
   //if ((_info.DestinationWidget != "") && (_info.SourceWidget != ""))
   {
      sendCompleteReq();
   }
   return true;
}


void DDManager::clearSession()
{
   hideAdorner();
   destroyAdorner();

   stopDragStartTimer();
   _info = DragDropInfo();
   _pendingOperation = PendingOperation();
   _wasTouchMoveIgnored = false;
}


void DDManager::initSession()
{
   ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::initSession()"));

   clearSession();
   _info.SessionId = getNextSessionId();
}


void DDManager::endSession()
{
   ETG_TRACE_USR1_DCL((TR_CLASS_HMI_FW_DRAG_DROP, "DDManager::endSession()"));

   clearSession();
   if (_acceptedTouch != NULL)
   {
      setAcceptedTouch(NULL);
   }
}


class DDWidgetFinderCallback : public WidgetCheckCallback
{
   public:
      DDWidgetFinderCallback(bool checkDragDropSource, const Courier::TouchInfo& touchInfo)
         : _checkDragDropSource(checkDragDropSource),
           _touchInfo(touchInfo),
           _widget(NULL),
           _touchPriority(0)
      {
      }

      virtual ~DDWidgetFinderCallback()
      {
         _widget = NULL;
      }

      virtual bool CheckWidget(Candera::Widget2D* widget)
      {
         Touchable2D* touchableWidget = dynamic_cast<Touchable2D*>(widget);

         if ((touchableWidget != NULL)
               && ((_checkDragDropSource && (touchableWidget->GetDragDropSourceEnabled()) && (touchableWidget->IsTouchable()))
                   || (!_checkDragDropSource && (touchableWidget->GetDragDropDestinationEnabled())))
               && ((_widget == NULL) || (touchableWidget->GetTouchPriority() > _touchPriority))
               && (touchableWidget->GetNode() != NULL)
               && (touchableWidget->GetNode()->IsEffectiveRenderingEnabled())
               && (touchableWidget->IsInsideBoundingRect(_touchInfo)))
         {
            _touchPriority = touchableWidget->GetTouchPriority();
            _widget = widget;
         }

         //continue searching through all the widgets to get the one with the highest touch priority
         return false;
      }

      Candera::Widget2D* GetWidget() const
      {
         return _widget;
      }

   private:
      bool _checkDragDropSource;
      Courier::TouchInfo _touchInfo;
      Candera::Widget2D* _widget;
      FeatStd::UInt32 _touchPriority;
};


Touchable2D* DDManager::findSourceWidget(const Courier::TouchInfo& touchInfo)
{
   DDWidgetFinderCallback callback(true, touchInfo);
   WidgetCheckReqMsg msg(&callback);
   if (_viewHandler != NULL)
   {
      _viewHandler->OnMessageViewTreeOnly(msg);
   }
   return Candera::Dynamic_Cast<Touchable2D*>(callback.GetWidget());
}


Touchable2D* DDManager::findDestinationWidget(const Courier::TouchInfo& touchInfo)
{
   DDWidgetFinderCallback callback(false, touchInfo);
   WidgetCheckReqMsg msg(&callback);
   if (_viewHandler != NULL)
   {
      _viewHandler->OnMessageViewTreeOnly(msg);
   }
   return Candera::Dynamic_Cast<Touchable2D*>(callback.GetWidget());
}


void DDManager::cancelOperation()
{
   sendStatusInfo(hmibase::DRAG_DROP_CANCELED);
   endSession();
}


void DDManager::createAdorner(Touchable2D& widget, const Courier::TouchInfo& touchInfo)
{
   destroyAdorner();

   if (_adornerManager == NULL)
   {
      if (_oldAdorner == NULL)
      {
         _oldAdorner = CANDERA_NEW(DDAdorner);
      }

      if (_oldAdorner != NULL)
      {
         _oldAdorner->create(widget, touchInfo);
      }
   }
   else
   {
      _adorner = _adornerManager->createAdorner(widget);
      if (!_adorner.PointsToNull())
      {
         Candera::Vector2 touchCoord1(static_cast<FeatStd::Float>(touchInfo.mX), static_cast<FeatStd::Float>(touchInfo.mY));
         _adorner->Data.set(touchCoord1);
      }
   }
}


void DDManager::destroyAdorner()
{
   if (_oldAdorner != NULL)
   {
      _oldAdorner->destroy();
   }

   if (_adornerManager != NULL)
   {
      _adornerManager->destroyAdorner(_adorner);
   }
   _adorner.Release();
}


void DDManager::showAdorner()
{
   if (_oldAdorner != NULL)
   {
      _oldAdorner->show();
   }

   if (_adornerManager != NULL)
   {
      _adornerManager->showAdorner(_adorner);
   }
}


void DDManager::hideAdorner()
{
   if (_oldAdorner != NULL)
   {
      _oldAdorner->hide();
   }

   if (_adornerManager != NULL)
   {
      _adornerManager->hideAdorner(_adorner);
   }
}


void DDManager::moveAdorner(const Courier::TouchInfo& touchInfo)
{
   if (_oldAdorner != NULL)
   {
      _oldAdorner->move(touchInfo);
   }

   if (!_adorner.PointsToNull())
   {
      Candera::Vector2 touchCoord1(static_cast<FeatStd::Float>(touchInfo.mX), static_cast<FeatStd::Float>(touchInfo.mY));
      Candera::Vector2* oldTouchCoord = _adorner->Data.get<Candera::Vector2>();
      if ((oldTouchCoord != NULL) && (_adornerManager != NULL))
      {
         _adornerManager->moveAdorner(_adorner, touchCoord1 - *oldTouchCoord);
      }
      _adorner->Data.set(touchCoord1);
   }
}


}
