/* ***************************************************************************************
* FILE:          DirectTextureProvider2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DirectTextureProvider2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Widgets/2D/DirectTextureProvider/DirectTextureProvider2D.h"
#include "Candera/Engine2D/Core/ProxyImage2D.h"
#include "CanderaPlatform/Device/Genivi/GeniviFrameBufferObject.h"
#include "Courier/Visualization/ViewScene2D.h"
#include "Courier/Visualization/Renderer.h"
#include "View/CGI/CgiExtensions/ViewScene2D.h"

#include <inttypes.h>
#include <math.h>
#include <time.h>

#include "Gadget/SyncBlockProducerFactory.h"
#include "hmibase/gadget/videobuffer/VideoBufferType.h"
#include "View/CGI/CgiExtensions/ImageLoader.h"

#ifndef VARIANT_S_FTR_ENABLE_GADGET_FILE_EXCHANGE
#ifdef VARIANT_S_FTR_ENABLE_USE_EGLKHR_EXTERNAL_TEXTURE_IMAGE
#include <CanderaPlatform/Device/Common/Base/ExternalTextureImage.h>
#include "CanderaPlatform/Device/Common/EGL/EglKhrExternalTextureImage.h"
#include <drm/drm_fourcc.h>
#else
#include <CanderaPlatform/Device/Common/Base/DirectTextureImage.h>
#endif
#endif

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_DIRECTTEXTURE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/DirectTextureProvider2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(DirectTextureProvider2D)

using namespace FeatStd;
using namespace Candera;

bool DirectTextureProvider2D::DebugUtil::s_initialized = false;
std::set<unsigned int> DirectTextureProvider2D::DebugUtil::s_instanceIdsToDebug;

void DirectTextureProvider2D::DebugUtil::Capture(unsigned int instanceId, hmibase::gadget::videobuffer::VideoBufferType* buffer, Courier::ViewId viewName)
{
   if (s_initialized == false)
   {
      s_initialized = true;
      // get instance ids to debug from environment
      char* tmp = getenv("WIDGET_DTP_DEBUG");// expect a comma separated list of ids

      if (tmp != 0)
      {
         const char* delim = ",";
         char* token = strtok(tmp, delim);

         while (token)
         {
            int id2log;
            if (sscanf(token, "%d", &id2log) != 1)
            {
               // error
            }
            else
            {
               s_instanceIdsToDebug.insert(id2log);
            }
            token = strtok(NULL, delim);
         }
      }
   }

   if ((s_instanceIdsToDebug.find(instanceId) != s_instanceIdsToDebug.end()) && buffer)
   {
#if !defined(CGI_GPU_SIMULATION)
      char filePath[150];
      static int i = 0;
      struct timespec spec;
      long ms = 0;
      clock_gettime(CLOCK_MONOTONIC, &spec);

      ms = spec.tv_sec * 1000 + static_cast<long>(round(static_cast<double>(spec.tv_nsec) / 1000000.0));

      snprintf(filePath, 150, "/tmp/image_provider_%04d_%02d__fd_%d__time_%ld__%s.png", ++i, instanceId, buffer->getFd(), ms, viewName.CStr());

      if (ImageLoader::saveImage(buffer->getDataPtr(), buffer->getDataSize(), buffer->getWidth(), buffer->getHeight(), filePath))
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "drm buffer for instance %d written to %s", instanceId, filePath));
      }
#endif
   }
}


DirectTextureProvider2D::DirectTextureProvider2D() :
   Base()
   , _mode(INVALID)
   , _lastExchangedBuffer(0)
   , _syncBlockId(0)
   , _nextCamera(0)
   , _initialBufferUpload(false)
{
}


DirectTextureProvider2D::~DirectTextureProvider2D()
{
   GeniviFrameBufferObject* fbos[] =
   {
      FeatStd::Internal::PointerToPointer<GeniviFrameBufferObject*>(GetFrameBuffer()),                         //lint !e446
      FeatStd::Internal::PointerToPointer<GeniviFrameBufferObject*>(GetFrameBuffer2()),                        //lint !e446
      FeatStd::Internal::PointerToPointer<GeniviFrameBufferObject*>(GetFrameBuffer3())
   };                                                                                                          //lint !e446

   for (unsigned int i = 0; i < sizeof(fbos) / sizeof(GeniviFrameBufferObject*); i++)
   {
      if (fbos[i] != 0)
      {
         fbos[i]->GetProperties().SetExternalColorTexture(0);//lint !e10 !e746 !e1013 !e1055 ToDo: If this line stays in final version we must apply for a permission to deactivate the lint msessages
         fbos[i]->Unload();
      }
   }

   // clear buffer
#if !defined(CGI_GPU_SIMULATION)
//   SyncBlockProducerFactory::DrmBufferContainer* d = SyncBlockProducerFactory::GetInstance().GetSyncBlockProducer(_syncBlockId, GetInstanceId(), this);
//   if (d)
//   {
//      struct drm_access_buffer* currBuffer = d->getCurrentBuffer();
//
//      // write memory pattern to drm buffer to clear content
//      if (currBuffer)
//      {
//         uint8_t* p = currBuffer->data;
//         size_t count = 0;
//         //                   R    G    B    A
//         uint8_t bytes[8] = {0xff, 0x04, 0xff, 0x04,
//                             0x00, 0xff, 0x00, 0xff
//                            };
//
//         while (count < currBuffer->size)
//         {
//            *(p + count) = bytes[count % 8];
//            ++count;
//         }
//         d->exchange();
//      }
//   }
   // NCG3D-31949: don't release drm buffer and leave it in cache for next usage
   hmibase::gadget::SyncBlockProducerFactory::GetInstance().ReleaseSyncBlockProducer(_syncBlockId, GetInstanceId(), this);
#endif

   _nextCamera = 0;

   hmibase::view::RenderJobStrategy::DeregisterListener(this);
}


void DirectTextureProvider2D::InitWidget()
{
   Base::InitWidget();

   // check for single or triple buffering
   if (GetCamera() && GetFrameBuffer() && (GetFrameBuffer()->GetUnitType() == c_geniviFrameBufferObjectType))
   {
      _mode = SINGLEBUFFERED;

      if (GetCamera2()
            && GetFrameBuffer2()
            && (GetFrameBuffer2()->GetUnitType() == c_geniviFrameBufferObjectType)
            && GetCamera3()
            && GetFrameBuffer3()
            && (GetFrameBuffer3()->GetUnitType() == c_geniviFrameBufferObjectType))
      {
         _mode = TRIPLEBUFFERED;
      }
   }

   if (_mode == SINGLEBUFFERED)
   {
      _nextCamera = GetCamera();
   }
}


void DirectTextureProvider2D::Update()
{
   if (_mode == INVALID)
   {
      return;
   }

   // activate camera
   ToggleCamera(_nextCamera);

#ifdef VARIANT_S_FTR_ENABLE_GADGET_FILE_EXCHANGE
   InitSyncBlock();
   hmibase::gadget::SyncBlockProducerFactory::DrmBufferContainer* d = hmibase::gadget::SyncBlockProducerFactory::GetInstance().GetSyncBlockProducer(_syncBlockId, GetInstanceId(), this);
   if (d == 0)
   {
      return;
   }
   int id = d->getCurrentBufferId();
   Candera::Camera2D* camera = GetCamera();

   if (_mode == TRIPLEBUFFERED)
   {
      if (_directTexturerMap.find(id) == _directTexturerMap.end())
      {
         Candera::Camera2D* cameras[] =
         {
            GetCamera(),
            GetCamera2(),
            GetCamera3()
         };

         for (size_t i = 0; i < sizeof(cameras) / sizeof(Camera2D*); i++)
         {
            bool cameraUsed = false;
            for (DirectTextureMap::iterator it = _directTexturerMap.begin(); it != _directTexturerMap.end(); ++it)
            {
               if (it->second.second == cameras[i])
               {
                  cameraUsed = true;
               }
            }

            if (!cameraUsed)
            {
               _initialBufferUpload = true;
               // this camera is unused
               TextureImageSharedPointer dummy;
               _directTexturerMap[id] = std::pair<TextureImageSharedPointer, const Candera::Camera2D*>(dummy, cameras[i]); //lint !e713

               ToggleCamera(cameras[i]);
               break;
            }
         }
      }
   }
   else
   {
      if (_directTexturerMap.empty())
      {
         _initialBufferUpload = true;
         TextureImageSharedPointer dummy;
         _directTexturerMap[id] = std::pair<TextureImageSharedPointer, const Candera::Camera2D*>(dummy, GetCamera()); //lint !e713
      }
   }
#else
   // always get a new buffer and upload the same on the same fbo
#if !defined(CGI_GPU_SIMULATION)
   InitSyncBlock();
   hmibase::gadget::SyncBlockProducerFactory::DrmBufferContainer* d = hmibase::gadget::SyncBlockProducerFactory::GetInstance().GetSyncBlockProducer(_syncBlockId, GetInstanceId(), this);
   if (d == 0)
   {
      ETG_TRACE_FATAL_DCL((APP_TRACECLASS_ID(), "no sync block producer for syncBlockId %d and instance id %d", _syncBlockId, GetInstanceId()));
      return;
   }

   int id = d->getCurrentBufferId();
   hmibase::gadget::videobuffer::VideoBufferType* currBuffer = d->getCurrentBuffer();

   if (currBuffer == 0)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "no current buffer in sync block producer for syncBlockId %d and instance id %d", _syncBlockId, GetInstanceId()));
      return;
   }

   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DRM buffer allocated instance id %d FB = %d data = %p with size = %u",
                       GetInstanceId(),
                       id,
                       currBuffer->getDataPtr(),
                       static_cast<uint32_t>(currBuffer->getDataSize())));

   // init buffer with previously uploaded content
   if (_lastExchangedBuffer)
   {
      if (_lastExchangedBuffer->getDataSize() == currBuffer->getDataSize())
      {
         memcpy(currBuffer->getDataPtr(), _lastExchangedBuffer->getDataPtr(), currBuffer->getDataSize());
      }
      else
      {
         // write memory pattern to drm buffer to clear content
         uint8_t* p = currBuffer->getDataPtr();
         size_t count = 0;
         //                   R    G    B    A
         uint8_t bytes[4] = {0x00, 0xff, 0x00, 0xff};

         while (count < currBuffer->getDataSize())
         {
            *(p + count) = bytes[count % 4];
            ++count;
         }
      }
   }

   DirectTextureMap::iterator it = _directTexturerMap.find(id);
   if (it == _directTexturerMap.end())
   {
#ifdef VARIANT_S_FTR_ENABLE_USE_EGLKHR_EXTERNAL_TEXTURE_IMAGE

      EGLint defaultAttribs[13];
      defaultAttribs[0] = EGL_DMA_BUF_PLANE0_FD_EXT;
      defaultAttribs[1] = currBuffer->getFd(); // textureImage.GetPlane(index)->m_descriptor;
      defaultAttribs[2] = EGL_DMA_BUF_PLANE0_OFFSET_EXT;
      defaultAttribs[3] = 0; // textureImage.GetPlane(index)->m_offset; may be 0
      defaultAttribs[4] = EGL_DMA_BUF_PLANE0_PITCH_EXT;
      defaultAttribs[5] = currBuffer->getPitch(); // textureImage.GetPlane(index)->m_pitch;  start of next line. this could be width * bytesperpixel
      defaultAttribs[6] = EGL_LINUX_DRM_FOURCC_EXT;
      defaultAttribs[7] = DRM_FORMAT_ARGB8888; // textureImage.GetCustomFormat();
      defaultAttribs[8] = EGL_WIDTH;
      defaultAttribs[9] = currBuffer->getWidth();// textureImage.GetWidth();
      defaultAttribs[10] = EGL_HEIGHT;
      defaultAttribs[11] = currBuffer->getHeight(); // textureImage.GetHeight();
      defaultAttribs[12] = EGL_NONE; //Terminator

      Candera::EglKhrExternalTextureImage::SharedPointer dt = Candera::EglKhrExternalTextureImage::Create();
      dt->SetRole(Candera::EglKhrExternalTextureImage::Producer); //Can only be set once on purpose!!!!!!!
      dt->SetAttributeList(defaultAttribs, sizeof(defaultAttribs) / sizeof(defaultAttribs[0])); // will be copied and stored
      dt->SetExternalBufferType(EGL_LINUX_DMA_BUF_EXT); // buffer type
      dt->SetExternalBuffer(static_cast<EGLClientBuffer>(0)); // actual buffer

      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTextureProvider::Update EglKhrExternalTextureImage created, plane.descriptor %d, width %d, height %d",
                          currBuffer->getFd(),
                          currBuffer->getWidth(),
                          currBuffer->getHeight()));
#else
      ::FeatStd::MemoryManagement::SharedPointer< ::Candera::DirectTextureImage > dt = ::Candera::DirectTextureImage::Create();
      //_directTexturerMap[currBuffer->fb] = DirectTextureImage::Create();
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTextureImage created."));
      //it = _directTexturerMap.find(currBuffer->fb);
      dt->SetWidth(currBuffer->getWidth());
      dt->SetHeight(currBuffer->getHeight());
      dt->SetFormat(DirectTextureImage::FormatRGBA);

      void* logical[] = { currBuffer->getDataPtr(), NULL, NULL, NULL };
      Candera::UInt physical[4] = { ~static_cast<UInt>(0U), ~static_cast<UInt>(0U), ~static_cast<UInt>(0U), ~static_cast<UInt>(0U) };
      dt->SetLogicalAddress(logical);
      dt->SetPhysicalAddress(physical);
#endif

      if (dt->Upload())
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTextureImage Upload SUCCESS."));
      }
      else
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTextureImage Upload FAILED!"));
      }

      Candera::Camera2D* camera = GetCamera();

      if (_mode == TRIPLEBUFFERED)
      {
         GeniviFrameBufferObject* fbos[] =
         {
            FeatStd::Internal::PointerToPointer<GeniviFrameBufferObject*>(GetFrameBuffer()),                   //lint !e446
            FeatStd::Internal::PointerToPointer<GeniviFrameBufferObject*>(GetFrameBuffer2()),                  //lint !e446
            FeatStd::Internal::PointerToPointer<GeniviFrameBufferObject*>(GetFrameBuffer3())
         };                                                                                                    //lint !e446

         // upload external color texture
         for (unsigned int i = 0; i < sizeof(fbos) / sizeof(GeniviFrameBufferObject*); i++)
         {
            if (fbos[i] != 0 && fbos[i]->GetProperties().GetExternalColorTexture() == 0) //lint !e10 !e746 !e1013 !e1055 ToDo: If this line stays in final version we must apply for a permission to deactivate the lint msessages
            {
               if (fbos[i]->ToRenderTarget2D() == GetCamera2()->GetRenderTarget()) //lint !e10 !e58 !e746 !e1013 !e1055 ToDo: If this line stays in final version we must apply for a permission to deactivate the lint msessages
               {
                  camera = GetCamera2();
               }
               else if (fbos[i]->ToRenderTarget2D() == GetCamera3()->GetRenderTarget()) //lint !e10 !e58 !e746 !e1013 !e1055 ToDo: If this line stays in final version we must apply for a permission to deactivate the lint msessages
               {
                  camera = GetCamera3();
               }
               else
               {
                  // use default camera from GetCamera
               }

               ToggleCamera(camera);

               FboUpload(fbos[i], dt);
               _initialBufferUpload = true;
               break;
            }
         }
      }

      _directTexturerMap[id] = std::pair<TextureImageSharedPointer, const Candera::Camera2D*>(dt, camera); //lint !e713
      it = _directTexturerMap.find(id);
   }

   if (_mode == SINGLEBUFFERED)
   {
      if (FeatStd::Internal::PointerToPointer<GeniviFrameBufferObject*>(GetFrameBuffer())->GetProperties().GetExternalColorTexture() == 0)
      {
         _initialBufferUpload = true;
      }
      FboUpload(FeatStd::Internal::PointerToPointer<GeniviFrameBufferObject*>(GetFrameBuffer()), it->second.first);
   }
#endif
#endif
}


void DirectTextureProvider2D::ToggleCamera(const Candera::Camera2D* camera)
{
   if (camera == 0)
   {
      return;
   }

   Camera2D* cameras[] =
   {
      GetCamera(),                                                                                             //lint !e446
      GetCamera2(),                                                                                            //lint !e446
      GetCamera3()
   };                                                                                                          //lint !e446

   for (unsigned int i = 0; i < sizeof(cameras) / sizeof(Camera2D*); i++)
   {
      bool enable = (camera == cameras[i]) ? true : false;
      if (cameras[i])
      {
         cameras[i]->SetRenderingEnabled(enable);
         if (enable)
         {
            ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Enable rendering for camera '%s'", cameras[i]->GetName()));
         }
      }
   }
}


void DirectTextureProvider2D::FboUpload(GeniviFrameBufferObject* fbo, TextureImageSharedPointer directTexture)
{
   if (fbo)
   {
      //Associate direct texture image with FBO.
      fbo->GetProperties().SetExternalColorTexture(directTexture->ToImageSource3D());//lint !e10 !e746 !e1013 !e1055 ToDo: If this line stays in final version we must apply for a permission to deactivate the lint msessages

      //Need to re-upload FBO.
      fbo->Unload();
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Starting Upload of FBO..."));
      if (!fbo->Upload())
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Failed to upload frame buffer object!"));
      }
      else
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "FBO upload successful."));
      }
   }
}


void DirectTextureProvider2D::OnParentViewLoad(bool load)
{
   if (load)
   {
      hmibase::view::ViewScene2D* viewScene = dynamic_cast<hmibase::view::ViewScene2D*>(GetParentView());
      if (viewScene)
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTextureProvider2D::OnParentViewLoad: disable dirty rectangle, %s", viewScene->GetId().CStr()));
         // don't use dirty rectangle for gadget provider scenes
         viewScene->UseDirtyRect(false);
         hmibase::view::RenderJobStrategy::RegisterListener(this);;
      }
      else
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "DirectTextureProvider2D::OnParentViewLoad: Not a viewScene2D, %s", GetParentView()->GetId().CStr()));
         hmibase::view::RenderJobStrategy::DeregisterListener(this);;
      }
   }
   else
   {
      GeniviFrameBufferObject* fbos[] =
      {
         FeatStd::Internal::PointerToPointer<GeniviFrameBufferObject*>(GetFrameBuffer()),                      //lint !e446
         FeatStd::Internal::PointerToPointer<GeniviFrameBufferObject*>(GetFrameBuffer2()),                     //lint !e446
         FeatStd::Internal::PointerToPointer<GeniviFrameBufferObject*>(GetFrameBuffer3())
      };                                                                                                       //lint !e446

      for (unsigned int i = 0; i < sizeof(fbos) / sizeof(GeniviFrameBufferObject*); i++)
      {
         if (fbos[i] != 0)
         {
            fbos[i]->GetProperties().SetExternalColorTexture(0);//lint !e10 !e746 !e1013 !e1055 ToDo: If this line stays in final version we must apply for a permission to deactivate the lint msessages
            fbos[i]->Unload();
         }
      }
   }
}


void DirectTextureProvider2D::OnParentViewActivate(bool active)
{
   Base::OnParentViewActivate(active);
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTextureProvider2D::OnParentViewActivate: Unloading buffers on hide active %d ReleaseBufferOnHide %d", active, GetReleaseBufferOnHide()));
   if (!active && GetReleaseBufferOnHide())
   {
#if !defined(CGI_GPU_SIMULATION)
      hmibase::gadget::SyncBlockProducerFactory::GetInstance().ReleaseSyncBlockProducer(_syncBlockId, GetInstanceId(), this);
#endif
   }
}


void DirectTextureProvider2D::InitSyncBlock()
{
   if (_syncBlockId == 0)
   {
#if !defined(CGI_GPU_SIMULATION)
      hmibase::gadget::SyncBlockProducerFactory::SyncBlockIdSet syncBlockIds = hmibase::gadget::SyncBlockProducerFactory::GetInstance().GetSyncBlockIDForInstance(GetInstanceId());

      if (syncBlockIds.empty())
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "No SyncBlockId for producer instance %d configured", GetInstanceId()));
         return;
      }
      else
      {
         // todo: how to handle multiple connections
         _syncBlockId = *(syncBlockIds.begin());
         if (_syncBlockId == 0)
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "Invalid SyncBlockId for producer instance %d", GetInstanceId()));
            return;
         }

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SyncBlockId %d found for Producer instance %d", _syncBlockId, GetInstanceId()));
      }

      hmibase::gadget::SyncBlockProducerFactory::DrmBufferContainer* d = hmibase::gadget::SyncBlockProducerFactory::GetInstance().GetSyncBlockProducer(_syncBlockId, GetInstanceId(), this);

      if (d == 0)
      {
         ETG_TRACE_FATAL_DCL((APP_TRACECLASS_ID(), "no sync block producer for syncBlockId %d and instance id %d", _syncBlockId, GetInstanceId()));
         return;
      }

      //if (d->isEmpty())
      {
         int w = FeatStd::Internal::PointerToPointer<GeniviFrameBufferObject*>(GetFrameBuffer())->GetProperties().GetWidth();//lint !e10 !e746 !e1013 !e1055 ToDo: If this line stays in final version we must apply for a permission to deactivate the lint msessages
         int h = FeatStd::Internal::PointerToPointer<GeniviFrameBufferObject*>(GetFrameBuffer())->GetProperties().GetHeight();//lint !e10 !e746 !e1013 !e1055 ToDo: If this line stays in final version we must apply for a permission to deactivate the lint msessages
         d->createDrmBuffers(w, h, 24, 32);
      }
#endif
   }
}


void DirectTextureProvider2D::OnPostSwapBuffer(Courier::Gdu& gdu)
{
   if (gdu.IsOffscreen())
   {
      Camera2D* cameras[] =
      {
         GetCamera(),                                                                                             //lint !e446
         GetCamera2(),                                                                                            //lint !e446
         GetCamera3()
      };                                                                                                          //lint !e446

      bool updated = false;
      for (unsigned int i = 0; i < sizeof(cameras) / sizeof(Camera2D*); i++)
      {
         if ((cameras[i]) && (cameras[i]->IsRenderingEnabled()))
         {
            if ((cameras[i]->GetRenderTarget()) && (gdu.GetGdu() == cameras[i]->GetRenderTarget()->GetGraphicDeviceUnit()))
            {
               updated = true;
            }
         }
      }

      if (updated)
      {
         // gdu swapped , now exchange direct texture
         if (_syncBlockId != 0)
         {
            if (_initialBufferUpload)
            {
               // force initial redraw on new offscreen buffer before sending buffer handle
               _initialBufferUpload = false;
               Invalidate();
            }
#if !defined(CGI_GPU_SIMULATION)
            else
            {
               hmibase::gadget::SyncBlockProducerFactory::DrmBufferContainer* d = hmibase::gadget::SyncBlockProducerFactory::GetInstance().GetSyncBlockProducer(_syncBlockId, GetInstanceId(), this);
               if (d)
               {
                  DebugUtil::Capture(GetInstanceId(), d->getCurrentBuffer(), GetParentView()->GetId());

                  if (d->getCurrentBuffer())
                  {
                     ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTextureProvider2D instance id %d exchange fb %d", GetInstanceId(), d->getCurrentBufferId()));
                  }
                  else
                  {
                     ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "DirectTextureProvider2D instance id %d", GetInstanceId()));
                  }

#ifdef VARIANT_S_FTR_ENABLE_GADGET_FILE_EXCHANGE
                  hmibase::gadget::videobuffer::VideoBufferType* videoBuffer = d->getCurrentBuffer();

                  if (videoBuffer)
                  {
                     glReadPixels(0, 0, videoBuffer->getWidth(), videoBuffer->getHeight(), GL_RGBA, GL_UNSIGNED_BYTE, videoBuffer->getDataPtr());

                     std::stringstream ss;
                     ss << "/tmp/gadget_" << GetInstanceId() << "_" << d->getCurrentBufferId() << ".png";

                     if (ImageLoader::saveImage(videoBuffer->getDataPtr(), videoBuffer->getDataSize(), videoBuffer->getWidth(), videoBuffer->getHeight(), ss.str().c_str(), true))
                     {
                        ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "gadget exchange buffer for instance %d written to %s", GetInstanceId(), ss.str().c_str()));
                     }
                     else
                     {
                        ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "saveImage to %s failed", ss.str().c_str()));
                     }
                  }
                  else
                  {
                     ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "Videobuffer for instance id %d is NULL, save image to RFS failed", GetInstanceId()));
                  }
#endif

                  // remember this buffer for later content copy
                  _lastExchangedBuffer = d->getCurrentBuffer();

                  int next = d->exchange();
                  DirectTextureMap::iterator it = _directTexturerMap.find(next);
                  if (it != _directTexturerMap.end())
                  {
                     _nextCamera = const_cast<Candera::Camera2D*>(it->second.second);
                  }
               }
            }
#endif
         }
      }
   }
}
