/* ***************************************************************************************
* FILE:          TEnumerator.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TEnumerator is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#if !defined(TEnumerator_H)
#define TEnumerator_H

#include <FeatStd/Config.h>
#include <FeatStd/Util/WarningMacros.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>

/**
 * The template TEnumerator provides an abstract interface to enumerate instances of a specific type.
 * The implementation of the enumerator is intended to not create any additional container but to provide the next instance by the by the call of the MoveNext method.
 * NOTE: the enumerator starts before the first item and has therefore no valid Current value before the first call of MoveNext. If MoveNext returns false Current also
 * contains no valid value. Only if MoveNext returns true Current provides a valid value.
 */
template<typename TVal>
class TEnumerator
{
      FEATSTD_LINT_CURRENT_SCOPE(1712, "No default constructor needed.")

   public:
      /**
       * The Enumerator interface is not supposed to be used directly. Use the TEnumerator interface instead.
       */
      class Enumerator
      {
         public:
            Enumerator() :
               _current(TVal())
            {
            }
            virtual ~Enumerator()
            {
            }

            /**
             * Call MoveNext to set Current to the next value.
             */
            virtual bool MoveNext()
            {
               return false;
            }

            /**
             * Returns the current enumeration value.
             */
            TVal Current()
            {
               return _current;
            }

         protected:
            TVal _current;

         private:
            FEATSTD_SHARED_POINTER_DECLARATION();
      };
      typedef FeatStd::MemoryManagement::SharedPointer<Enumerator> EnumeratorPtr;

      /**
       * Call MoveNext to set Current to the next value.
       */
      bool MoveNext()
      {
         return (_enumerator.PointsToNull()) ? false : _enumerator->MoveNext();
      }

      /**
       * Returns the current enumeration value.
       */
      TVal Current()
      {
         return (_enumerator.PointsToNull()) ? TVal() : _enumerator->Current();
      }

      TEnumerator()
      {
      }
      /**
       * Use this constructor to return a useable enumerator.
       */
      explicit TEnumerator(Enumerator* enumerator) :
         _enumerator(enumerator)
      {
      }

      /**
       * The TEnumerator is allowed to be copied. It will only copy the underlying implentation.
       */
      TEnumerator(const TEnumerator& enumerator) :
         _enumerator(enumerator._enumerator)
      {
      }

      TEnumerator& operator=(const TEnumerator& enumerator)
      {
         _enumerator = enumerator._enumerator;
         return *this;
      }
   private:
      EnumeratorPtr _enumerator;
};


#endif
