/* ***************************************************************************************
* FILE:          PrepareClonedItemCachingBehavior.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PrepareClonedItemCachingBehavior is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "PrepareClonedItemCachingBehavior.h"

#include <Candera/System/Mathematics/Math.h>

#include "hmi_trace_if.h"
#include "Widgets/widget_etg_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/PrepareClonedItemCachingBehavior.cpp.trc.h"
#endif

PrepareClonedItemCachingBehavior::PrepareClonedItemCachingBehavior(OwnerInfoRetriever& ownerRetriever,
      const ContentProvider& contentProvider,
      const ITemplateRetriever& templateRetriever,
      BaseWidget2D* owner,
      Courier::UInt32 ownerId,
      ISizeContainer& sizeContainer,
      FeatStd::UInt8 preparedBufferSize,
      FeatStd::UInt8 prepareThreshold) :
   Base(ownerRetriever, _cachingStrategy),
   _cachingStrategy(),
   _contentProvider(contentProvider),
   _templateRetriever(templateRetriever),
   _owner(owner),
   _ownerId(ownerId),
   _sizeContainer(sizeContainer),
   _preparedItemBufferSize(preparedBufferSize),
   _prepareThreshold(prepareThreshold)
{
}


PrepareClonedItemCachingBehavior::~PrepareClonedItemCachingBehavior()
{
}


void PrepareClonedItemCachingBehavior::OnPostItemsUpdateFinished(FeatStd::UInt32 firstVisibleIndex, FeatStd::UInt32 lastVisibleIndex)
{
   FeatStd::UInt32 listSize = _contentProvider.GetVirtualListSize();
   FeatStd::UInt32 visibleItemSize = (lastVisibleIndex - firstVisibleIndex + 1);

   FeatStd::UInt32 currentPreparedSize = _cachingStrategy.GetCurrentCacheSize() - visibleItemSize;
   FeatStd::UInt32 firstCachedIndex = _cachingStrategy.GetFirstCachedIndex();
   FeatStd::UInt32 lastCachedIndex = _cachingStrategy.GetLastCachedIndex();

   FeatStd::UInt32 deltaToCacheStart = (firstVisibleIndex - firstCachedIndex);
   FeatStd::UInt32 deltaToCacheEnd = (lastCachedIndex - lastVisibleIndex);

   FeatStd::UInt32 preparedItemSizeUp = (deltaToCacheStart < _prepareThreshold) ? Candera::Math::Minimum(_preparedItemBufferSize, firstVisibleIndex) : 0;
   FeatStd::UInt32 preparedItemSizeDown = (deltaToCacheEnd < _prepareThreshold) ? Candera::Math::Minimum(_preparedItemBufferSize, (listSize - lastVisibleIndex - 1)) : 0;

   if ((preparedItemSizeUp  > 0) && (preparedItemSizeDown > 0))
   {
      // Divide the prepared items above and below the visible area
      preparedItemSizeUp = Candera::Math::Minimum(_preparedItemBufferSize / 2, preparedItemSizeUp);
      preparedItemSizeDown = Candera::Math::Minimum(Candera::Math::Maximum(_preparedItemBufferSize / 2, (_preparedItemBufferSize - preparedItemSizeUp)), preparedItemSizeDown);
   }

   else if ((preparedItemSizeUp > 0) && (preparedItemSizeUp < _preparedItemBufferSize))
   {
      // if prepared buffer not full, fill the remaining space with prepared items for the area below the visible area if buffer space available
      preparedItemSizeDown = Candera::Math::Maximum(FeatStd::Int32(_preparedItemBufferSize - preparedItemSizeUp - currentPreparedSize), (FeatStd::Int32) 0);
   }

   else if ((preparedItemSizeDown > 0) && preparedItemSizeDown < _preparedItemBufferSize)
   {
      // if prepared buffer not full, fill the remaining space with prepared items for the area above the visible area if buffer space available
      preparedItemSizeUp = Candera::Math::Maximum(FeatStd::Int32(_preparedItemBufferSize - preparedItemSizeDown - currentPreparedSize), (FeatStd::Int32) 0);
   }

   else
   {
      // do nothing
   }

   PrepareItemsUp((firstCachedIndex - 1), preparedItemSizeUp);
   PrepareItemsDown((lastCachedIndex + 1), preparedItemSizeDown);

   // Discard old items in case buffer size is exceeded
   GetCloningStrategy().Flush(firstVisibleIndex, lastVisibleIndex);
}


void PrepareClonedItemCachingBehavior::OnPostItemsUpdated(FeatStd::UInt32 firstVisibleIndex, FeatStd::UInt32 lastVisibleIndex)
{
   FeatStd::UInt32 listSize = _contentProvider.GetVirtualListSize();
   FeatStd::UInt32 visibleItemSize = (lastVisibleIndex - firstVisibleIndex + 1);
   _cachingStrategy.SetCacheSize(Candera::Math::Minimum((visibleItemSize + _preparedItemBufferSize), listSize));
}


void PrepareClonedItemCachingBehavior::PrepareItem(FeatStd::UInt32 index)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PrepareClonedItemCachingBehavior::PrepareItem - preparing node with index: %d", index));

   const tSharedPtrListDataItem& data = _contentProvider.GetItem(index);

   ControlTemplateInstance* instance = GetCloningStrategy().CloneNode(_templateRetriever, data, _owner, _ownerId, index).GetPointerToSharedInstance();

   // Pre-update and pre-measure prepared instance
   if (0 != instance)
   {
      instance->Update();
      Candera::Node2D* rootNode = instance->GetItemRootNode();
      if (0 != rootNode)
      {
         _sizeContainer.MeasureClientSize(rootNode);
      }
   }
}


void PrepareClonedItemCachingBehavior::PrepareItemsUp(FeatStd::UInt32 startIndex, FeatStd::UInt32 noOfItems)
{
   FeatStd::UInt32 preparedItemSize = 0;
   for (FeatStd::Int i = startIndex; ((i >= 0) && (preparedItemSize < noOfItems)); i--)
   {
      PrepareItem(i);
      preparedItemSize++;
   }
}


void PrepareClonedItemCachingBehavior::PrepareItemsDown(FeatStd::UInt32 startIndex, FeatStd::UInt32 noOfItems)
{
   FeatStd::UInt32 preparedItemSize = 0;
   FeatStd::Int32 listSize = _contentProvider.GetVirtualListSize();
   for (FeatStd::Int i = startIndex; ((i < listSize) && (preparedItemSize < noOfItems)); i++)
   {
      PrepareItem(i);
      preparedItemSize++;
   }
}
