/* ***************************************************************************************
* FILE:          OnRequestCloneCachingStrategy.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  OnRequestCloneCachingStrategy is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "ControlTemplate.h"
#include "OnRequestCloneCachingStrategy.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/OnRequestCloneCachingStrategy.cpp.trc.h"
#endif
#include "CloneEvents.h"
#include "CloneCachingStrategyHelper.h"


OnRequestCloneCachingStrategy::OnRequestCloneCachingStrategy() :
   _other(0)
{
}


OnRequestCloneCachingStrategy::~OnRequestCloneCachingStrategy()
{
   _other = 0;
}


ControlTemplateInstancePtr OnRequestCloneCachingStrategy::Retrieve(Candera::Node2D& templateNode, const tSharedPtrIDataItem& source, BaseWidget2D* owner, FeatStd::UInt32 ownerId, FeatStd::UInt32 index, const ListWidget2DBase::TemplateScrollAnimationType& templateAnimations)
{
   ControlTemplateInstancePtr clonedInstance(0);

   if (0 != _other)
   {
      clonedInstance = _other->Retrieve(templateNode, source, owner, ownerId, index, templateAnimations);
   }

   // update the local cache if retrieved instance is not null; otherwise return the locally cached one
   CTICache::iterator it(_cache.find(index));
   if (it != _cache.end())
   {
      if (clonedInstance.PointsToNull())
      {
         clonedInstance = it->second;
      }
      else
      {
         ControlTemplateInstancePtr& cachedInstance(it->second);
         cachedInstance = clonedInstance;
      }
   }
   return clonedInstance;
}


void OnRequestCloneCachingStrategy::ProcessAfterCloning(ControlTemplateInstancePtr& instance, FeatStd::UInt32 index)
{
   CTICache::iterator it(_cache.find(index));

   if (it != _cache.end())
   {
      _cache[index] = instance;
   }

   if (0 != _other)
   {
      _other->ProcessAfterCloning(instance, index);
   }
}


bool OnRequestCloneCachingStrategy::ProcessBeforeDisposed(ControlTemplateInstance* instance)
{
   bool dispose(true);

   if (0 != _other)
   {
      dispose = _other->ProcessBeforeDisposed(instance);
   }

   return dispose;
}


void OnRequestCloneCachingStrategy::ProcessBeforeFlush(FeatStd::UInt32 firstIndex, FeatStd::UInt32 lastIndex)
{
   if (0 != _other)
   {
      _other->ProcessBeforeFlush(firstIndex, lastIndex);
   }
}


void OnRequestCloneCachingStrategy::Finalize()
{
   _cache.clear();

   if (0 != _other)
   {
      _other->Finalize();
   }
}


void OnRequestCloneCachingStrategy::Retain(FeatStd::UInt32 index)
{
   CTICache::iterator it(_cache.find(index));

   if (it == _cache.end())
   {
      _cache[index] = ControlTemplateInstancePtr(0);
   }
}


void OnRequestCloneCachingStrategy::Release(FeatStd::UInt32 index)
{
   CTICache::iterator it(_cache.find(index));

   if (it != _cache.end())
   {
      _cache.erase(it);
   }
}


bool OnRequestCloneCachingStrategy::OnMessage(const Courier::Message& msg)
{
   bool consumed(false);

   for (CTICache::iterator it(_cache.begin()); (!consumed) && (it != _cache.end()); ++it)
   {
      ControlTemplateInstancePtr cti((*it).second);
      if (!cti.PointsToNull())
      {
         consumed = cti->OnMessage(msg);
      }
   }
   return consumed;
}


void OnRequestCloneCachingStrategy::Update()
{
   for (CTICache::iterator it(_cache.begin()); it != _cache.end(); ++it)
   {
      ControlTemplateInstancePtr cti((*it).second);
      if (!cti.PointsToNull())
      {
         cti->Update();
      }
   }
}
