/* ***************************************************************************************
* FILE:          ControlTemplateInstance.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ControlTemplateInstance is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef CONTROL_TEMPLATE_INSTANCE_H
#define CONTROL_TEMPLATE_INSTANCE_H

#include <Courier/Foundation/FoundationMsgs.h>
#include <Courier/Version.h>
#include "Widgets/2D/Common/FlexScrollable.h"
#include "Widgets/2D/Common/MetaInfoEnums.h"
#include "CTIAnimationsManager.h"
#include <Courier/Visualization/ViewScene2D.h>
#include "CanderaWidget/String/String.h"
#include "ControlTemplateMap.h"
#include "ClonedAnimationModifier.h"
#include "DefaultControlTemplateMap.h"

#include <map>

class BaseWidget2D;
class ControlTemplateInstanceDisposer;
/**
 * The ControlTemplateInstance is a container for all nodes and widget that have been cloned from a control template.
 * It will delegate messages and updates to the contained widgets.
 */
class ControlTemplateInstance : private CTIAnimationsManagerListener
{
      friend struct FeatStd::Internal::BaseClassRelation;

   public:
      FEATSTD_RTTI_DECLARATION();

      /**
      * Returns true if expand animation finished.
      * Meant to be 1 time called. ( When is called it resets the flag )
      */
      typedef Candera::ArrayProperty<Candera::Animation::AnimationPlayer::SharedPointer> AnimationsType;

      /**
       * Returns true in the given position intersects with the given node in the given view.
       */
      static bool IsItemIntersectingPoint(Courier::ViewScene2D* view2D, Candera::Node2D* node, Candera::UInt32 pointX, Candera::UInt32 pointY);

      /**
       * Default constructor.
       */
      ControlTemplateInstance();

      /**
       * If not already disposed the content will be disposed and returned to the ControlTemplateCloningStrategy.
       */
      virtual ~ControlTemplateInstance();

      /**
       * Sets the owner of this ControlTemplateInstance. Only to be called by a ControlTemplateCloningStrategy.
       */
      void SetOwner(BaseWidget2D* owner)
      {
         FEATSTD_DEBUG_ASSERT(0 == _owner);
         FEATSTD_DEBUG_ASSERT(0 != owner);
         _owner = owner;
      }

      /**
      * Sets the owner id of this ControlTemplateInstance. Only to be called by a ControlTemplateCloningStrategy.
      */
      void SetOwnerId(FeatStd::UInt32 ownerId)
      {
         _ownerId = ownerId;
      }

      /**
      * Returns the owner id of this ControlTemplateInstance.
      */
      FeatStd::UInt32 GetOwnerId() const
      {
         return _ownerId;
      }

      /**
      * Sets the index of this ControlTemplateInstance. Only to be called by a ControlTemplateCloningStrategy.
      */
      void SetIndex(FeatStd::UInt32 index)
      {
         _index = index;
      }

      /**
      * Returns the index of this ControlTemplateInstance.
      */
      FeatStd::UInt32 GetIndex() const
      {
         return _index;
      }

      /**
       * Sets the item root node of this ControlTemplateInstance. Only to be called by a ControlTemplateCloningStrategy.
       */
      void SetItemRootNode(Candera::Node2D* itemsGroup)
      {
         FEATSTD_DEBUG_ASSERT(0 == _itemRootNode);
         FEATSTD_DEBUG_ASSERT(0 != itemsGroup);
         _itemRootNode = itemsGroup;
      }

      /**
      * Sets the template node after which the clone was obtained. Only to be called by a ControlTemplateCloningStrategy.
      */
      void SetTemplateNode(Candera::Node2D* templateNode)
      {
         _templateNode = templateNode;
      }

      Candera::Node2D* GetTemplateNode() const
      {
         return _templateNode;
      }

      /**
       * Returns the item root node of this ControlTemplateInstance.
       */
      Candera::Node2D* GetItemRootNode() const
      {
         return _itemRootNode;
      }

      /**
      * Returns the item node which should have cached layout.
      */
      Candera::Node2D* GetCachedLayoutNode() const
      {
         return (_itemRootNode != 0) ? _itemRootNode->GetFirstChild() : 0;
      }

      void SetScrollAnimationOffset(Candera::Int16 animationOffset)
      {
         _animationOffset = animationOffset;
      }

      void SetScrollAnimationInput(Candera::ScrollAnimationInput::Enum animationInput)
      {
         _scrollAnimationInput = animationInput;
      }

      /**
       * Returns the owner of this ControlTemplateInstance.
       */
      BaseWidget2D* GetOwner() const
      {
         return _owner;
      }

      /**
       * Delegates the update to the contained widgets.
       */
      void Update();

      /**
       * Called when the itemRootNode has been arranged - starts the associated animations.
       */
      void PositionAnimate(Candera::Float itemPosition);

      /**
      * Called when the itemRootNode has been arranged - starts the associated animations.
      */
      void SpeedAnimate(Candera::UInt32 itemSpeed);

      /**
       * Delegates the message to the contained widgets.
       */
      bool OnMessage(const Courier::Message& msg);

      void Init();

      void Clear();

      /**
       * Updates the internal cloned animation lists based on the given content.
       */
      void UpdateClonedAnimations(const AnimationsType& animations, const ControlTemplateMap& controlTemplateMap);

      AnimationsContainer& GetExpandAnimationsManager()
      {
         CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1536, "intended exposing of low access member");
         return _expandAnimationsManager;
      }

      void SetDisposer(ControlTemplateInstanceDisposer* disposer);

      DefaultControlTemplateMap& GetControlTemplateMap()
      {
         CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1536, "intended exposing of low access member");
         return _controlTemplateMap;
      }

      /**
      * Returns true if expand animation finished.
      * Meant to be 1 time called. ( When is called it resets the flag )
      */
      bool TestExpandPlaybackFinish()
      {
         if (_expandPlaybackFinished)
         {
            _expandPlaybackFinished = false;
            return true;
         }
         else
         {
            return false;
         }
      }

      bool ClonePropertiesInternal();
   private:

      typedef std::map<void*, Candera::Animation::AnimationController::SharedPointer> AnimationControllers;
      typedef std::map<const FeatStd::Char*, FeatStd::String> NamesCollection;

      class CompositeAnimationTraverser : public Candera::TreeTraverser2D
      {
         public:
            CompositeAnimationTraverser(ControlTemplateInstance* templateInstance, const ControlTemplateMap& controlTemplateMap, AnimationControllers& updatedScrollAnimations, AnimationControllers& updatedExpandAnimations, AnimationControllers& updatedSpeedAnimations);
            virtual Candera::TreeTraverser2D::TraverserAction ProcessNode(Candera::Node2D& node);

         private:
            FEATSTD_MAKE_CLASS_UNCOPYABLE(CompositeAnimationTraverser);

            ControlTemplateInstance* _templateInstance;
            const ControlTemplateMap& _controlTemplateMap;
            AnimationControllers& _updatedScrollAnimations;
            AnimationControllers& _updatedExpandAnimations;
            AnimationControllers& _updatedSpeedAnimations;
      };

      template <typename T> friend class FeatStd::MemoryManagement::SharedPointer;
      friend class ControlTemplateCloneTraverser2D;
      friend class ControlTemplateDisposeTraverser;
      friend class ControlTemplateCloneHelper;
      friend class ControlTemplate;
      friend class AnimationCloner;
      friend class CompositeAnimationTraverser;

      void Retain()
      {
         ++_retainCount._value;
      }

      void Release()
      {
         --_retainCount._value;
         if (0 == _retainCount._value)
         {
            Dispose();
         }
      }

      struct RetainCount
      {
         RetainCount() :
            _value(0)
         {
         }
         FeatStd::Int _value;
      } _retainCount;

      const Courier::Vector<Candera::WidgetBase*>& GetWidgets() const
      {
         return _widgets;
      }

      void Dispose();

      void SetWidgetName(Candera::WidgetBase& widget, const Candera::String& name);

      void SetWidgetStringId(Candera::WidgetBase& widget, const Candera::StringIdentifier* stringId);

      void AddWidget(Candera::WidgetBase* widget);
      void AddWidgetToUpdateAfterScrollAnimations(Candera::WidgetBase* widget);

      virtual void OnPlaybackFinished(bool forward);

      void UpdateClonedAnimation(Candera::Animation::AnimationPlayer::SharedPointer& player, const ControlTemplateMap& controlTemplateMap, AnimationControllers& updatedScrollAnimations, AnimationControllers& updatedExpandAnimations, AnimationControllers& updatedSpeedAnimations);
      const Candera::Animation::AnimationController::SharedPointer UpdateClonedAnimationByType(Candera::Animation::AnimationPlayer::SharedPointer& player, const ControlTemplateMap& controlTemplateMap, AnimationControllers& srcAnimations, AnimationControllers& dstAnimations, ClonedAnimationModifier* modifier);

      /**
      * Disposes the nodes and widgets of this ControlTemplateInstance. Only to be called by a ControlTemplateCloningStrategy.
      */
      void DisposeContent();

      Courier::Vector<Candera::WidgetBase*> _widgets;
      Courier::Vector<Candera::WidgetBase*> _widgetsToUpdateAfterScrollAnimation;
      NamesCollection _names;

      Courier::Vector<const Candera::StringIdentifier*> _stringIds;

      AnimationControllers _scrollAnimations;
      AnimationControllers _expandAnimations;
      AnimationControllers _speedAnimations;
      Candera::ScrollAnimationInput::Enum _scrollAnimationInput;
      bool _expandPlaybackFinished;

      BaseWidget2D* _owner;
      FeatStd::UInt32 _ownerId;
      FeatStd::UInt32 _index;
      Candera::Node2D* _itemRootNode;
      Candera::Node2D* _templateNode;
      Candera::Int16 _animationOffset;
      CTIAnimationsManager _expandAnimationsManager;
      ControlTemplateInstanceDisposer* _disposer;

      static FeatStd::Int _scrollAnimationSensitivity;
      DefaultControlTemplateMap _controlTemplateMap;
};


typedef Candera::MemoryManagement::SharedPointer<ControlTemplateInstance> ControlTemplateInstancePtr;

#endif
