/* ***************************************************************************************
* FILE:          ControlTemplateCloningStrategy.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ControlTemplateCloningStrategy is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef CONTROL_TEMPLATE_CLONING_STRATEGY_H
#define CONTROL_TEMPLATE_CLONING_STRATEGY_H

#include "ControlTemplateInstance.h"
#include <map>
#include <stack>
#include <Candera/EngineBase/DynamicProperties/DynamicPropertyHost.h>
#include <FeatStd/Util/ConvenienceMacros.h>
#include "Widgets/2D/Common/EventHub.h"
#include "ControlTemplateInstanceOwner.h"
#include "DefaultControlTemplateMap.h"
#include <Candera/Engine2D/Core/TreeTraverser2D.h>
#include <Widgets/2D/List/generated/ListWidget2DBase.h>
#include "CloneCachingStrategy.h"
#include "ControlTemplateInstanceDisposer.h"
#include <Widgets/2D/List/Content/ITemplateRetriever.h>

/**
 * The ControlTemplateCloningStrategy can be used to clone control templates (nodes and associated widgets).
 * There is a default ControlTemplateCloningStrategy instance that can be accessed as a singleton.
 * The cloned nodes and widgets are managed by a ControlTemplateInstance. Call the CloneNode method to create a new clone.
 * The exact cloning strategy depends on the underlying specific implementation.
 * As soon as the last reference to the ControlTemplateInstance is destroyed the ControlTemplateInstance is returned to
 * the ControlTemplateCloningStrategy by calling its Dispose method.
 * The base implementation will dispose all nodes and widgets if the Dispose method is called.
 */
class ControlTemplateCloningStrategy : public Candera::DynamicProperties::DynamicPropertyHost, public ControlTemplateInstanceDisposer
{
   public:
      FEATSTD_RTTI_DECLARATION();

      class DisposeEvent : public EventHub::Event
      {
         public:
            virtual ~DisposeEvent()
            {
               _instance = 0;
            }

            ControlTemplateInstance* GetInstance() const
            {
               return _instance;
            }

         protected:
            friend class ControlTemplateCloningStrategy;

            DisposeEvent(ControlTemplateInstance* instance) :
               _instance(instance)
            {
            }

            ControlTemplateInstance* _instance;
      };

      ControlTemplateCloningStrategy(CloneCachingStrategy& cachingStrategy);

      /**
       * Returns a clone of the given template that is initialized by the given data source and associated with the owner.
       * The owner is responsible to manage the ControlTemplateInstance (messages, updates, lieftime, ...).
       */
      ControlTemplateInstancePtr CloneNode(const ITemplateRetriever& templateRetriever, const tSharedPtrListDataItem& source, BaseWidget2D* owner, FeatStd::UInt32 ownerId, FeatStd::UInt32 index);

      /**
       * Flush is called to indicate that the best point to remove any temporarily cached ControlTemplateInstance should be removed from the cache.
       */
      void Flush(FeatStd::UInt32 /*firstVisibleIndex*/, FeatStd::UInt32 /*lastVisibleIndex*/);

      /**
       * The Finalize method should be called before the owner is destroyed.
       */
      void Finalize();

      /**
       *  Retrieves whether the layout area is set or not
       *  @param node The node whose maximum size is checked
       *  @return true, if the layout area is set, false if not
       */
      CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1764, node, CANDERA_LINT_REASON_NONCONST)
      static bool IsLayoutAreaSet(Candera::Node2D& node)
      {
         return node.IsValueSet(CdaDynamicPropertyInstance(LayoutArea));
      }

      /**
       *  Retrieves the layout area used for laying out this node.
       *  @param node The node whose margin is sought.
       *  @return The layout area used for laying out this node.
       */
      CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1764, node, CANDERA_LINT_REASON_NONCONST)
      static const Candera::Rectangle& GetLayoutArea(Candera::Node2D& node)
      {
         return node.GetValue(CdaDynamicPropertyInstance(LayoutArea));
      }

      /**
       *  Sets the layout area used for laying out this node.
       *  @param node   The node, whose margin shall be set.
       *  @param margin The layout area used for laying out this node.
       */
      static void SetLayoutArea(Candera::Node2D& node, const Candera::Rectangle& layoutArea)
      {
         static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(LayoutArea), layoutArea));
      }

      /**
          *  Provides the parent for dynamic property scene graph inheritance.
          *  @param  host    The host, whose parent is sought.
          *  @return         The parent of host if available, null otherwise.
          */
      static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost* host)
      {
         CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1774, type check above ensures cast is safe)
         return static_cast<const Candera::Node2D*>(host)->GetParent();
      }

   protected:/*
      friend class ControlTemplateInstance;
      friend class ReuseControlTemplateCloningStrategy;
      friend class PoolControlTemplateCloningStrategy;
*/
   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(ControlTemplateCloningStrategy);

      static Candera::Rectangle& LayoutAreaDefault();

      CdaDynamicProperties(ControlTemplateCloningStrategy, Candera::DynamicProperties::DynamicPropertyHost)
      CdaDynamicProperty(LayoutArea, Candera::Rectangle)
      CdaDynamicPropertyDefaultValue(LayoutAreaDefault())
      CdaDynamicPropertyDescription("The area in which the node has been layouted.")
      CdaDynamicPropertyCategory("Layout")
      CdaDynamicPropertyEnd();
      CdaDynamicPropertiesEnd();

      CloneCachingStrategy& _cachingStrategy;

      ControlTemplateInstancePtr Clone(Candera::Node2D& templateNode, const tSharedPtrIDataItem& source, BaseWidget2D* owner, FeatStd::UInt32 ownerId, FeatStd::UInt32 index, const ListWidget2DBase::TemplateScrollAnimationType& templateAnimations);

      void Dispose(ControlTemplateInstance* instance) override;
};


#endif
