/* ***************************************************************************************
* FILE:          ControlTemplate.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ControlTemplate is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#if !defined(ControlTemplate_H)
#define ControlTemplate_H

#include <Candera/EngineBase/DynamicProperties/DynamicPropertyHost.h>
#include <Candera/Engine2D/Core/Node2D.h>
#include "ControlTemplateInstance.h"
#include "ControlTemplateData.h"
#include "DataItem.h"

/**
 * The ControlTemplate class provides an interface to access relevant data that is stored per node. Several interfaces allow a lookup
 * of inherited value by setting the followParent to true.
 */
class ControlTemplate : public Candera::DynamicProperties::DynamicPropertyHost
{
   public:
      /**
       * This method is supposed to be called only by the cloning strategy.
       * The ControlTemplateInstance will be set only for the root node of the cloned control template.
       */
      static void SetControlTemplateInstance(Candera::Node2D& node, ControlTemplateInstance* controlTemplateInstance);

      /**
       * Returns true if the node is the root node of a cloned control template and has therefore a ControlTemplateInstance set.
       */
      static bool IsControlTemplateInstanceSet(const Candera::Node2D& node);

      /**
       * Returns the ControlTemplateInstance of this node if set.
       */
      static ControlTemplateInstance* GetControlTemplateInstance(const Candera::Node2D& node);

      /**
       * This method is supposed to be called only by the cloning strategy.
       * Adds an associated widget to the node.
       */
      static void AddAssociatedWidget(Candera::Node2D& node, Candera::WidgetBase* widget);

      /**
       * Sets a data context on the node.
       */
      static void SetDataContext(Candera::Node2D& node, const tSharedPtrIDataItem& value);

      /**
       * Returns true if a data context is set on the node (or any parent if followParent is set to true).
       */
      static bool IsDataContextSet(const Candera::Node2D& node, bool followParent = true);

      /**
       * Returns the data context that is set on the node (or any parent if followParent is set to true).
       */
      static tSharedPtrIDataItem GetDataContext(const Candera::Node2D& node, bool followParent = true);

      /**
       * Marks the node as a control template node that is used for cloning only.
       */
      static void SetTemplateNode(Candera::Node2D& node, bool value);

      /**
       * Returns true if the node is marked as a control template node that is used for cloning only.
       * NOTE: control template nodes within a control template are not cloned because their use is only to be a template for further cloning.
       * In conclusion also references to these nodes and referenced widgets within these template nodes are not replaced with cloned versions.
       */
      static bool IsTemplateNode(const Candera::Node2D& node, bool followParent = true);

      /**
       * Returns an enumerator of all widgets that are associated with this node. This method is not limited to cloned instances.
       * It will also return the widget that are part of the original scene.
       */
      static WidgetBaseEnumerator EnumerateAssociatedWidgets(Courier::ViewScene& view, const Candera::Node2D& node);

      /**
      * Sets ControlTemplateData to 0, the ControlTemplateData is deleted if deleteTemplateData is set to true.
      */
      static void ResetControlTemplateData(Candera::Node2D& node, bool deleteTemplateData);

      static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost* host);


#define ControlTemplateBindingIndexInterface(bindingName) \
    static void Set##bindingName##BindingIndex(Candera::Node2D& node, const FeatStd::UInt32& value); \
    static bool Is##bindingName##BindingIndexSet(const Candera::Node2D& node); \
    static FeatStd::Int32 Get##bindingName##BindingIndex(const Candera::Node2D& node)

      ControlTemplateBindingIndexInterface(Enabled);
      ControlTemplateBindingIndexInterface(Name);
      ControlTemplateBindingIndexInterface(Text);
      ControlTemplateBindingIndexInterface(Selected);
      ControlTemplateBindingIndexInterface(Selected2);
      ControlTemplateBindingIndexInterface(Items);
      ControlTemplateBindingIndexInterface(UserData);
      ControlTemplateBindingIndexInterface(BitmapPath);

#undef ControlTemplateBindingIndexInterface

   private:
      friend class Candera::Node2D;

      static ControlTemplateData* TryGetControlTemplateDataWithDataContext(const Candera::Node2D& node, bool followParent);
   public:
      static ControlTemplateData* TryGetControlTemplateData(const Candera::Node2D& node);
   private:
      static ControlTemplateData* GetControlTemplateData(Candera::Node2D& node);

      static const Candera::String& EmptyString();


#define ControlTemplateBindingProperty(bindingName, description) \
    CdaDynamicProperty(bindingName##BindingIndex, FeatStd::Int32); \
    CdaDynamicPropertyDescription(description) \
    CdaDynamicPropertyDefaultValue(-1) \
    CdaDynamicPropertyEnd()

      CdaDynamicProperties(ControlTemplate, DynamicPropertyHost);
      ControlTemplateBindingProperty(Enabled, "Defines the template binding of the control template for enable status property.");
      ControlTemplateBindingProperty(Name, "Defines the template binding of the control template for Name property.");
      ControlTemplateBindingProperty(Text, "Defines the template binding of the control template for Text property.");
      ControlTemplateBindingProperty(Selected, "Defines the template binding of the control template for Selected property.");
      ControlTemplateBindingProperty(Selected2, "Defines the template binding of the control template for Selected2 property.");
      ControlTemplateBindingProperty(Items, "Defines the template binding of the control template for dynamic items.");
      ControlTemplateBindingProperty(UserData, "Defines the template binding of the control template for user data.");
      ControlTemplateBindingProperty(BitmapPath, "Defines the template binding of the control template for BitmapPath property.");
      CdaDynamicPropertyUnregistered(ControlTemplateData, ControlTemplateData*);
      CdaDynamicPropertyEnd();
      CdaDynamicPropertiesEnd();
#undef ControlTemplateBindingProperty
};


#endif
