/* ***************************************************************************************
* FILE:          CTIAnimationsManager.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  CTIAnimationsManager is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/CTIAnimationsManager.cpp.trc.h"
#endif

#include "CTIAnimationsManager.h"


using namespace Candera;


CTIAnimationsManager::CTIAnimationsManager() :
   _animationStatus(AtStart),
   _animations(),
   _groupPlayer(Animation::AnimationGroupPlayer::Create()),
   _timeDispatcher(Animation::AnimationTimeDispatcher::Create()),
   _listener(0)
{
   if ((!_groupPlayer.PointsToNull()) && (!_timeDispatcher.PointsToNull()))
   {
      _groupPlayer->AddAnimationPlayerListener(this);
      _timeDispatcher->AddPlayer(_groupPlayer);
   }
}


CTIAnimationsManager::~CTIAnimationsManager()
{
   CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1579, "Cleanup made by DisposeContent.")
   DisposeContent();
}


void CTIAnimationsManager::DisposeContent()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "CTIAnimationsManager::DisposeContent this=%p", this));

   _animations.Clear();

   if (!_groupPlayer.PointsToNull())
   {
      _groupPlayer->RemoveAllPlayers();
   }

   _groupPlayer = Animation::AnimationGroupPlayer::SharedPointer(0);
   _timeDispatcher = Animation::AnimationTimeDispatcher::SharedPointer(0);
   _listener = 0;
}


void CTIAnimationsManager::AddAnimation(const Animation::AnimationController::SharedPointer& controller)
{
   if ((!_groupPlayer.PointsToNull()) && (!_groupPlayer->IsEnabled()) && (!_timeDispatcher.PointsToNull()) && (!controller.PointsToNull()))
   {
      const Animation::AnimationPlayer::SharedPointer& player(Animation::AnimationPlayer::Create());
      if (!player.PointsToNull())
      {
         player->SetController(controller);
         player->SetSequenceDurationMs(GetDuration(*controller.GetPointerToSharedInstance()));

         _groupPlayer->AddPlayer(player);
         _timeDispatcher->AddPlayer(player);

         _animations.Add(player);
      }
   }
}


void CTIAnimationsManager::RemoveAllAnimations()
{
   if ((!_groupPlayer.PointsToNull()) && (!_groupPlayer->IsEnabled()) && (!_timeDispatcher.PointsToNull()))
   {
      for (AnimationPlayers::ConstIterator it(_animations.ConstBegin()); it != _animations.ConstEnd(); ++it)
      {
         const Animation::AnimationPlayer::SharedPointer& player(*it);
         if (!player.PointsToNull())
         {
            _timeDispatcher->RemovePlayer(player);

            player->SetController(Animation::AnimationController::SharedPointer(0));
         }
      }

      _groupPlayer->RemoveAllPlayers();
      _animations.Clear();
   }
}


void CTIAnimationsManager::SetWorldTime(Animation::WorldTimeType worldTime)
{
   if (!_timeDispatcher.PointsToNull())
   {
      _timeDispatcher->SetWorldTimeMs(worldTime);
   }
}


void CTIAnimationsManager::DispatchWorldTime()
{
   if (AreRunning())
   {
      DispatchToPlayers();
   }
   else if ((_animationStatus == AtEnd) || (_animationStatus == AtStart))
   {
      for (AnimationPlayers::ConstIterator it(_animations.ConstBegin()); it != _animations.ConstEnd(); ++it)
      {
         const Animation::AnimationPlayer::SharedPointer& player(*it);
         const Animation::AnimationController::SharedPointer& controller((*it)->GetController());

         Animation::SequenceTimeType sequenceDurationMs((_animationStatus == AtEnd) ? player->GetSequenceDurationMs() : 0);
         controller->AnimateSequenceTime(sequenceDurationMs);
      }
   }
}


bool CTIAnimationsManager::Start(PlaybackDirection direction)
{
   if (!_groupPlayer.PointsToNull() && !_animations.Size() == 0)
   {
      bool shouldStart((direction == ForwardPlayback) && (_animationStatus == AtStart));
      shouldStart = shouldStart || ((direction == ReversePlayback) && (_animationStatus == AtEnd));

      if (shouldStart && (!AreRunning()))
      {
         if ((_animationStatus == AtStart) || (_animationStatus == AtEnd))
         {
            const Animation::AnimationPlayer::PlayDirection direction(_animationStatus == AtStart ? Animation::AnimationPlayer::Forward : Animation::AnimationPlayer::Reverse);

            for (AnimationPlayers::ConstIterator it(_animations.ConstBegin()); it != _animations.ConstEnd(); ++it)
            {
               const Animation::AnimationPlayer::SharedPointer& player(*it);
               player->SetDirection(direction);
            }
         }

         if (_groupPlayer->Start())
         {
            if (_animationStatus == AtStart)
            {
               _animationStatus = RunningForward;
            }
            else if (_animationStatus == AtEnd)
            {
               _animationStatus = RunningBackward;
            }
         }
      }
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "CTIAnimationsManager::Start this=%p, running = %d", this, AreRunning()));
   return AreRunning();
}


bool CTIAnimationsManager::AreRunning() const
{
   return (!_groupPlayer.PointsToNull()) && (_groupPlayer->IsEnabled());
}


bool CTIAnimationsManager::ToBegin()
{
   bool changed = false;
   if (_animationStatus != AtStart)
   {
      _animationStatus = AtStart;
      changed = true;
   }

   return changed;
}


bool CTIAnimationsManager::ToEnd()
{
   bool changed = false;
   if (_animationStatus != AtEnd)
   {
      _animationStatus = AtEnd;
      changed = true;
   }

   return changed;
}


void CTIAnimationsManager::OnPastEnd(Animation::AnimationPlayerBase* /*animationPlayer*/, Int32 /*completedIterationsCount*/)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "CTIAnimationsManager::OnPastEnd this=%p", this));
   OnPlaybackFinished();
}


void CTIAnimationsManager::OnPlaybackFinished()
{
   if (_animationStatus == RunningForward)
   {
      _animationStatus = AtEnd;
   }
   else if (_animationStatus == RunningBackward)
   {
      _animationStatus = AtStart;
   }

   bool forward(_animationStatus == AtEnd);
   NotifyPlaybackFinished(forward);
}


void CTIAnimationsManager::NotifyPlaybackFinished(bool forward) const
{
   if (0 != _listener)
   {
      _listener->OnPlaybackFinished(forward);
   }
}


void CTIAnimationsManager::DispatchToPlayers() const
{
   if (!_timeDispatcher.PointsToNull())
   {
      _timeDispatcher->DispatchTime();
   }
}


Candera::Animation::SequenceTimeType CTIAnimationsManager::GetDuration(Candera::Animation::AnimationController& controller) const
{
   Int32 maxSequenceTime(0);
   Int32 numberOfProperties = controller.GetNumberOfProperties();
   for (Int32 index(0); index < numberOfProperties; ++index)
   {
      const Animation::AnimationBlendedProperty::SharedPointer& prop(controller.GetProperty(index));
      if (!prop.PointsToNull())
      {
         Int32 numberOfKeyframeSequences(prop->GetNumberOfKeyframeSequences());
         for (Int32 sequence(0); sequence < numberOfKeyframeSequences; ++sequence)
         {
            const Animation::AnimationKeyframeSequence::SharedPointer& keyframeSequence(prop->GetKeyframeSequence(sequence));

            if (!keyframeSequence.PointsToNull())
            {
               const Animation::AnimationKeyframeSequence::Keyframe& keyframe(keyframeSequence->GetKeyframe(keyframeSequence->GetKeyframeCount() - 1));

               if (keyframe.m_sequencetimeMs > maxSequenceTime)
               {
                  maxSequenceTime = keyframe.m_sequencetimeMs;
               }
            }
         }
      }
   }
   return maxSequenceTime;
}
