/* ***************************************************************************************
      * FILE:          AnimationModifier.cpp
      * SW-COMPONENT:  HMI-BASE
      *  DESCRIPTION:  AnimationModifier is part of HMI-Base Widget Library
      *    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
      *
      * The reproduction, distribution and utilization of this file as well as the
      * communication of its contents to others without express authorization is
      * prohibited. Offenders will be held liable for the payment of damages.
      * All rights reserved in the event of the grant of a patent, utility model or design.
      *
      *************************************************************************************** */
#include "widget2D_std_if.h"

#include "AnimationModifier.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/AnimationModifier.cpp.trc.h"
#endif

#include "FeatStd/Diagnostics/Log.h"

#include "Candera/System/MetaInfo/PropertyMetaInfoAnimationPropertySetters.h"
#include "CanderaWidget/Widget2D/Widget2D.h"
#include "ControlTemplate.h"
#include "FeatStd/Util/Rtti.h"

using namespace Candera;
using namespace Courier;
using namespace Candera::Animation;


namespace AnimationModifierHelpers {

class ParallelNodesTraverser2D
{
   public:
      enum TraverserAction
      {
         ProceedTraversing,              ///< Traversing continues with the next child (if the current node has any) or sibling.
         StopTraversingForDescendants,   ///< Traversing will stop for all child nodes and will continue with the next sibling.
         StopTraversing                  ///< Traversing will stop here.
      };

      ParallelNodesTraverser2D() {}
      virtual ~ParallelNodesTraverser2D() {}

      /**
       * Depth-first traversing of the given tree2.
       * Parents are visited before children.
       *
       * @param root1, root2 - the roots of the trees to traverse.
       */
      void Traverse(Node2D& root1, Node2D& root2)
      {
         Node2D* currentNode1 = &root1;
         Node2D* currentNode2 = &root2;
         bool allChildrenOfRootProcessed1 = false;
         bool allChildrenOfRootProcessed2 = false;
         Node2D* nextChild1 = currentNode1->GetFirstChild();
         Node2D* nextChild2 = currentNode2->GetFirstChild();
         TraverserAction action = ProceedTraversing;

         while ((!allChildrenOfRootProcessed1) && (!allChildrenOfRootProcessed2) && (action != StopTraversing))
         {
            action = ProceedTraversing;
            // Process the current node before its children.
            if ((nextChild1 == currentNode1->GetFirstChild()) && (nextChild2 == currentNode2->GetFirstChild()))
            {
               action = ProcessNode(*currentNode1, *currentNode2);
            }

            // Process the children (if traversal has not been stopped by ProcessNode()'s return at the parent).
            if (action != StopTraversing)
            {
               const bool allChildrenAlreadyProcessed1 = (nextChild1 == 0);
               const bool allChildrenAlreadyProcessed2 = (nextChild2 == 0);

               if (allChildrenAlreadyProcessed1 || allChildrenAlreadyProcessed2 || action == StopTraversingForDescendants)
               {
                  // Up to parent, with next of parent's children (== currentNode's next sibling).
                  if (currentNode1 == &root1)
                  {
                     allChildrenOfRootProcessed1 = true;
                  }
                  else if (currentNode2 == &root2)
                  {
                     allChildrenOfRootProcessed2 = true;
                  }
                  else
                  {
                     nextChild1 = currentNode1->GetNextSibling();
                     nextChild2 = currentNode2->GetNextSibling();
                  }
                  currentNode1 = currentNode1->GetParent();
                  currentNode2 = currentNode2->GetParent();
               }
               else
               {
                  // down to child
                  currentNode1 = nextChild1;
                  currentNode2 = nextChild2;

                  nextChild1 = currentNode1->GetFirstChild();
                  nextChild2 = currentNode2->GetFirstChild();
               }
            }
         }
      }

   private:
      virtual TraverserAction ProcessNode(Node2D& node1, Node2D& node2) = 0;
};


class ParallelSearchTraverser : public ParallelNodesTraverser2D
{
   public:
      ParallelSearchTraverser(Transformable2D* searched) :
         _searched(searched),
         _found(0)
      { }

      Node2D* GetFoundNode() const
      {
         return _found;
      }

   private:
      Transformable2D* _searched;
      Node2D* _found;

      virtual TraverserAction ProcessNode(Node2D& node1, Node2D& node2) override
      {
         if (_searched == &node1)
         {
            _found = &node2;
            return StopTraversing;
         }
         return ProceedTraversing;
      }
};


static Node2D* GetSimilarChild(Node2D* srcRoot, Node2D* dstRoot, Transformable2D* child)
{
   Node2D* similar(0);
   if (0 != srcRoot && 0 != dstRoot)
   {
      ParallelSearchTraverser traverser(child);
      traverser.Traverse(*srcRoot, *dstRoot);

      similar = traverser.GetFoundNode();
   }

   return similar;
}


static bool ReplaceInAbsolute(AnimationBlendedProperty& prop, AnimationBlendedProperty& propClone, Candera::Node2D* rootSrc, Node2D* rootDest)
{
   bool replaced(false);

   BaseTransformable2DPropertySetter::SharedPointer propSetter(Dynamic_Cast<BaseTransformable2DPropertySetter::SharedPointer>(prop.GetAnimationPropertySetter()));
   BaseTransformable2DPropertySetter::SharedPointer propSetterClone(Dynamic_Cast<BaseTransformable2DPropertySetter::SharedPointer>(propClone.GetAnimationPropertySetter()));
   if ((!propSetter.PointsToNull()) && (!propSetterClone.PointsToNull()))
   {
      Transformable2D* transformable(propSetter->GetTransformable());
      Node2D* dst(GetSimilarChild(rootSrc, rootDest, transformable));

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AnimationModifierHelpers::ReplaceInAbsolute child=%p, similar=%p", transformable, dst));

      replaced = (dst != 0);
      if (replaced)
      {
         propSetterClone->SetTransformable(dst);
      }
      else
      {
         propSetterClone->SetTransformable(transformable);
      }
   }
   return replaced;
}


static bool ReplaceInRelative(AnimationBlendedProperty& prop, AnimationBlendedProperty& propClone, Candera::Node2D* rootSrc, Node2D* rootDest)
{
   bool replaced(false);

   BaseTransformable2DRelativePropertySetter::SharedPointer propSetter(Dynamic_Cast<BaseTransformable2DRelativePropertySetter::SharedPointer>(prop.GetAnimationPropertySetter()));
   BaseTransformable2DRelativePropertySetter::SharedPointer propSetterClone(Dynamic_Cast<BaseTransformable2DRelativePropertySetter::SharedPointer>(propClone.GetAnimationPropertySetter()));
   if ((!propSetter.PointsToNull()) && (!propSetterClone.PointsToNull()))
   {
      Transformable2D* transformable(propSetter->GetTransformable());
      Node2D* dst(GetSimilarChild(rootSrc, rootDest, transformable));

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AnimationModifierHelpers::ReplaceInRelative child=%p, similar=%p", transformable, dst));

      replaced = (dst != 0);
      if (replaced)
      {
         propSetterClone->SetTransformable(dst);
      }
      else
      {
         propSetterClone->SetTransformable(transformable);
      }
   }
   return replaced;
}


static bool ReplaceNode(AnimationBlendedProperty& prop, AnimationBlendedProperty& propClone, Candera::Node2D* rootSrc, Node2D* rootDest)
{
   bool replaced(false);

   BaseNode2DPropertySetter::SharedPointer propSetter(Dynamic_Cast<BaseNode2DPropertySetter::SharedPointer>(prop.GetAnimationPropertySetter()));
   BaseNode2DPropertySetter::SharedPointer propSetterClone(Dynamic_Cast<BaseNode2DPropertySetter::SharedPointer>(propClone.GetAnimationPropertySetter()));
   if ((!propSetter.PointsToNull()) && (!propSetterClone.PointsToNull()))
   {
      Node2D* node(propSetter->GetNode());
      Node2D* dst(GetSimilarChild(rootSrc, rootDest, node));
      replaced = (dst != 0);

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AnimationModifierHelpers::ReplaceNode child=%p, similar=%p", node, dst));

      if (replaced)
      {
         propSetterClone->SetNode(dst);
      }
      else
      {
         propSetterClone->SetNode(node);
      }
   }

   return replaced;
}


static bool ReplaceWidget(AnimationBlendedProperty& prop, AnimationBlendedProperty& propClone, Candera::Node2D* rootSrc, Node2D* rootDest)
{
   bool replaced(false);

   Candera::Internal::ObjectPropertySetter<WidgetBase>* propSetter(Dynamic_Cast<Candera::Internal::ObjectPropertySetter<WidgetBase>*>(prop.GetAnimationPropertySetter().GetPointerToSharedInstance()));
   if (0 != propSetter)
   {
      Widget2D* srcWidget(Dynamic_Cast<Widget2D*>(propSetter->GetAnimatedObject()));
      if (0 != srcWidget)
      {
         Node2D* srcNode(srcWidget->GetNode());
         Node2D* dstNode(GetSimilarChild(rootSrc, rootDest, srcNode));

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AnimationModifierHelpers::ReplaceWidget child=%p, similar=%p", srcNode, dstNode));

         View* parentView(srcWidget->GetParentView());
         if ((0 != parentView) && (0 != srcNode) && (0 != dstNode))
         {
            ViewScene2D* viewScene2D(parentView->ToViewScene2D());
            if (0 != viewScene2D)
            {
               const Candera::Char* srcTypeName(srcWidget->GetTypeName());
               WidgetBaseEnumerator dstEn(ControlTemplate::EnumerateAssociatedWidgets(*viewScene2D, *dstNode));

               bool found(false);
               while ((!found) && dstEn.MoveNext())
               {
                  Candera::WidgetBase* dstWidget(dstEn.Current());
                  if (0 != dstWidget)
                  {
                     const Candera::Char* dstTypeName(dstWidget->GetTypeName());

                     // direct Char* comparison is intended
                     found = (dstWidget != 0) && (srcTypeName == dstTypeName);
                  }
               }

               if (found)
               {
                  Candera::WidgetBase* dstWidget(dstEn.Current());

                  ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AnimationModifierHelpers::ReplaceWidget found=%p", dstWidget));

                  if (0 != dstWidget)
                  {
                     MetaInfo::WidgetMetaInfo* objectMetaInfo(dstWidget->GetMetaInfo());
                     MetaInfo::WidgetPropertyMetaInfo* propMetaInfo(0);
                     if (0 != objectMetaInfo)
                     {
                        propMetaInfo = objectMetaInfo->LookupItem(propSetter->GetPropertyName());
                     }

                     if (0 != propMetaInfo)
                     {
                        Int channel[4] = { 0, 1, 2, 3 };
                        Animation::AnimationPropertySetter::SharedPointer propSetterClone(propMetaInfo->CreateAnimationPropertySetter(dstWidget, 4, channel));

                        if (propSetterClone != 0)
                        {
                           propClone.SetAnimationPropertySetter(propSetterClone);
                           replaced = true;
                        }
                     }
                  }
               }
            }
         }
      }

      if (!replaced)
      {
         propClone.SetAnimationPropertySetter(prop.GetAnimationPropertySetter());
      }
   }

   return replaced;
}


}


bool AnimationModifier::Modify(const AnimationController::SharedPointer& animation, const Candera::Animation::AnimationController::SharedPointer& animationClone, Candera::Node2D* rootSrc, Node2D* rootDest) const
{
   using namespace AnimationModifierHelpers;
   bool modified(false);

   if ((!animation.PointsToNull()) && (!animationClone.PointsToNull()) && (0 != rootSrc) && (0 != rootDest))
   {
      SizeType numberOfProperties(animation->GetNumberOfProperties());
      for (SizeType propIndex(0); (propIndex < numberOfProperties); ++propIndex)
      {
         AnimationBlendedProperty::SharedPointer prop(animation->GetProperty(propIndex));
         AnimationBlendedProperty::SharedPointer propClone(animationClone->GetProperty(propIndex));
         if ((!prop.PointsToNull()) && (!propClone.PointsToNull()))
         {
            bool replaced(false);
            replaced = replaced || ReplaceWidget(*prop, *propClone, rootSrc, rootDest);
            replaced = replaced || ReplaceInAbsolute(*prop, *propClone, rootSrc, rootDest);
            replaced = replaced || ReplaceNode(*prop, *propClone, rootSrc, rootDest);
            replaced = replaced || ReplaceInRelative(*prop, *propClone, rootSrc, rootDest);
            modified = modified || replaced;

            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AnimationModifier::Modify propertyIndex=%d, replaced=%d, prop=%p, propClone=%p, rootSrc=%p, rootDst=%p", propIndex, replaced
                                , prop.GetPointerToSharedInstance(), propClone.GetPointerToSharedInstance(), rootSrc, rootDest));
         }
      }
   }

   return modified;
}
