/* ***************************************************************************************
* FILE:          AnimationCloner.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AnimationCloner is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef ANIMATION_CLONER_H
#define ANIMATION_CLONER_H

#include "DefaultControlTemplateMap.h"
#include <Candera/EngineBase/Animation/AnimationBlendedProperty.h>
#include <Candera/Engine2D/Core/Node2D.h>
#include <Candera/System/Rtti/Rtti.h>
#include "ControlTemplateInstance.h"

/**
 * The AnimationCloner clones an animation.
 */
namespace Candera {
namespace Animation {
class AnimationPlayer;
}


}

class AnimationCloner
{
   public:

      /**
       * Returns a clone of the animation template controller, with the corresponding cloned counterparts as animated objects.
       */
      static Candera::Animation::AnimationController::SharedPointer CloneAnimationController(const Candera::Animation::AnimationPlayer* templateAnimation,
            ControlTemplateInstancePtr controlTemplateInstance,
            const ControlTemplateMap& controlTemplateMap);

      static Candera::Animation::AnimationController::SharedPointer DuplicateAnimationController(const Candera::Animation::AnimationPlayer* templateAnimation);

   protected:

   private:

      static Candera::Animation::AnimationController::SharedPointer Clone(const Candera::Animation::AnimationPlayer* templateAnimation, ControlTemplateInstancePtr controlTemplateInstance, const ControlTemplateMap& controlTemplateMap, bool acceptOriginalWidget);

      static void CloneKeyframeSequences(const Candera::Animation::AnimationBlendedProperty& animPropertyTemplate, Candera::Animation::AnimationBlendedProperty& animPropertyClone);

      static void CloneKeyFrameSequence(const Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationKeyframeSequence>& kfs, const Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationKeyframeSequence>& clonedKfs);

      static bool CloneAnimationPropertySetter(const Candera::Animation::AnimationBlendedProperty& animPropertyTemplate,
            Candera::Animation::AnimationBlendedProperty& animPropertyClone,
            ControlTemplateInstancePtr controlTemplateInstance,
            const ControlTemplateMap& controlTemplateMap, bool acceptOriginalWidget);

      template <class TPropertySetter>
      static bool SetClonedTransformablePropertySetter(Candera::Animation::AnimationPropertySetter* animPropertySetterTemplate,
            Candera::Animation::AnimationBlendedProperty& animPropertyClone,
            const ControlTemplateMap& controlTemplateMap)
      {
         bool rc = false;
         if (animPropertySetterTemplate->IsTypeOf(TPropertySetter::GetTypeId()))
         {
            TPropertySetter* propSetterTemplate = Candera::Dynamic_Cast<TPropertySetter*>(animPropertySetterTemplate);
            if (propSetterTemplate != 0)
            {
               typename TPropertySetter::SharedPointer propSetterClone = TPropertySetter::Create();
               propSetterClone->SetTransformable(controlTemplateMap.ResolveNodeClone(Candera::Dynamic_Cast<Candera::Node2D*>(propSetterTemplate->GetTransformable())));
               animPropertyClone.SetAnimationPropertySetter(propSetterClone);
               rc = true;
            }
         }
         return rc;
      }

      template <class TPropertySetter>
      static bool SetClonedNodePropertySetter(Candera::Animation::AnimationPropertySetter* animPropertySetterTemplate,
                                              Candera::Animation::AnimationBlendedProperty& animPropertyClone,
                                              const ControlTemplateMap& controlTemplateMap)
      {
         bool rc = false;
         if (animPropertySetterTemplate->IsTypeOf(TPropertySetter::GetTypeId()))
         {
            TPropertySetter* propSetterTemplate = Candera::Dynamic_Cast<TPropertySetter*>(animPropertySetterTemplate);
            if (propSetterTemplate != 0)
            {
               typename TPropertySetter::SharedPointer propSetterClone = TPropertySetter::Create();
               propSetterClone->SetNode(controlTemplateMap.ResolveNodeClone(Candera::Dynamic_Cast<Candera::Node2D*>(propSetterTemplate->GetNode())));
               animPropertyClone.SetAnimationPropertySetter(propSetterClone);
               rc = true;
            }
         }
         return rc;
      }

      // use this for types derived from AbstractNodePropery
      template <class TPropertySetter>
      static bool SetClonedAbstractNodePropertySetter(Candera::Animation::AnimationPropertySetter* animPropertySetterTemplate,
            Candera::Animation::AnimationBlendedProperty& animPropertyClone,
            const ControlTemplateMap& controlTemplateMap)
      {
#if ((CANDERA_VERSION_MAJOR > 3) || ((CANDERA_VERSION_MAJOR == 3) && (CANDERA_VERSION_MINOR >= 5)))
         bool rc = false;
         if (animPropertySetterTemplate->IsTypeOf(TPropertySetter::GetTypeId()))
         {
            TPropertySetter* propSetterTemplate = Candera::Dynamic_Cast<TPropertySetter*>(animPropertySetterTemplate);
            if (propSetterTemplate != 0)
            {
               typename TPropertySetter::SharedPointer propSetterClone = TPropertySetter::Create();
               propSetterClone->SetNode(controlTemplateMap.ResolveNodeClone((propSetterTemplate->GetNode().ToNode2D())));
               animPropertyClone.SetAnimationPropertySetter(propSetterClone);
               rc = true;
            }
         }
         return rc;
#else
         return SetClonedNodePropertySetter<TPropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap);
#endif
      }
};


#endif
